# -*- coding: utf-8 -*-

from dateutil.parser import parse
import os
import re
import time

import sandbox.common.types.resource as ctr
import sandbox.projects.resource_types as rt

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.bsyeti.common import dump_timestamp, basic_releasers, BigbFlamegraphPlotterBin
from sandbox.sandboxsdk.channel import channel

DAY = 24 * 3600
WEEK = 7 * DAY


def timestamp_from_date(date):
    return int(time.mktime(date.timetuple()))


class TimeInterval(object):
    def __init__(self, start_ts, end_ts, name=None):
        self._start_ts = int(start_ts)
        self._end_ts = int(end_ts)
        if name is None:
            self._name = "from_" + str(start_ts) + "_to_" + str(end_ts)
        else:
            self._name = name

    def name(self):
        return self._name

    def start_ts(self):
        return self._start_ts

    def end_ts(self):
        return self._end_ts


def week(now_ts, name="week"):
    return TimeInterval(now_ts - WEEK, now_ts, name)


def day(now_ts):
    return TimeInterval(now_ts - DAY, now_ts, "day")


def week_ago(now_ts):
    return week(now_ts - WEEK, "week_ago")


def parse_date(date):
    return timestamp_from_date(parse(date))


def parse_interval(interval_string, now_ts):
    if interval_string == "week_ago":
        return week_ago(now_ts)
    if interval_string == "week":
        return week(now_ts)
    if interval_string == "day":
        return day(now_ts)

    intterval_regex = re.compile("^FROM(.*)TO(.*)$")
    match = intterval_regex.search(interval_string)
    if len(match.groups()) == 2:
        return TimeInterval(parse_date(match.group(1)), parse_date(match.group(2)))

    raise ValueError(
        'Expected to have one of day, week, wee_ago, or string "^FROM(.*)TO(.*)$"'
    )


def parse_compares(compare_string):
    intterval_regex = re.compile("(\s*[0-9]+)\s*WITH\s*([0-9]+)\s*")
    match = intterval_regex.search(compare_string)
    if match:
        return (int(match.group(1)), int(match.group(2)))

    raise ValueError(
        'Expected to have one of day, week, wee_ago, or string ""(\s*[0-9]+)\s*WITH\s*([0-9]+)\s*""'
    )


def name(service_name, interval):
    return "{}_{}".format(service_name, interval.name())


class LocalExecutor(object):
    def __init__(self, fetcher, flames, folder, yt_path, env, cluster, host, log):
        self._cluster = cluster
        self._env = env
        self._fetcher = str(fetcher)
        self._flames = str(flames)
        self._folder = folder
        self._host = host
        self._log = log
        self._yt_path = yt_path

    def fetch_data(self, service, interval):
        fetcher_cmd = [
            self._fetcher,
            "--logs-path",
            self._yt_path,
            "--service",
            service,
            "--start-ts",
            str(interval.start_ts()),
            "--end-ts",
            str(interval.end_ts()),
        ]
        if self._cluster:
            fetcher_cmd += ["--contour", str(self._cluster)]
        if self._host:
            fetcher_cmd += ["--host", str(self._host)]
        with open(name(service, interval), "w+") as outputFile:
            sp.check_call(
                fetcher_cmd,
                stdout=outputFile,
                stderr=self._log.stderr,
                env=self._env,
            )

    def draw_flame(self, name):
        flames_cmd = [
            str(self._flames),
            "--in-data",
            name,
            "--out-svg",
            os.path.join(self._folder, name + "_flamegraph.html"),
        ]
        sp.check_call(
            flames_cmd, stdout=self._log.stdout, stderr=self._log.stderr, env=self._env
        )

    def draw_compare(self, base, new):
        flames_cmd = [
            str(self._flames),
            "--in-data",
            base,
            "--compare",
            new,
            "--out-svg",
            os.path.join(self._folder, base + "_compared_with_" + new + ".html"),
        ]
        sp.check_call(
            flames_cmd, stdout=self._log.stdout, stderr=self._log.stderr, env=self._env
        )


class BigbProfilerYtFetcherBin(rt.ARCADIA_PROJECT):
    """
    resource with fetcher preparing profiler results
    """

    releasable = True
    any_arch = True
    auto_backup = True
    releasers = basic_releasers
    release_subscribers = basic_releasers


class BigbProfilerResults(sdk2.Resource):
    """
    results of poor mans profiler in bigb services
    """

    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers
    producer = sdk2.parameters.String("BigB")


class MakeBigbProfilerFlames(sdk2.Task):
    """
    task to build geo dict with permalinks
    """

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Requirements(sdk2.Requirements):
        cores = 2  # exactly 2 cores
        ram = 10000

    class Parameters(sdk2.Parameters):
        fetcher_id = sdk2.parameters.LastReleasedResource(
            "bigb profiler fetcher",
            resource_type=BigbProfilerYtFetcherBin,
            state=(ctr.State.READY,),
            required=True,
        )
        flames_id = sdk2.parameters.LastReleasedResource(
            "flames",
            resource_type=BigbFlamegraphPlotterBin,
            state=(ctr.State.READY,),
            required=True,
        )
        producer = sdk2.parameters.String(
            "Name of actor, that produces profiler logs",
            default="BigB",
            required=False,
        )
        services = sdk2.parameters.String(
            "List of services, comma separated",
            default="eagle,buzzard,caesar",
            required=False,
        )
        cluster = sdk2.parameters.String(
            "by default take data for all clusters",
            required=False,
        )
        yt_path = sdk2.parameters.String(
            "Path to logfeller logs",
            default="//logs/bigb-poormansprofiler-production-log/",
            required=False,
        )
        vault = sdk2.parameters.String(
            "vault yav token, with yt_token and yql_token",
            default="sec-01djab78jh9dvsymnjzwhk1mpf",
            required=False,
        )
        host = sdk2.parameters.String(
            "Host to take data from",
            required=False,
        )
        mail_to = sdk2.parameters.String(
            "List of mails, comma separated",
            default="",
            required=False,
        )
        intervals = sdk2.parameters.List(
            'list of time intervals to build flame graphs, in format "FROM <DATE1> TO <DATE2>" or "week", "day", "week_ago"',
            default=["day", "week", "week_ago"],
            required=False,
        )
        compares = sdk2.parameters.List(
            "list of compares",
            default=["1 WITH 2"],
            required=False,
        )
        do_release = sdk2.parameters.Bool(
            "Make this resourse released (USE ONLY FOR SCHEDULERS!)",
            default=False,
            required=False,
        )

    def on_execute(self):
        log = sdk2.helpers.ProcessLog(self, logger="bigb_profiler_loggin")
        fetcher_res = sdk2.ResourceData(self.Parameters.fetcher_id)
        flames_res = sdk2.ResourceData(self.Parameters.flames_id)

        folder = "flames"
        os.mkdir(folder)
        vault = self.Parameters.vault
        env = {
            "YT_TOKEN": sdk2.Vault.data(vault + "[yt_token]"),
            "YQL_TOKEN": sdk2.Vault.data(vault + "[yql_token]"),
            "TMP": ".",
            "TMPDIR": ".",
        }

        t = time.time()
        intervals_list = [
            parse_interval(interval, t) for interval in self.Parameters.intervals
        ]
        compare_list = [parse_compares(compare) for compare in self.Parameters.compares]
        services = self.Parameters.services.split(",")
        executor = LocalExecutor(
            fetcher_res.path,
            flames_res.path,
            folder,
            self.Parameters.yt_path,
            env,
            self.Parameters.cluster,
            self.Parameters.host,
            log
        )
        for service in services:
            for interval in intervals_list:
                executor.fetch_data(service=service, interval=interval)
                executor.draw_flame(name(service, interval))
            for base, new in compare_list:
                assert base < len(intervals_list)
                assert new < len(intervals_list)
                executor.draw_compare(
                    name(service, intervals_list[base]),
                    name(service, intervals_list[new]),
                )

        dump_timestamp(folder, t)
        res = BigbProfilerResults(
            self, "BigbB Profiler Results", folder, producer=self.Parameters.producer
        )
        log.close()
        if self.Parameters.do_release:
            res.released = "stable"
        if self.Parameters.mail_to != "":
            channel.sandbox.send_email(
                self.Parameters.mail_to.split(","),
                [],
                "Bigb poor mans profiler results from {}".format(time.ctime(int(t))),
                "Link to resource with results: https://proxy.sandbox.yandex-team.ru/{}".format(
                    res.id
                ),
            )
