# -*- coding: utf-8 -*-

import os
from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.bsyeti.common import basic_releasers
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk.paths import get_logs_folder


class MakeVisitHistoryLogBin(sdk2.Resource):
    """
    program to make visithistory log
    """
    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers


class MakeVisitHistoryLog(sdk2.Task):
    """
    task to build VisitHistory table
    """
    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 4096

    class Parameters(sdk2.Parameters):

        make_visithistory_log_bin_id = sdk2.parameters.LastReleasedResource(
            'Binary to make visithistory log',
            resource_type=MakeVisitHistoryLogBin,
            state=(ctr.State.READY, ),
            required=True
        )
        yt_proxy = sdk2.parameters.String(
            "YT cluster",
            default="hahn",
            required=True
        )
        visithistory_path = sdk2.parameters.String(
            "Yt path to visithistory output directory",
            default="//home/bigb/visithistory",
            required=True
        )
        active_goals_path = sdk2.parameters.String(
            "Yt path to ActiveGoals table",
            default="//home/yabs/dict/ActiveGoal",
            required=True
        )
        ad_goals_path = sdk2.parameters.String(
            "Yt path to ad_goals table",
            default="//home/metrika/export/ad_goals",
            required=True
        )
        solomon_cluster = sdk2.parameters.String(
            "Solomon cluster",
            default="test",
            required=True
        )
        history_days = sdk2.parameters.Integer(
            "History days",
            default=520,
            required=True,
        )
        max_part_days = sdk2.parameters.Integer(
            "Max log days in request",
            default=80,
            required=True,
        )
        sleep_after_part_second = sdk2.parameters.Integer(
            "Sleep seconds after request part launched",
            default=20,
            required=True,
        )
        max_goals_in_request = sdk2.parameters.Integer(
            "Max goals in request",
            default=60,
            required=True,
        )
        keep_log_days = sdk2.parameters.Integer(
            "Days to keep visithistory log",
            default=7,
            required=True,
        )
        tokens = sdk2.parameters.YavSecret(
            "Yav secret with tokens (yt_token, solomon_token)",
            required=True
        )
        yt_log_level = sdk2.parameters.String(
            "Log level",
            default="DEBUG",
            required=True,
            choices=[
                ("INFO", "INFO"),
                ("ERROR", "ERROR"),
                ("WARNING", "WARNING"),
                ("DUBUG", "DUBUG"),
            ],
        )

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.make_visithistory_log_bin_id)

        env = {
            "YT_PROXY": self.Parameters.yt_proxy,
            "YT_TOKEN": self.Parameters.tokens.data()["yt_token"],
            "SOLOMON_TOKEN": self.Parameters.tokens.data()["solomon_token"],
            "YT_LOG_LEVEL": self.Parameters.yt_log_level
        }

        cmd = map(str, [
            bin_res.path,
            "--visithistory-path", self.Parameters.visithistory_path,
            "--active-goals-path", self.Parameters.active_goals_path,
            "--ad-goals-path", self.Parameters.ad_goals_path,
            "--history-days", self.Parameters.history_days,
            "--max-part-days", self.Parameters.max_part_days,
            "--sleep-after-part-second", self.Parameters.sleep_after_part_second,
            "--max-goals-in-request", self.Parameters.max_goals_in_request,
            "--keep-log-days", self.Parameters.keep_log_days,
            "--solomon-cluster", self.Parameters.solomon_cluster,
            "--yt-log-file", os.path.join(get_logs_folder(), "yt.log"),
        ])

        with sdk2.helpers.ProcessLog(self, logger="make_visithistory_log") as l:
            sp.check_call(cmd, stdout=l.stdout, stderr=l.stderr, env=env)
