# -*- coding: utf-8 -*-

from sandbox import sdk2
import time
import logging
import sandbox.common.types.resource as ctr

from sandbox.projects.common import utils
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.bsyeti.common import dump_timestamp, basic_releasers


class BigbMkdbTool(sdk2.Resource):
    """
    Tool for creating shared bigb bases
    """

    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers
    executable = True


class BigbBases(sdk2.Resource):
    """
    Shared bigb bases
    """

    any_arch = True
    auto_backup = True
    calc_md5 = True
    share = True
    releasable = True
    releasers = basic_releasers
    base_group = sdk2.parameters.String(
        "Group of bases",
        required=True
    )


class MakeBigbBases(sdk2.Task):
    """
    task to build bigb bases using mkdb tool
    """
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 16 << 10  # 16GiB or less

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        binary_id = sdk2.parameters.Resource(
            'Mkdb tool resource',
            resource_type=BigbMkdbTool,
            state=(ctr.State.READY, )
        )
        do_release = sdk2.parameters.Bool(
            "Make this resourse released (USE ONLY FOR SCHEDULERS!)",
            default=False,
            required=False,
            do_not_copy=True
        )
        base_group = sdk2.parameters.String(
            "Base Group to build",
            required=True
        )

    def on_execute(self):
        if self.Parameters.binary_id:
            tool_path = str(sdk2.ResourceData(self.Parameters.binary_id).path)
        else:
            res_id = utils.get_and_check_last_released_resource_id(BigbMkdbTool)
            logging.info("Found last released mkdb tool: %s", str(res_id))
            tool_path = str(sdk2.ResourceData(sdk2.Resource[res_id]).path)

        folder = "bigb-bases"
        env = {
            "YT_TOKEN": sdk2.Vault.data("sec-01djab78jh9dvsymnjzwhk1mpf[yt_token]"),  # "zomb-yeti" secret from yav
            "YQL_TOKEN": sdk2.Vault.data("sec-01djab78jh9dvsymnjzwhk1mpf[yql_token]"),  # "zomb-yeti" secret from yav
            "TMP": ".",
            "TMPDIR": ".",
        }
        cmd = [
            tool_path,
            "-O", folder,
            "-G", self.Parameters.base_group,
        ]
        t = time.time()

        logging.info("Start building bigb bases with cmd: {}.".format(cmd))

        with sdk2.helpers.ProcessLog(self, logger="make_bigb_bases") as l:
            sp.check_call(cmd, stdout=l.stdout, stderr=l.stderr, env=env)
        logging.info("Build finished successfully.")
        dump_timestamp(folder, t)
        res = BigbBases(self, "Bigb Bases", folder, ttl=14)
        res.base_group = self.Parameters.base_group
        if self.Parameters.do_release:
            res.released = 'stable'
