import logging
import os
import subprocess

from sandbox import sdk2
from sandbox.common.types import task as task_types
from sandbox.projects.cajuper import resources
from sandbox.projects.cajuper.common import switch

CONFIG_GENERATOR_TASK_TYPE = 'GENERATE_CAJUPER_CHUNKS_CONFIGS'


class AggregateCajuperChunksConfigs(sdk2.Task):
    """
    Aggregates chunks-location configs
    """
    class Parameters(sdk2.Task.Parameters):
        project = sdk2.parameters.String('Project', default='web', required=True)
        db_timestamp = sdk2.parameters.Integer('DB timestamp', required=True)
        tier_name = sdk2.parameters.String('Tier', default='WebTier1', required=True)
        locations = sdk2.parameters.List('Locations', required=True)

    def on_execute(self):
        self.hint(self.Parameters.db_timestamp)
        if 'configs_tasks' not in self.Context:
            self.Context.configs_tasks = {}

        workdir = os.path.join(os.getcwd(), 'all_configs')
        os.mkdir(workdir)

        for location in self.Parameters.locations:
            if location not in self.Context.configs_tasks:
                tags = [self.Parameters.project, location, self.Parameters.tier_name]
                hints = [self.Parameters.db_timestamp]

                source_task = sdk2.Task.find(
                    type=CONFIG_GENERATOR_TASK_TYPE,
                    status=task_types.Status.SUCCESS,
                    tags=tags,
                    all_tags=True,
                    hints=hints,
                    all_hints=True,
                ).order(-sdk2.Task.id).first()

                if not source_task:
                    raise RuntimeError(
                        'Could not find {} task with tags {} and hints {}'.format(CONFIG_GENERATOR_TASK_TYPE, tags, hints)
                    )

                self.Context.configs_tasks[location] = source_task.id

            source_resource = switch.find_resource(
                task_id=self.Context.configs_tasks[location],
                resource_type=resources.CajuperChunksConfigsBundle
            )
            resource_data = sdk2.ResourceData(source_resource)
            ret = subprocess.Popen(['tar', '-xf', str(resource_data.path), '-C', workdir]).wait()
            logging.info('tar extract %s exits with code %s', str(resource_data.path), ret)

        tarball_filename = os.path.join(os.getcwd(), 'configs.tar.gz')
        subprocess.Popen(['tar', '-czvf', tarball_filename, '-C', workdir] + os.listdir(workdir)).wait()

        resource = sdk2.ResourceData(resources.CajuperChunksConfigsBundle(
            self,
            'Cajuper chunks configs bundle for {}, {} aggregated from {}'.format(
                self.Parameters.db_timestamp, self.Parameters.tier_name,
                self.Parameters.locations
            ),
            os.path.basename(tarball_filename),
            ttl=365
        ))
        resource.ready()
