import os
import subprocess

from sandbox import sdk2
from sandbox.projects.cajuper import resources
from sandbox.sdk2.service_resources import SandboxTasksBinary


class GenerateCajuperChunksConfigs(sdk2.Task):
    """
    Generates chunks-location configs
    """
    class Requirements(sdk2.Requirements):
        ram = 64 * 1024  # 64G

    class Parameters(sdk2.Task.Parameters):
        db_timestamp = sdk2.parameters.Integer('DB timestamp', required=True)
        location = sdk2.parameters.String('Location', required=True)
        topology = sdk2.parameters.String('Topology', required=True)
        tier_name = sdk2.parameters.String('Tier name', default='WebTier1', required=True)
        namespace_prefix = sdk2.parameters.String('Namespace', required=True)
        chunk_namespace_prefix = sdk2.parameters.String('Chunk namespace', default='/web/prod/chunks', required=True)
        remote_storage_group = sdk2.parameters.String('Remote storage group', required=True)
        basesearch_groups = sdk2.parameters.List('Basesearch groups', required=False)
        subresources = sdk2.parameters.List('Subresources', required=False)
        basesearch_hamster_groups = sdk2.parameters.List('Basesearch hamster groups', required=False)
        endpoint_set_template = sdk2.parameters.String(
            'Basesearch enspointset template', required=False, default=None
        )
        endpoint_set_hamster_template = sdk2.parameters.String(
            'Hamster basesearch enspointset template', required=False, default=None
        )
        timeouts_multiplicator = sdk2.parameters.Float('RS requests timeouts multiplicator', required=False, default=1)

    def on_execute(self):
        self.hint(self.Parameters.db_timestamp)

        if not self.Parameters.basesearch_groups and not self.Parameters.basesearch_hamster_groups:
            raise RuntimeError('Neither basesearch groups nor basesearch hamster groups were specified')

        import yt.wrapper as yt
        yt.config['token'] = sdk2.Vault.data(self.owner, 'yt-token')  # noqa

        import infra.callisto.tools.rs_config_generator.lib as config_generator
        configs = config_generator.generate_configs(
            self.Parameters.db_timestamp,
            self.Parameters.location,
            self.Parameters.topology,
            self.Parameters.remote_storage_group,
            self.Parameters.basesearch_groups,
            self.Parameters.basesearch_hamster_groups,
            self.Parameters.namespace_prefix,
            self.Parameters.chunk_namespace_prefix,
            self.Parameters.subresources or ('',),
            self.Parameters.tier_name,
            self.Parameters.endpoint_set_template or None,
            self.Parameters.endpoint_set_hamster_template or None,
            self.Parameters.timeouts_multiplicator,
        )
        self._check_output(configs)

        dns_cache = config_generator.generate_dns_cache(
            self.Parameters.topology,
            self.Parameters.remote_storage_group,
            self.Parameters.basesearch_groups,
            self.Parameters.basesearch_hamster_groups,
        )

        tarball_filename = _make_configs_tarball(
            self.Parameters.location, configs, dns_cache
        )
        resource = sdk2.ResourceData(resources.CajuperChunksConfigsBundle(
            self,
            'Cajuper chunks configs bundle for {}, {} at {}'.format(
                self.Parameters.db_timestamp, self.Parameters.tier_name, self.Parameters.location,
            ),
            os.path.basename(tarball_filename),
            ttl=365
        ))
        resource.ready()

    def on_save(self):
        self.Requirements.tasks_resource = SandboxTasksBinary.find(attrs={
            'target': 'cajuper/GenerateCajuperChunksConfigs/bin'
        }).first().id

    @staticmethod
    def _check_output(configs):
        if not configs:
            raise RuntimeError('Empty set of configs generated')


def _make_configs_tarball(sub_dir, configs, dns_cache):
    root = os.path.join(os.getcwd(), 'remote_storage_configs')
    os.mkdir(root)
    config_path = os.path.join(root, sub_dir)
    os.mkdir(config_path)

    for config_filename, config in configs.iteritems():
        with open(os.path.join(config_path, config_filename), 'w') as f:
            f.write(config)

    import google.protobuf.text_format as text_format
    general_dns_cache_path = os.path.join(config_path, 'general_dns_cache.proto')
    with open(general_dns_cache_path, 'wb') as f:
        f.write(text_format.MessageToString(dns_cache))

    tarball_filename = os.path.join(os.getcwd(), 'configs.tar.gz')
    subprocess.Popen(['tar', '-czvf', tarball_filename, '-C', os.path.dirname(root), os.path.basename(root)]).wait()
    return tarball_filename
