import logging

from sandbox import sdk2
from sandbox.common import errors as common_errors
from sandbox.common.types import task as task_types
from sandbox.projects.common.nanny import nanny

from sandbox.projects.cajuper import resources
from sandbox.projects.cajuper.common import switch


PROJECT = 'images'
LOCATION = 'pip'
TIERS = ('ImgTier0', )


class SwitchImagesPipConfigs(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Create and release rs configs for pip.
    WARNING: execution of this task leads to deploy of config files to pip, be careful!
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1 << 10  # 1Gb

    class Parameters(sdk2.Task.Parameters):
        db_timestamp = sdk2.parameters.Integer('DB timestamp', required=True)
        sleep_timeout = sdk2.parameters.Integer('Sleep timeout in seconds', required=True, default=300)
        auto_release = sdk2.parameters.Bool('Auto release', default=False)

    def _copy_chunks_configs(self):
        for tier_name in TIERS:
            new_resource_type = switch.get_chunk_configs_bundle_resource_type(tier_name)
            if not new_resource_type:
                raise RuntimeError('Tier {} not supported'.format(tier_name))

            resource_copy = switch.copy_resource(
                self,
                switch.find_resource(
                    task_id=self.Context.configs_generator_tasks_id[tier_name],
                    resource_type=resources.CajuperChunksConfigsBundle,
                ),
                new_resource_type,
                '{}_configs.tar.gz'.format(tier_name)
            )
            resource_copy.ready()

    def on_execute(self):
        self.hint(self.Parameters.db_timestamp)

        self.Context.configs_generator_tasks_id = {}

        with self.memoize_stage.find_subtasks(max_runs=10, commit_on_entrance=False, commit_on_wait=False):
            for tier_name in TIERS:
                configs_generator_task = sdk2.Task.find(
                    type='GENERATE_CAJUPER_CHUNKS_CONFIGS',
                    tags=[PROJECT, LOCATION, tier_name],
                    all_tags=True,
                    hints=[self.Parameters.db_timestamp],
                    all_hints=True,
                ).order(-sdk2.Task.id).first()

                if not configs_generator_task:
                    logging.info(
                        'GENERATE_CAJUPER_CHUNKS_CONFIGS for db_timestamp {} not found, '
                        'task will be sent to deep sleep for {} seconds'.format(
                            self.Parameters.db_timestamp, self.Parameters.sleep_timeout
                        )
                    )
                    raise sdk2.WaitTime(self.Parameters.sleep_timeout)

                logging.info('Proper GENERATE_CAJUPER_CHUNKS_CONFIGS task found: {}'.format(configs_generator_task.id))
                if configs_generator_task.status not in (task_types.Status.Group.FINISH | task_types.Status.Group.BREAK):
                    logging.info('Waiting for task {} to be ready'.format(configs_generator_task.id))
                    raise sdk2.WaitTask(
                        [configs_generator_task.id],
                        task_types.Status.Group.FINISH | task_types.Status.Group.BREAK,
                        wait_all=True
                    )

                if configs_generator_task.status not in task_types.Status.SUCCESS:
                    raise common_errors.TaskError('Last configs generator %s failed', configs_generator_task)

                self.Context.configs_generator_tasks_id[tier_name] = configs_generator_task.id

        with self.memoize_stage.copy_subtasks_resources(max_runs=5, commit_on_entrance=False, commit_on_wait=False):
            self._copy_chunks_configs()

            if self.Parameters.auto_release:
                release_parameters = {
                    'release_status': task_types.ReleaseStatus.STABLE,
                    'releaser': self.author,
                    'email_notifications': {
                        'to': [self.author],
                        'cc': []
                    }
                }
                self.on_release(release_parameters)

    def on_release(self, additional_parameters):
        additional_parameters['release_subject'] = self.Parameters.description
        with self.memoize_stage.nanny_release(commit_on_entrance=False):
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

        with self.memoize_stage.do_release(commit_on_entrance=False):
            sdk2.Task.on_release(self, additional_parameters)
