# coding=utf-8
from sandbox import sdk2
from sandbox.common.types import task as task_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.cajuper import resources
from sandbox.projects.cajuper import AggregateCajuperChunksConfigs
from sandbox.projects.cajuper.common import switch


PROJECT = 'VIDEO'

TIERS_AT_LOCATIONS = {
    #  'VideoPlatinum': ('sas', 'man', 'vla'),
    'VideoPlatinum': ('sas', 'vla'),
}

RECIPE_URL = 'https://nanny.yandex-team.ru/ui/#/services/dashboards/catalog/video_base/deployments/create/'
COMMENTS_TEMPLATE = (
    'Переключите, пожалуйста, VIDEO на новую базу: {timestamp}\n'
    'Для выкатки релиза используйте рецепт deploy_video_database_with_locks_queue: {recipe}'
)


class LocationParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('{location}') as location_group:
        for tier_name in TIERS_AT_LOCATIONS:
            sdk2.helpers.set_parameter(
                '{}_chunk_configs_bundle_resource'.format(tier_name),
                sdk2.parameters.Resource(
                    'Enforce configs resource for {}'.format(tier_name),
                    resource_type=resources.CajuperChunksConfigsBundle
                )
            )


class SwitchVideoDbIndex(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Create resource with search index state
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 2 << 9  # 200Mb

    class Parameters(sdk2.Task.Parameters):
        db_timestamp = sdk2.parameters.Integer('DB timestamp', required=True)
        auto_release = sdk2.parameters.Bool('Auto release', default=False)

        all_locations = set()
        for locations in TIERS_AT_LOCATIONS.itervalues():
            all_locations.update(locations)

        for loc in sorted(all_locations):
            sdk2.helpers.set_parameter(
                'location_{}'.format(loc),
                LocationParameters(
                    prefix='{}_'.format(loc),
                    label_substs={'location': loc.upper()},
                ),
            )

    def on_execute(self):
        self.hint(self.Parameters.db_timestamp)
        with self.memoize_stage.create_subtasks(commit_on_entrance=False):
            self.Context.state_creator_id = switch.run_state_creator(self, self.Parameters.db_timestamp, PROJECT).id
            self.Context.chunks_configs_loader_id = {
                tier_name: self._run_configs_loader(tier_name, locations).id
                for tier_name, locations in TIERS_AT_LOCATIONS.iteritems()
            }

            raise sdk2.WaitTask(
                [self.Context.state_creator_id] + self.Context.chunks_configs_loader_id.values(),
                task_types.Status.Group.FINISH | task_types.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.copy_subtasks_resources(commit_on_entrance=False):
            for resource in switch.copy_index_state_resources(self, self.Context.state_creator_id, PROJECT):
                resource.ready()
            for tier_name in TIERS_AT_LOCATIONS:
                self._copy_chunks_configs(tier_name)

        if self.Parameters.auto_release:
            release_parameters = {
                'release_status': task_types.ReleaseStatus.STABLE,
                'releaser': self.author,
                'email_notifications': {
                    'to': [self.author],
                    'cc': []
                }
            }
            self.on_release(release_parameters)

    def on_release(self, additional_parameters):
        additional_parameters['release_subject'] = self.Parameters.description
        additional_parameters['release_comments'] = COMMENTS_TEMPLATE.format(
            timestamp=self.Parameters.db_timestamp,
            recipe=RECIPE_URL,
        )
        with self.memoize_stage.nanny_release(commit_on_entrance=False):
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

        with self.memoize_stage.do_release(commit_on_entrance=False):
            sdk2.Task.on_release(self, additional_parameters)

    def _run_configs_loader(self, tier_name, locations):
        def get_parameter(location):
            return getattr(self.Parameters, '{}_{}_chunk_configs_bundle_resource'.format(location, tier_name))

        enforces_resources = {
            loc: get_parameter(loc).task_id
            for loc in locations
            if get_parameter(loc)
        }

        sub_task = AggregateCajuperChunksConfigs.AggregateCajuperChunksConfigs(
            self,
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            project=PROJECT,
            db_timestamp=self.Parameters.db_timestamp,
            tier_name=tier_name,
            locations=locations,
            configs_tasks=enforces_resources,
            description='get {} chunks configs for {}'.format(PROJECT, self.Parameters.db_timestamp),
            tags=[PROJECT, tier_name],
        ).enqueue()
        return sub_task

    def _copy_chunks_configs(self, tier_name):
        new_resource_type = switch.get_chunk_configs_bundle_resource_type(tier_name)
        if not new_resource_type:
            raise RuntimeError('Tier {} not supported'.format(tier_name))

        switch.copy_resource(
            self,
            switch.find_resource(
                task_id=self.Context.chunks_configs_loader_id[tier_name],
                resource_type=resources.CajuperChunksConfigsBundle,
            ),
            new_resource_type,
            '{}_configs.tar.gz'.format(tier_name)
        ).ready()
