# coding: utf-8

import logging
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.errors import TemporaryError
from sandbox.projects import resource_types
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
import tempfile
import requests


class CanvasFfConvert(sdk2.Task):
    """ This task runs ffmpeg to convert audio """

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 1
        url = sdk2.parameters.Url("Source url", required=True)
        webhook_url = sdk2.parameters.String("Webhook URL")
        with sdk2.parameters.Output:
            out_url = sdk2.parameters.String("output url", required=False)

    def download_wget(self, src, out):
        """качает через wget файл"""
        logging.info("Downloading file")
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen("wget -t9 -O '{}' '{}'".format(out, src), shell=True, stdout=pl.stdout, stderr=sp.STDOUT)
            pr.wait()
            if pr.returncode != 0:
                msg = 'wget {} exit code {}'.format(src, pr.returncode)
                raise TemporaryError(msg)

    def run_ffmpeg(self, src, out):
        """Перекодирует файл в нужный формат через ffmpeg"""
        logging.info("Converting file")
        logging.info(str(self.ffmpeg_bin))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd")) as pl:
            pr = sp.Popen("{} -i {} -vn -ar 44100 -ac 2 -ab 192k -f mp3 -filter:a loudnorm {}".format(self.ffmpeg_bin, src, out), shell=True, stdout=pl.stdout, stderr=sp.STDOUT)
            pr.wait()
            if pr.returncode != 0:
                msg = 'ffmpeg {} exit code {}'.format(src, pr.returncode)
                raise TemporaryError(msg)

    def convert(self, url):
        """конвертация аудиофайла"""
        wget_temp_path = tempfile.NamedTemporaryFile().name
        self.download_wget(url, wget_temp_path)
        ffmpeg_out_temp_path = tempfile.NamedTemporaryFile().name + ".mp3"
        self.run_ffmpeg(wget_temp_path, ffmpeg_out_temp_path)
        # Заливает в stillage
        logging.info("Uploading file")
        res = requests.post(
            "https://stillage.mediaselling.yandex.net/api/v1/files/content/bin",
            headers={'Accept': 'application/json', 'Content-Type': 'application/octet-stream'},
            params={"fileName": "ad.mp3"},
            data=open(ffmpeg_out_temp_path).read(),
            verify=False
        )
        logging.info(str(res.json()))
        return res.json()['url']

    def on_execute(self):
        logging.info("Downloading resource with ffmpeg binary")
        ffmpeg_res = list(sdk2.Resource.find(resource_types.FFMPEG_BIN,
                                             state=[ctr.State.READY],
                                             attrs=dict(released=ctt.ReleaseStatus.STABLE)).limit(1))[0]
        ffmpeg = sdk2.ResourceData(ffmpeg_res)
        logging.info("Resource downloaded")
        self.ffmpeg_bin = str(ffmpeg.path / "ffmpeg")
        out_url = self.convert(self.Parameters.url)
        self.Parameters.out_url = out_url
        # вызывает webhook_url
        if self.Parameters.webhook_url:
            try:
                s = requests.Session()
                data = {'task_id': self.id, 'out_url': out_url}
                logging.debug('web hook to %s with %s', self.Parameters.webhook_url, data)
                s.mount('http://', requests.adapters.HTTPAdapter(max_retries=3))
                s.post(self.Parameters.webhook_url, json=data, timeout=60, verify=False)
            except Exception:
                logging.exception('web hook failed send post reqest to %s', self.Parameters.webhook_url)
