# -*- coding: utf-8 -*-

import logging
import urlparse

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.advq.common.parameters import YtParameters


logger = logging.getLogger(__name__)


class CanvasImportVideoCategories(sdk2.Task):
    """
    Ex background worker's task
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        # TODO: find out how to use versions and not to fail :(
        environments = (
            PipEnvironment("requests"),
            PipEnvironment("yandex-yt"),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YtParameters):
        video_additions_api_url = sdk2.parameters.Url(
            "URL to video-additions backend",
            default="http://video-additions-backend.production.canvas.mediaselling.stable.qloud-d.yandex.net/",
            required=True,
        )
        categories_table_path = sdk2.parameters.String(
            "Path to table with categories on YT",
            default="//home/catalogia/mediagroup2videodirectgroup",
            required=True,
        )

    def get_yt_token(self):
        return sdk2.Vault.data(
            self.Parameters.yt_token_vault_user,
            self.Parameters.yt_token_vault_name,
        )

    def on_execute(self):
        import requests

        import yt.wrapper as yt

        yt_client = yt.YtClient(
            proxy=str(self.Parameters.yt_proxy),
            token=self.get_yt_token(),
            config={"proxy": {"request_retry_enable": False}},
        )

        logger.info('Starting importing video categories')

        video_additions_api_url = str(self.Parameters.video_additions_api_url)
        table_path = str(self.Parameters.categories_table_path)

        table = yt_client.read_table(table_path, format=yt.JsonFormat(attributes={"encode_utf8": False}))
        new_categories = [{
            "MediaGroupId": unicode(row['MediagroupID']),
            "MediaGroupName": row['Mediagroup'],
            "VideoDirectGroupId": unicode(row['VideodirectgroupID']),
            "VideoDirectGroupName": row['Videodirectgroup'],
        } for row in table]

        if not new_categories:
            raise Exception('No categories was read from YT table "%s"' % (table_path, ))

        # Send categories to the app
        result = requests.put(urlparse.urljoin(video_additions_api_url, '/media_groups'),
                              json=new_categories)
        if result.status_code != 200:
            raise Exception('PUT {}/media_groups responded {} ({})'.format(
                video_additions_api_url,
                result.status_code,
                result.text,
            ))

        logger.info('Categories saved OK')
