# coding: utf-8

import logging
import os
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.errors import TemporaryError
import requests
import time
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
import urllib

DEFAULT_CONTAINER_ID = 1696812598
STILLAGE_RETRIES_NUM = 5


def init_session():
    """Делаем 5 ретраев на запрос"""
    session = requests.Session()
    retry = Retry(
        total=5,
        read=5,
        connect=5,
        backoff_factor=0.3,
        status_forcelist=(500, 502, 504, 404, 429),
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)

    return session


def prepare_cmd_for_mp42packshot(ffmpeg_bin, src, out, scene_threshold):
    return [
        ffmpeg_bin,
        "-i", src,
        "-ss", "0.8",
        "-vf", "select='lt(scene\,%f)',scale=320:-1:flags=lanczos" % scene_threshold,
        "-frames:v", "1",
        out
    ]


def mp42packshot(task, ffmpeg_bin, src, out, template_id=None):
    """Получает один кадр для обложки в формате png из mp4 с помощью ffmpeg"""
    logging.info("Creating packshot")

    scene_threshold = 0.000005
    if template_id is not None and template_id == "template21":
        scene_threshold = 0.0003

    # Почти всегда двух попыток хватает, но для очень редких случаев просто повышаем порог
    # Одна итерация цикла идет максимум пару секунд
    while not os.path.exists(out):
        logging.info("Attempt to make packshot with scene threshold = %f" % scene_threshold)
        with sdk2.helpers.ProcessLog(task, logger=logging.getLogger("cmd")) as pl:
            mp42packshot_cmd = prepare_cmd_for_mp42packshot(ffmpeg_bin, src, out, scene_threshold)

            pr = sp.Popen(mp42packshot_cmd, stdout=pl.stdout, stderr=sp.STDOUT)
            pr.wait()

            if pr.returncode != 0:
                msg = 'ffmpeg {} exit code {}'.format(src, pr.returncode)
                raise TemporaryError(msg)

        scene_threshold *= 10
    logging.info("Packshot is done")


def prepare_cmd_for_mp42preview(ffmpeg_bin, src, out):
    # откидываем звук, сжимаем видео до 320 по ширине и ускоряем в 3 раза
    return [
        ffmpeg_bin,
        "-i", src,
        "-an",
        "-vf", "setpts=0.33*PTS,scale='320:trunc(ow/a/2)*2'",
        "-c:v", "libx264",
        "-crf", "23",
        "-profile:v", "baseline",
        "-level", "3.0",
        "-pix_fmt", "yuv420p",
        "-movflags", "faststart",
        out
    ]


def mp42preview(task, ffmpeg_bin, src, out):
    """Генерирует маленькое ускоренное превью из mp4 с помощью ffmpeg"""
    logging.info("Creating preview")

    with sdk2.helpers.ProcessLog(task, logger=logging.getLogger("cmd")) as pl:
        mp42preview_cmd = prepare_cmd_for_mp42preview(ffmpeg_bin, src, out)

        pr = sp.Popen(mp42preview_cmd, stdout=pl.stdout, stderr=sp.STDOUT)
        pr.wait()

        if pr.returncode != 0:
            msg = 'ffmpeg {} exit code {}'.format(src, pr.returncode)
            raise TemporaryError(msg)

    logging.info("Preview is done")


def upload_to_stillage(session, data, file_name, force_use_file_name=False, attachment_filename=None):
    # стиллаж иногда возвращает ошибку в самом теле ответа, даже если код ответа 200 OK
    headers = {
        'Accept': 'application/json',
        'Content-Type': 'application/octet-stream'
    }
    # если хотим, чтобы файл скачивался сразу без открытия в новой вкладке
    if attachment_filename:
        encoded_attachment_file_name = (
            attachment_filename.encode('utf-8')
            if isinstance(attachment_filename, unicode)
            else attachment_filename
        )
        content_disposition = "attachment; filename*=UTF-8''%s" % urllib.quote(encoded_attachment_file_name)

        headers['X-Mds-Content-Disposition'] = content_disposition

    for retry in range(STILLAGE_RETRIES_NUM):
        response = session.post(
            "https://stillage.mediaselling.yandex.net/api/v1/files/content/bin",
            headers=headers,
            params={"fileName": file_name, "forceUseFilename": force_use_file_name},
            data=data,
            verify=False
        )

        try:
            response_dict = response.json()
            logging.info("Stillage response: %s" % response_dict)

            if response_dict.get("status", 200) == 200:
                return response
        except Exception as e:
            logging.exception("Can't decode stillage response: %s" % str(e))

        if retry + 1 < STILLAGE_RETRIES_NUM:
            logging.warning("Can't upload to stillage, retrying...")
            time.sleep(1)

    msg = "Can't upload to stillage, all attempts have failed"
    logging.error(msg)
    raise RuntimeError(msg)


class CommonVideoParameters(sdk2.Parameters):
    """
    Общие параметры про съемку видео
    """
    container = sdk2.parameters.Container(
        "LXC Container",
        default_value=DEFAULT_CONTAINER_ID,
        required=True
    )
    max_restarts = 1

    url = sdk2.parameters.Url("Source html url", required=True)
    audio_url = sdk2.parameters.Url("Source audio url", required=False)
    width = sdk2.parameters.Integer("Video width", required=True)
    height = sdk2.parameters.Integer("Video height", required=True)
    fps = sdk2.parameters.Integer("Video frame rate (fps)", default=25, required=True)
    webhook_url = sdk2.parameters.String("Webhook URL")
    name = sdk2.parameters.String("Video name", required=False)
    template_id = sdk2.parameters.String("Template id (only for info)", required=False)

    with sdk2.parameters.Output:
        mp4_url = sdk2.parameters.String("mp4 url", required=False)
        mp4_stillage_id = sdk2.parameters.String("mp4 stillage ID", required=False)
        preview_url = sdk2.parameters.String("preview url", required=False)
        preview_stillage_id = sdk2.parameters.String("preview stillage ID", required=False)
        packshot_url = sdk2.parameters.String("packshot url", required=False)
        packshot_stillage_id = sdk2.parameters.String("packshot stillage ID", required=False)
        success = sdk2.parameters.Bool("success or not", required=False)
