# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.advq.common.parameters import YtParameters

from . import dumpers


def get_drivers_radio_group():
    with sdk2.parameters.RadioGroup("Dumper class", required=True) as yt_dumper:
        for attr in dir(dumpers):
            if not attr.startswith("_") and attr != "Driver":
                obj = getattr(dumpers, attr)
                # check isinstance firstly to prevent exception inside issubclass for modules
                if isinstance(obj, type) and issubclass(obj, dumpers.Driver):
                    yt_dumper.values[attr] = yt_dumper.Value(value=attr)

    return yt_dumper


class CanvasYtDumper(sdk2.Task):
    """
    Dump some canvas data to YT
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        # TODO: find out how to use versions and not to fail :(
        environments = (
            PipEnvironment("pymongo"),
            # выставляем версию вручную, так как в 0.10.3 есть баг: DIRECT-128436
            PipEnvironment("yandex-yt", "0.9.35"),
            PipEnvironment("psycopg2-binary"),
            PipEnvironment("brotli"),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YtParameters):
        yt_dumper = get_drivers_radio_group()

        yt_dump_dir = sdk2.parameters.String("YT dump dir (empty = default for selected driver)", required=False)
        overwrite_existing = sdk2.parameters.Bool("Overwrite existing dump", required=False)

        # could contain DB password, that's why stored in the vault
        mongo_connection_uri_vault_user = sdk2.parameters.String(
            "Mongo connection URI vault user",
            default="CANVAS",
        )
        mongo_connection_uri_vault_name = sdk2.parameters.String(
            "Mongo connection URI vault name",
            default="canvas-production-mongo",
        )

        # could contain DB password, that's why stored in the vault
        lpc_psycopg2_connection_string_vault_user = sdk2.parameters.String(
            "psycopg2 connection string vault user",
            default="CANVAS",
        )
        lpc_psycopg2_connection_string_vault_name = sdk2.parameters.String(
            "psycopg2 connection string vault name",
            default="lpc-psycopg2-production",
        )

    def get_yt_token(self):
        return sdk2.Vault.data(
            self.Parameters.yt_token_vault_user,
            self.Parameters.yt_token_vault_name,
        )

    def get_mongo_connection_uri(self):
        return str(sdk2.Vault.data(
            self.Parameters.mongo_connection_uri_vault_user,
            self.Parameters.mongo_connection_uri_vault_name,
        ))

    def get_lpc_psycopg2_connection(self):
        if (
                self.Parameters.lpc_psycopg2_connection_string_vault_user
                and self.Parameters.lpc_psycopg2_connection_string_vault_name
        ):
            import psycopg2

            connection_string = str(sdk2.Vault.data(
                self.Parameters.lpc_psycopg2_connection_string_vault_user,
                self.Parameters.lpc_psycopg2_connection_string_vault_name,
            ))
            return psycopg2.connect(connection_string)
        else:
            return None

    def on_execute(self):
        import pymongo
        import yt.wrapper as yt

        yt_client = yt.YtClient(proxy=str(self.Parameters.yt_proxy), token=self.get_yt_token())
        yt_client.config['write_parallel']['enable'] = True
        yt_client.config['write_parallel']['unordered'] = True
        yt_client.config['write_parallel']['max_thread_count'] = 20
        # 1 Mb
        yt_client.config['write_retries']['chunk_size'] = 32 * 1024 * 1024
        # brotli allowes parallel writing in python2
        yt_client.config['proxy']['content_encoding'] = 'br'

        mongo = pymongo.MongoClient(host=self.get_mongo_connection_uri()).canvas

        dumper_class = getattr(dumpers, str(self.Parameters.yt_dumper))

        dumper_class(
            yt_client=yt_client,
            mongo=mongo,
            lpc_psycopg2_connection=self.get_lpc_psycopg2_connection(),
            overwrite_existing=bool(self.Parameters.overwrite_existing),
            yt_dump_dir=str(self.Parameters.yt_dump_dir) or None,
        ).run()
