# coding=utf-8
from __future__ import unicode_literals

import os
import shlex

from pathlib2 import Path

import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.sdk2.helpers import gdb
from sandbox.projects.capacity_infra.CapacityInfraUtils.\
    CapacityInfraJugglerReporter import CapacityInfraJugglerReporterMixin


class CommonParameters(sdk2.Parameters):
    owner = 'robot-capaplan'


class CapacityInfraRunBinaryDispenserPreCalc(binary_task.LastRefreshableBinary, CapacityInfraJugglerReporterMixin, sdk2.Task):
    juggler_host = 'capacity-planning-sandbox'

    class Parameters(CommonParameters):
        with sdk2.parameters.Group('Binary') as binary:
            binary_from_resource = sdk2.parameters.Bool('Binary from Resource', required=True, default=True)
            with binary_from_resource.value[True]:
                binary_by_id = sdk2.parameters.Bool('Resolve via Id', default=True, description='otherwise use resource type')
                with binary_by_id.value[True]:
                    resource = sdk2.parameters.Resource('Resource Id', required=True)
                with binary_by_id.value[False]:
                    resource_type = sdk2.parameters.String('Resource Type', required=True)
                    resource_attrs = sdk2.parameters.Dict('Resource Attributes')

            with binary_from_resource.value[False]:
                arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia URL', required=True, default_value='arcadia-arc:/#trunk')
                yamake_path = sdk2.parameters.String('yamake path', required=True, description='For example, relative/path/from/arcadia/leet_project (default release)')

        with sdk2.parameters.Group('Run Parameters') as run:
            argv = sdk2.parameters.String('Arguments', description='-f --bar 1234')
            env_vars = sdk2.parameters.Dict('Env', description='key value')

        with sdk2.parameters.Group('Juggler') as juggler:
            send_to_juggler = sdk2.parameters.Bool('Send Result to Juggler', default=False, required=True)
            with send_to_juggler.value[True]:
                event_from_stdout = sdk2.parameters.Bool(
                    'event from stdout', default=True, required=True,
                    description='Event will be parsed from the last stdout line of the following format [status, description] or [service, host, status, description]'
                )
                juggler_service = sdk2.parameters.String('Juggler Service', description='Override the same from stdout')
                juggler_host = sdk2.parameters.String('Juggler Host', description='Override stdout')

        _binary = binary_task.binary_release_parameters_list(none=True)

    def on_prepare(self):
        self.juggler_host = self.Parameters.juggler_host
        self.juggler_service = self.Parameters.juggler_service

    def on_execute(self):
        env = {
            'DISP_OAUTH_TOKEN': sdk2.yav.Secret('sec-01ebaf8vc87am1v0k3tbyvbj4h').data()['disp_token'],
            'CH_USER_PASS': sdk2.yav.Secret('sec-01ebaf8vc87am1v0k3tbyvbj4h').data()['ch_user_pass']
        }
        env.update(self.Parameters.env_vars)

        binary_task.LastRefreshableBinary.on_execute(self)

        if self.Parameters.binary_from_resource:
            if self.Parameters.binary_by_id:
                binary = self.Parameters.resource
            else:
                binary = sdk2.Resource.find(type=self.Parameters.resource_type, state=ctr.State.READY, attrs=self.Parameters.resource_attrs).first()
                self.set_info('<a href="https://sandbox.yandex-team.ru/resource/{}">Resource</a>'.format(binary.id), do_escape=False)
            binary_path = sdk2.ResourceData(binary).path.as_posix()
        else:
            with self.memoize_stage.yamake(commit_on_entrance=False):
                self.Context.yamake_task = YaMake2(
                    self,
                    checkout_arcadia_from_url=self.Parameters.arcadia_url,
                    targets=self.Parameters.yamake_path,
                    use_arc_instead_of_aapi=True
                ).enqueue().id
                raise sdk2.WaitTask(self.Context.yamake_task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

            yamake_task = sdk2.Task[self.Context.yamake_task]
            if yamake_task.status not in ctt.Status.Group.SUCCEED:
                raise TaskFailure('Child task failed')
            binary = sdk2.ResourceData(
                sdk2.Resource.find(task=yamake_task, state=ctr.State.READY, type=resource_types.BUILD_OUTPUT).first()
            )
            binary_dir = binary.path.joinpath(self.Parameters.yamake_path)
            binary_path = next((
                path.as_posix() for path in Path(binary_dir).glob('*')
                if path.is_file() and os.access(path.as_posix(), os.X_OK) and not path.suffix
            ), None)
        if not binary_path:
            raise Exception('Binary not found')

        with sdk2.helpers.ProcessLog(self, logger='run-binary-precalc') as log:
            for stream in ['out', 'err']:
                self.set_info(gdb.get_html_view_for_logs_file(
                    'Std' + stream, getattr(log, 'std' + stream).path.relative_to(self.log_path()), self.log_resource),
                    do_escape=False
                )

            sdk2.helpers.subprocess.check_call(
                [binary_path] + shlex.split(self.Parameters.argv),
                env=env, stdout=log.stdout, stderr=log.stderr
            )

            if self.Parameters.event_from_stdout:
                self.Context.stdout_path = log.stdout.path.as_posix()

    def get_event(self):
        if self.Parameters.event_from_stdout:
            service = host = None
            info = open(self.Context.stdout_path).readlines()[-1].split(' ', 3)
            if len(info) == 2:
                status, description = info
            elif len(info) == 4:
                service, host, status, description = info
            else:
                status, description = 'CRIT', 'Program must print [status, description] or [service, host, status, description] line'

            return self.juggler_service or service, self.juggler_host or host, status, description.strip()
        else:
            return super(CapacityInfraRunBinaryDispenserPreCalc, self).get_event()

    def _send_juggler_event(self, status):
        if self.Parameters.send_to_juggler:
            super(CapacityInfraRunBinaryDispenserPreCalc, self)._send_juggler_event(status)
