# -*- coding: utf-8 -*-

import logging

from sandbox.common.config import Registry
from sandbox.common.types.misc import Installation
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link
from sandbox.projects.common.juggler import jclient


class CapacityInfraJugglerReporterMixin(object):
    juggler_host = 'capacity-planning-sandbox'
    juggler_service = None

    def get_event(self):
        return (None,) * 4

    def _send_juggler_event(self, task_status):
        service, host, status, description = self.get_event()
        host = self.juggler_host or host
        service = self.juggler_service or service or self.type.name

        if not status:
            if task_status in Status.Group.SUCCEED:
                if self._juggler_warn_predicate():
                    status = 'WARN'
                else:
                    status = 'OK'
            elif task_status in Status.Group.SCHEDULER_FAILURE:
                status = 'CRIT'

        juggler_event = {
            'host': host,
            'service': service,
            'status': status,
            'description': '{} [ {} ]'.format(description or '', get_task_link(self.id)).strip()
        }

        event_string = '{host}:{service} [{status}] {description}'.format(**juggler_event)

        if Registry().common.installation != Installation.LOCAL:
            if self._juggler_predicate(task_status):
                jclient.send_events_to_juggler(**juggler_event)
                logging.info('Juggler monitoring sent: {}'.format(event_string))
            else:
                logging.info("Don't send monitoring because of predicate: {}".format(event_string))

        else:
            logging.info("Don't send monitoring because of local installation: {}".format(event_string))

    def _juggler_predicate(self, status):
        return bool(self.scheduler)

    def _juggler_warn_predicate(self):
        return False

    def on_finish(self, previous_status, status):
        self._send_juggler_event(status)
        super(CapacityInfraJugglerReporterMixin, self).on_finish(previous_status, status)

    def on_break(self, previous_status, status):
        self._send_juggler_event(status)
        super(CapacityInfraJugglerReporterMixin, self).on_break(previous_status, status)
