import os

import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog, subprocess
from sandbox.sandboxsdk.environments import VirtualEnvironment


class CarsharingAnalytics(sdk2.Task):
    """
    Run some carsharing analytics script
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024
        cores = 1
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        script_path = sdk2.parameters.ArcadiaUrl(
            "Path to script in Arcadia", required=True,
            default_value="arcadia:/arc/trunk/arcadia/"
        )
        vault_env = sdk2.parameters.List("Vault items for script: ENVVAR=vault_key:vault_owner")
        requirements = sdk2.parameters.List("Python packages to install in VirtualEnvironment")

    def _prepare_env(self):
        env = os.environ.copy()
        env["REQUESTS_CA_BUNDLE"] = "/etc/ssl/certs/ca-certificates.crt"
        for item in self.Parameters.vault_env:
            name, value = item.split("=")
            key, owner = value.split(":")
            value = sdk2.Vault.data(owner, key)
            env[name] = value
        return env

    def on_execute(self):
        script = os.path.basename(self.Parameters.script_path)
        sdk2.vcs.svn.Arcadia.export(self.Parameters.script_path, script)

        with VirtualEnvironment() as venv:
            venv.pip("-U pip==20.1.1")
            venv.pip(" ".join((
                "yandex-yt==0.9.29",
                "yandex-yt-yson-bindings-skynet==0.3.32-0",
                "yandex-yt-transfer-manager-client==0.0.32.post0",
                "urllib3==1.25.11",
                "requests==2.25.0",
            )))

            if self.Parameters.requirements:
                def is_good_lib(lib):
                    for name in ("yandex-yt", "urllib3", "requests"):
                        if name in lib:
                            return False
                    return True

                requirements = list(filter(
                    is_good_lib, self.Parameters.requirements,
                ))
                if requirements:
                    venv.pip(" ".join(requirements))

            with ProcessLog(self, logger="script") as pl:
                subprocess.check_call(
                    [venv.executable, script],
                    stdout=pl.stdout, stderr=subprocess.STDOUT,
                    env=self._prepare_env(),
                )
