from __future__ import print_function

import logging
import os

from sandbox import sdk2
from sandbox.projects.sandbox.resources import LXC_CONTAINER
from sandbox.sandboxsdk import environments

from sandbox.projects.catboost.util.resources import CatBoostPythonPackageWheel


class CatBoostMkWheel(sdk2.Resource):
    '''
    CatBoost mk_wheel.py
    '''
    executable = True


class MakeCatBoostPythonPackageWheel(sdk2.Task):
    class Parameters(sdk2.Parameters):
        mk_wheel = sdk2.parameters.Resource('actual mk_wheel.py', resource_type=CatBoostMkWheel, required=True)
        svn_revision = sdk2.parameters.Integer('svn revision', default=None)
        with_cuda_support = sdk2.parameters.Bool('with CUDA support', default=False)
        with_jupyter_widget_support = sdk2.parameters.Bool('with Jupyter Widget support', default=False)
        with sdk2.parameters.RadioGroup('python version') as python_version:
            python_version.values['2.7'] = python_version.Value('2.7')
            python_version.values['3.5'] = python_version.Value('3.5', default=True)
            python_version.values['3.6'] = python_version.Value('3.6')
            python_version.values['3.7'] = python_version.Value('3.7')
            python_version.values['3.8'] = python_version.Value('3.8')
        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=1477520355,
            resource_type=LXC_CONTAINER,
            platform="linux_ubuntu_16.04_xenial",
            required=True,
        )

    class Requirements(sdk2.Requirements):
        disk_space = 40 * 1024
        ram = 8 * 1024
        cores = 28
        environments = (environments.SvnEnvironment(),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        ARCADIA_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia'

        if self.Parameters.svn_revision:
            svn_revision = self.Parameters.svn_revision
        else:
            svn_revision = sdk2.vcs.svn.Svn.info(ARCADIA_URL)['commit_revision']

        if self.Parameters.with_cuda_support:
            cuda_arg = ''
        else:
            cuda_arg = '-DHAVE_CUDA=no'

        if self.Parameters.with_jupyter_widget_support:
            jupyter_widget_arg = '--build-widget=yes'
        else:
            jupyter_widget_arg = '--build-widget=no'

        python_version = self.Parameters.python_version

        cmd = 'svn cat {arcadia_url}/ya | python - clone -r{svn_revision}'.format(
            arcadia_url=ARCADIA_URL,
            svn_revision=svn_revision
        )
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("arcadia_clone")) as pl:
            sdk2.helpers.subprocess.check_call(cmd, shell=True, stdout=pl.stdout, stderr=pl.stdout)

        os.chdir('arcadia')

        cmds = [
            'svn up -r{svn_revision} --set-depth=immediates catboost'.format(svn_revision=svn_revision),
            'svn up -r{svn_revision} --set-depth=infinity catboost/python-package'.format(
                svn_revision=svn_revision
            ),
            'cp {mk_wheel} ./catboost/python-package/mk_wheel_actual.py'.format(
                mk_wheel=str(sdk2.ResourceData(self.Parameters.mk_wheel).path)
            )
        ]

        for cmd in cmds:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmds")) as pl:
                sdk2.helpers.subprocess.check_call(cmd, shell=True, stdout=pl.stdout, stderr=pl.stdout)

        os.chdir('catboost/python-package')

        mk_wheel_cmd = (
            'python{python_version} ./mk_wheel_actual.py --checkout {cuda_arg} {jupyter_widget_arg} '
            + ' -DOS_SDK=local -DUSE_ARCADIA_PYTHON=no -DPYTHON_CONFIG=/usr/bin/python{python_version}-config'
            + ' -DCATBOOST_OPENSOURCE -DCFLAGS=-DCATBOOST_OPENSOURCE=yes'
            + ' --host-platform-flag CATBOOST_OPENSOURCE=yes'
            + " --host-platform-flag 'CFLAGS=-DCATBOOST_OPENSOURCE=yes'"
        ).format(python_version=python_version, cuda_arg=cuda_arg, jupyter_widget_arg=jupyter_widget_arg)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("mk_wheel")) as pl:
            wheel_path = sdk2.helpers.subprocess.check_output(
                mk_wheel_cmd,
                shell=True,
                stderr=pl.stdout
            ).split('\n')[-2]

        wheel_res = sdk2.ResourceData(
            CatBoostPythonPackageWheel(
                self,
                'CatBoost python package wheel',
                wheel_path,
                python_version=python_version,
                svn_revision=svn_revision,
                with_cuda_support=self.Parameters.with_cuda_support,
                with_jupyter_widget_support=self.Parameters.with_jupyter_widget_support
            )
        )
        wheel_res.ready()
