# coding: utf-8

import logging
import time
import requests

import sandbox.projects.release_machine.components.all as rmc
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.startrek_helper as rm_st
import sandbox.projects.release_machine.input_params2 as rm_params
import sandbox.projects.release_machine.security as rm_sec

from sandbox import common, sdk2
from sandbox.common.types.task import Semaphores
from sandbox.projects.chats.lib import CannonStatus
from sandbox.projects.common import decorators
from sandbox.projects.common import link_builder
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.feedback_platform.resources import FEEDBACK_PACKAGE
from sandbox.projects.release_machine.core import const as rm_const

TARGET_NANNY_SERVICE_STATE = 'ACTIVE'


class LoadtestLogs(sdk2.Resource):
    """ Plaintext file resource """


def create_nanny_client(oauth_token):
    # type: (str, str) -> NannyClient
    nanny_api_url = u'https://nanny.yandex-team.ru'  # without / at the end
    return NannyClient(api_url=nanny_api_url, oauth_token=oauth_token)


@decorators.retries(max_tries=3)
def deploy_docker_image_to_nanny_service(nanny_client, docker_image, service_id, comment):
    # type: (NannyClient, str, str, str) -> None
    instance_spec = nanny_client.get_service_instance_spec(service_id=service_id)
    instance_spec['content']['dockerImage']['name'] = docker_image
    instance_spec['comment'] = comment
    return nanny_client.update_service_instance_spec(service_id=service_id, instance_spec_data=instance_spec)


@decorators.retries(max_tries=3)
def is_nanny_service_snapshot_deployed(nanny_client, service_id, snapshot_id):
    current_state = nanny_client.get_service_current_state(service_id=service_id)
    for snapshot in current_state['content']['active_snapshots']:
        if snapshot['snapshot_id'] == snapshot_id:
            logging.info(u'Current state of snapshot %s is %s', snapshot['snapshot_id'], snapshot['state'])
            return snapshot['state'] == TARGET_NANNY_SERVICE_STATE


def deploy_docker_image_to_nanny_services(nanny_oauth_token, docker_image, service_ids, comment):
    """
    Deploy given docker image from registry.yandex.net to given Nanny services.
    """
    nanny_client = create_nanny_client(oauth_token=nanny_oauth_token)
    active_snapshot_ids = dict()
    for service_id in service_ids:
        answer = deploy_docker_image_to_nanny_service(nanny_client=nanny_client, docker_image=docker_image, service_id=service_id, comment=comment)
        active_snapshot_ids[service_id] = answer['runtime_attrs']['_id']

    # wait for each service deploy
    for service_id in active_snapshot_ids:
        deployed = False
        logging.info(u'Waiting for service %s to become %s', service_id, TARGET_NANNY_SERVICE_STATE)

        while not deployed:
            time.sleep(10)
            deployed = is_nanny_service_snapshot_deployed(nanny_client=nanny_client, service_id=service_id, snapshot_id=active_snapshot_ids[service_id])


class ChatsRunLoadtest(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.all_rm & task_env.TaskTags.startrek_client

    class Parameters(rm_params.ComponentName2):
        release_number = sdk2.parameters.Integer('Release number')

        cannon_base_url = sdk2.parameters.Url('Cannon base URL', default_value='http://wb3i2gr4lax2754l.vla.yp-c.yandex.net');
        threads = sdk2.parameters.Integer('Threads', default=1)
        rps = sdk2.parameters.Integer('RPS', default=15)

        docker_image_to_deploy = sdk2.parameters.Resource('Docker image to deploy', resource_type=FEEDBACK_PACKAGE, required=False)
        services_to_deploy = sdk2.parameters.List('List of Nanny service IDs to deploy given Docker image', default=['lt_chats_api', 'lt_chats_backend'])

        with sdk2.parameters.CheckGroup('Select ammo') as gen:
            gen.values.existing_chats = gen.Value('Existing Chats', checked=True)
            gen.values.new_chats = gen.Value('New Chats', checked=True)
            gen.values.api_chats = gen.Value('Api chats', checked=True)
            gen.values.from_file = gen.Value('From file', checked=True)

    def _deploy_to_services(self):
        docker_image = self._get_docker_image()
        if docker_image and self.Parameters.services_to_deploy:
            logging.info(u'Found Docker image %s', docker_image)
            # docker_image is like 'registry.yandex.net/webmaster/feedback:stable-45-8-5166737'
            # we need only 'webmaster/feedback:stable-45-8-5166737'
            _, docker_image = docker_image.split('/', 1)
            comment = u'Deploy {} for loadtest (SB {})'.format(docker_image, self.id)
            oauth_token = rm_sec.get_rm_token(self)
            deploy_docker_image_to_nanny_services(nanny_oauth_token=oauth_token, docker_image=docker_image, service_ids=self.Parameters.services_to_deploy, comment=comment)
        else:
            logging.info(u'Docker image not found or no services were specified for deploy. Loadtest will be performed without deploy.')

    def _get_docker_image(self):
        res = self.Parameters.docker_image_to_deploy
        if res and hasattr(res, 'resource_version'):
            return res.resource_version
        return None

    def _start_fire(self):
        fire_url = u'{}/fire'.format(self.Parameters.cannon_base_url)
        fire_params = {
            'gen': ','.join(self.Parameters.gen),
            'rps': self.Parameters.rps,
            'threads': self.Parameters.threads,
        }
        logging.info(u'Fire! Url: %s. Params: %s', fire_url, fire_params)
        response = requests.get(url=fire_url, params=fire_params)
        response.raise_for_status()
        logging.info(u'Successful: %s', response.text)

    def _get_status(self):
        status_url = u'{}/status'.format(self.Parameters.cannon_base_url)
        logging.info(u'Request status. Url: %s', status_url)
        response = requests.get(url=status_url)
        response.raise_for_status()
        logging.info(u'Successful: %s', response.text)

        return CannonStatus.from_json(response.json())

    def _fetch_output(self):
        get_output_url = u'{}/get_output'.format(self.Parameters.cannon_base_url)
        logging.info(u'Request output. Url: %s', get_output_url)
        response = requests.get(url=get_output_url)
        response.raise_for_status()
        logging.info(u'Successful')

        return response.text

    def _reload_cannon(self):
        reload_url = u'{}/reload'.format(self.Parameters.cannon_base_url)
        logging.info(u'Reload! Url: %s.', reload_url)
        response = requests.get(url=reload_url)
        response.raise_for_status()
        logging.info(u'Successful: %s', response.text)

    def _make_output_resource(self, output):
        logging.info(u'Packing output resource.')
        resource = sdk2.ResourceData(LoadtestLogs(self, 'Loadtest logs', 'loadtest.txt'))
        resource.path.write_bytes(output.encode('utf-8'))
        logging.info(u'Successful')

    def _report_startrek(self):
        release_number = self.Parameters.release_number
        component_name = self.Parameters.component_name

        if release_number and component_name:
            logging.info(u'Creating comment in release ticket for component %s release number %s', release_number, component_name)
            try:
                st = rm_st.STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
                c_info = rmc.COMPONENTS[component_name]()
                st.write_grouped_comment(
                    rm_const.TicketGroups.PerfTest,
                    "",
                    "Chats loadtest {}: started".format(link_builder.task_wiki_link(self.id)),
                    release_number,
                    c_info,
                )
            except Exception as e:
                logging.exception(u'Cannot create a comment in release ticket: %s', str(e))
        else:
            logging.info(u'The task is running not in release machine. A comment is not created in release ticket')

    def _make_semaphore_name(self):
        return u'chats_loadtest_semaphore'

    def on_enqueue(self):
        self.Requirements.semaphores = Semaphores(acquires=[Semaphores.Acquire(name=self._make_semaphore_name(), capacity=1)])

    def on_execute(self):
        self._report_startrek()
        self._deploy_to_services()
        self._reload_cannon()
        self._start_fire()

        while self._get_status().code == CannonStatus.Code.InAction:
            time.sleep(10)

        current_status = self._get_status()
        if current_status.code == CannonStatus.Code.Done:
            logging.info(u'Loadtest is finished')
            output = self._fetch_output()
            self._make_output_resource(output)
        else:
            raise common.errors.TaskError(u'Loadtest failed. {}'.format(current_status.to_json()))

        self._reload_cannon()
