import logging
import os
import copy
import subprocess
import re

from sandbox import sdk2
from sandbox.projects.yt.common.tasks.YtYaMakeUpload import YtYaMakeUpload
import sandbox.common.types.misc as ctm

logger = logging.getLogger('chyt')


class ChytUploadYtserverClickhouse(YtYaMakeUpload):
    class Parameters(YtYaMakeUpload.Parameters):
        with sdk2.parameters.Group("CHYT parameters:") as chyt_block:
            extra_tags = sdk2.parameters.List("Arbitrary tags to appear in binary filename")

    def _invoke(self, bin_path, extra_args):
        if not isinstance(extra_args, list):
            extra_args = [extra_args]
        invoke_args = [bin_path] + extra_args
        logger.debug("Invoking%s} to find out commit", invoke_args)
        output = subprocess.check_output(invoke_args, stderr=subprocess.STDOUT).strip().split()
        return list(map(str.strip, output))

    def _format_component_url(self, version, commit):
        template = "https://a.yandex-team.ru/arc_vcs/history/yt/yt/server/clckhouse_server/?from={commit}"
        return self._to_yson_url(template.format(commit=commit))

    def _to_yson_url(self, url):
        import yt.yson as yson
        return yson.to_yson_type(url, attributes={"_type_tag": "url"})

    def _init_attributes(self, bin_path):
        if self.Context.bin_attributes is not ctm.NotExists:
            return

        result = {}

        yt_version, = self._invoke(bin_path, "--yt-version")
        logger.debug("--yt-version: %s", yt_version)

        yt_commit = re.search("~([0-9a-z]*)", yt_version).groups()[0]
        result.update({
            "yt_version": yt_version,
            "yt_verion_url": self._format_component_url(yt_version, yt_commit),
        })
        ch_version, = self._invoke(bin_path, "--clickhouse-version")
        logger.debug("--clickhouse-version: %s", ch_version)
        result.update({
            "ch_version": ch_version,
        })
        chyt_version, = self._invoke(bin_path, "--version")
        logger.debug("--version: %s", chyt_version)
        chyt_commit = re.search("~([0-9a-z]*)", chyt_version).groups()[0]
        result.update({
            "version": chyt_version,
            "commit": chyt_commit,
            "version_url": self._format_component_url(chyt_version, chyt_commit)
        })

        logger.debug("Bin attributes: %s", result)

        self.Context.bin_attributes = result

    def get_destination_file_name(self, file_path):
        assert os.path.basename(file_path) == "ytserver-clickhouse"

        self._init_attributes(file_path)
        attributes = self.Context.bin_attributes
        tags = "".join("+" + tag for tag in self.Parameters.extra_tags)

        version = attributes["version"].replace("/", "__")
        filename = "ytserver-clickhouse-{}{}".format(version, tags)

        return filename

    def get_extra_attributes(self, file_path):
        self._init_attributes(file_path)

        attributes = copy.deepcopy(self.Context.bin_attributes)
        attributes.update({
            "executable": True,
        })

        return attributes
