from sandbox import sdk2
from sandbox.common.types import task as ctt

from sandbox.projects.chz import ChzRecommenderPerfTest
from sandbox.projects.chz import resources as chz_resources
from sandbox.projects.chz import shoot_common
from sandbox.projects.chz.common import tasks_bundle
from sandbox.projects.collections.recommender_base import shooting_task2
from sandbox.projects.common import dolbilka2
from sandbox.projects.tank import executor2 as tank_executor

import json
import logging


RESOURCE_REPORT_LINK = 'resource_report_link'


class ChzRecommenderPairPerfTest(tasks_bundle.ChzBinaryTask, shoot_common.PlanChooserTaskMixin, sdk2.Task):
    """ Task for testing Chz Recommender performance on two stands and comparing the result """

    class Parameters(sdk2.Parameters):
        push_tasks_resource = True
        ext_params = tasks_bundle.chz_binary_task_parameters

        shoot_params = shoot_common.ChzShootRequestParameters

        with sdk2.parameters.Group("Daemons' resources") as daemons_resources:
            test_stand_resources = sdk2.parameters.Dict('Resources for test stand')
            control_stand_resources = sdk2.parameters.Dict('Resources for control stand')

        with sdk2.parameters.Group("Extra params rewrites") as extra_params_rewrites:
            test_params_rewrites = sdk2.parameters.Dict('Params rewrites for test shoot task')
            control_params_rewrites = sdk2.parameters.Dict('Params rewrites for control shoot task')

        test = sdk2.parameters.Bool('test teches', default=False)

        with sdk2.parameters.Group("Shooting params") as shooting_params:
            dolbilka_param = dolbilka2.DolbilkaExecutor2.Parameters
            lunapark_param = tank_executor.LunaparkPlugin.Parameters
            offline_param = tank_executor.OfflinePlugin.Parameters

        with sdk2.parameters.Output():
            used_plan = sdk2.parameters.Resource('Used plan', resource_type=chz_resources.CHZ_PLAN_FOR_DOLBILO)
            test_shoot_task = sdk2.parameters.Task('Created test task', task_type=ChzRecommenderPerfTest.ChzRecommenderPerfTest)
            control_shoot_task = sdk2.parameters.Task('Created control task', task_type=ChzRecommenderPerfTest.ChzRecommenderPerfTest)

            diff_in_configurations = sdk2.parameters.Dict("Diff in resources' configurations")
            diff_in_stats = sdk2.parameters.String("Dumped pairs from diff'ed dolbilo stats")

    class Requirements(sdk2.Requirements):
        pass

    class Context(sdk2.Context):
        footer_content = ''

    def on_execute(self):
        with self.memoize_stage.launch_shoots():
            self.Parameters.used_plan = self.choose_plan(target_port=shoot_common.SERVICE_PORT)

            self.Parameters.test_shoot_task = self._run_shoot_task(
                mode='test',
                **self.Parameters.test_stand_resources
            )

            self.Parameters.control_shoot_task = self._run_shoot_task(
                mode='control',
                **self.Parameters.control_stand_resources
            )

            raise sdk2.WaitTask(
                tasks=[self.Parameters.test_shoot_task, self.Parameters.control_shoot_task],
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        self._analyze_shoots()
        self.Context.footer_content = self._make_footer_content()

    def _get_common_shooting_params(self):
        d = {
            'specify_plan': True,
            'shooting_plan': self.Parameters.used_plan,
            'write_access_log': False,
            'port': shoot_common.SERVICE_PORT,
        }

        if self.Parameters.test:
            d.update(**self.Parameters.dolbilka_param)
            d.update(**self.Parameters.lunapark_param)
            d.update(**self.Parameters.offline_param)
        else:
            d.update(self._param_block_to_dict(dolbilka2.DolbilkaExecutor2.Parameters))
            d.update(self._param_block_to_dict(tank_executor.LunaparkPlugin.Parameters))
            d.update(self._param_block_to_dict(tank_executor.OfflinePlugin.Parameters))

        if self.Parameters.use_shoot_defaults_for_chz:
            d.update(shoot_common.get_chz_specific_shoot_defaults())

        logging.debug('Result dict:\n%s', d)
        return d

    def _param_block_to_dict(self, params_cls):
        return {
            p.name: getattr(self.Parameters, p.name)
            for p in params_cls
        }

    def _run_shoot_task(self, mode, **extra_params):
        assert mode in ('test', 'control'), 'Unexpected mode: {}'.format(mode)

        extra_params.setdefault('description', '{} shoot'.format(mode))
        extra_params.setdefault('tags', []).append(mode.upper())

        extra_params.update(self._get_common_shooting_params())

        if mode == 'test':
            extra_params.update(self.Parameters.test_params_rewrites)
        elif mode == 'control':
            extra_params.update(self.Parameters.control_params_rewrites)

        task = ChzRecommenderPerfTest.ChzRecommenderPerfTest(
            self,
            **extra_params
        )

        task.enqueue()
        return task

    def _analyze_shoots(self):
        test_stats, test_config = self._get_subtask_info(self.Parameters.test_shoot_task)
        control_stats, control_config = self._get_subtask_info(self.Parameters.control_shoot_task)

        self.Parameters.diff_in_configurations = self._assert_configurations(
            test_result_config=test_config,
            control_result_config=control_config
        )

        self.Parameters.diff_in_stats = json.dumps(self._diff_stats(test_stats=test_stats, control_stats=control_stats))

    @staticmethod
    def _get_subtask_info(shoot_task):
        assert shoot_task.status in ctt.Status.Group.SUCCEED, 'Unexpected status for task {}'.format(shoot_task)

        return (
            shoot_task.Context.new_stats,
            shoot_task.Parameters.used_daemon_resources,
        )

    def _diff_stats(self, test_stats, control_stats):
        result = {}

        for stat_key, _, _ in shooting_task2.ShootingTask2.new_stats_types:
            result[stat_key] = [test_stats[stat_key], control_stats[stat_key]]

        result[RESOURCE_REPORT_LINK] = [
            shoot_common.make_ui_friendly_report_pic_link(test_stats),
            shoot_common.make_ui_friendly_report_pic_link(control_stats),
        ]

        return result

    @staticmethod
    def _assert_configuration_has_param_values(result_configuration, params_configuration):
        for k in params_configuration:
            result_item = result_configuration[k]
            expected_item = params_configuration[k]
            if result_item != expected_item:
                raise ValueError(
                    'Unmatched {} in configuration {}, expected {}'.format(k, result_configuration, expected_item)
                )

    def _assert_configurations(self, test_result_config, control_result_config):
        assert(set(test_result_config) & set(control_result_config) == set(test_result_config))

        self._assert_configuration_has_param_values(test_result_config, self.Parameters.test_stand_resources)
        self._assert_configuration_has_param_values(control_result_config, self.Parameters.control_stand_resources)

        somewhere_set = set(self.Parameters.test_stand_resources) | set(self.Parameters.control_stand_resources)

        autofilled = set(test_result_config) - somewhere_set

        diff_items = {}
        for k in autofilled:
            test_item = test_result_config[k]
            control_item = control_result_config[k]
            if test_item != control_item:
                diff_items[k] = [test_item, control_item]

        return diff_items

    @staticmethod
    def _to_td_tag(text):
        return "<td style=\"border: 1px; border-style: solid\">{}</td>".format(text)

    def _make_footer_content(self):
        body = "<tr>{t1}{t2}{t3}</tr>".format(
            t1=self._to_td_tag('statistic'),
            t2=self._to_td_tag('test'),
            t3=self._to_td_tag('control'),
        )

        footer_rules = shooting_task2.ShootingTask2.new_stats_types + (
            (RESOURCE_REPORT_LINK, "report link", "<a href='{}'>link</a>"),
        )

        diff_in_stats = json.loads(self.Parameters.diff_in_stats)
        for key, title, fmt in sorted(footer_rules):
            test_value, control_value = diff_in_stats[key]

            body += "<tr>{title}{test_stat}{control_stat}</tr>".format(
                title=self._to_td_tag(title),
                test_stat=self._to_td_tag(fmt.format(test_value)),
                control_stat=self._to_td_tag(fmt.format(control_value)),
            )

        return "<h3>Performance stats</h3><table style=\"border: 1px; border-collapse: collapse; border-style: solid\">{}</table>".format(body)

    @sdk2.footer()
    def footer(self):
        return self.Context.footer_content or "Calculating..."
