import os
import re

from sandbox import sdk2
from sandbox.common import urls

from sandbox.projects.common.vcs import arc
from sandbox.projects.chz.common import tasks_bundle
from sandbox.projects.release_machine.components.configs.chz import ci_consts

# SET(SOURCE 123)
YMAKE_RE_FOR_EXTRACTION = re.compile(r'(?<=SET\(SOURCE )\d+(?=\))')


REMOTE_BRANCH_PREFIX = 'users/robot-mlp-chz'


PRESETS = {
    'shard': 'dj/services/chz/data/shard/ya.make'
}


class ChzRecommenderPatchResourceFile(tasks_bundle.ChzBinaryTask, sdk2.Task):
    """ Task for patching chz data ya.makes in Arcadia """

    class Parameters(sdk2.Parameters):
        ext_params = tasks_bundle.chz_binary_task_parameters

        resource = sdk2.parameters.Resource('Resource id to add to patch', required=True)

        with sdk2.parameters.RadioGroup("Resource mode") as resource_mode:
            resource_mode.values["manual"] = resource_mode.Value("Manual mode", default=True)

            for k in PRESETS:
                resource_mode.values[k] = resource_mode.Value(k.capitalize())

        with resource_mode.value["manual"]:
            arcadia_ymake_subpath = sdk2.parameters.String(
                'Path to ya.make (subpath inside arcadia/ folder)',
                required=True
            )

            branch_prefix = sdk2.parameters.String('Small prefix to be included in branch name', required=True)

        arc_token_secret = sdk2.parameters.YavSecret(
            'secret with arc token for commit like sec-****#ARC_TOKEN',
            default='sec-01fan7cc3w8qj6krqqd2jyqxpy',
        )

        run_with_autorelease = sdk2.parameters.Bool(
            'Run with autorelease',
            description="Launches flow with resources' autorelease at the end (using special branch name pattern)",
            default=False
        )

        with sdk2.parameters.Output():
            pr_create_stdout = sdk2.parameters.String('PR create stdout')

    def get_patched_content(self, org_ya_make_path):
        with open(org_ya_make_path) as h:
            orig_lines = h.readlines()

        str_id = str(self.Parameters.resource.id)

        return [
            YMAKE_RE_FOR_EXTRACTION.sub(str_id, line)
            for line in orig_lines
        ]

    def on_execute(self):
        if self.Parameters.resource_mode == "manual":
            assert(self.Parameters.arcadia_ymake_subpath.endswith('/ya.make'))
            arcadia_ymake_subpath = self.Parameters.arcadia_ymake_subpath
            branch_prefix = self.Parameters.branch_prefix
        else:
            arcadia_ymake_subpath = PRESETS[self.Parameters.resource_mode]
            branch_prefix = self.Parameters.resource_mode

        branch_name = self.branch_name(branch_prefix)
        commit_msg = "Up {ymake} with resource {res}\n\nPR create task: {task}".format(
            ymake=arcadia_ymake_subpath,
            res=urls.get_resource_link(self.Parameters.resource.id),
            task=urls.get_task_link(self.id),
        )

        arc_client = arc.Arc(arc_oauth_token=self.Parameters.arc_token_secret.data()['ARC_TOKEN'])
        with arc_client.mount_path(path=None, changeset=None, fetch_all=False) as arc_mount_path:
            arc_client.checkout(arc_mount_path, branch=branch_name, create_branch=True)

            ya_make_path = os.path.join(arc_mount_path, arcadia_ymake_subpath)
            patch_lines = self.get_patched_content(ya_make_path)

            with open(ya_make_path, 'w') as mutable_ya_make:
                mutable_ya_make.write(''.join(patch_lines))

            arc_client.add(arc_mount_path, path=ya_make_path)
            arc_client.commit(arc_mount_path, message=commit_msg)
            arc_client.push(arc_mount_path, upstream='{}/{}'.format(REMOTE_BRANCH_PREFIX, branch_name))
            self.Parameters.pr_create_stdout = arc_client.pr_create(arc_mount_path, message=branch_name, publish=True)

    def branch_name(self, branch_prefix):
        if self.Parameters.run_with_autorelease:
            branch_prefix = ci_consts.AUTORELEASE_BRANCH_PREFIX + "_" + branch_prefix

        return branch_prefix + "_" + str(self.Parameters.resource.id)
