from sandbox import sdk2
from sandbox.sdk2 import helpers as sdk_helpers

from sandbox.projects.chz import resources as chz_resources
from sandbox.projects.chz import shoot_common
from sandbox.projects.chz.common import tasks_bundle
from sandbox.projects.common import dolbilka2
from sandbox.projects.collections.recommender_base import shooting_task2
from sandbox.projects.dj.services.chz import fs_manipulate
from sandbox.projects.dj.services.chz.instances import sandbox_env as sandbox_env_instances
from sandbox.projects.dj.services.chz.specs import specs
from sandbox.projects.dj.services.chz.spec_in_tasks import parameters as spec_in_tasks_parameters
from sandbox.projects.dj.services.chz.spec_in_tasks import resources as spec_in_tasks_resources
from sandbox.projects.tank import executor2 as tank_executor

import attr
import json
import logging
import os
import tempfile


ACCESS_LOG_MAX_ROWS_PER_YT_WRITE_REQUEST = 1024


class ChzRecommenderPerfTest(tasks_bundle.ChzBinaryTask, shooting_task2.ShootingTask2, shoot_common.PlanChooserTaskMixin, sdk2.Task):
    """ Task for testing base Chz Recommender performance """

    class Parameters(sdk2.Parameters):
        ext_params = tasks_bundle.chz_binary_task_parameters

        shoot_params = shoot_common.ChzShootRequestParameters

        with sdk2.parameters.Group('Daemon resources'):
            chz_daemon_parameters = spec_in_tasks_parameters.chz_daemon_parameters

        port = sdk2.parameters.Integer('Port to run on', default=shoot_common.SERVICE_PORT)

        write_access_log = sdk2.parameters.Bool('Write log to YT', default=False)
        with write_access_log.value[True]:
            yt_token = sdk2.parameters.Vault('yt token vault entry (owner:name or name of the secret or Vault path)')
            access_log_yt_table = sdk2.parameters.String('yt table')
            access_log_yt_cluster = sdk2.parameters.String('yt cluster', default='hahn')
            merge_log_table = sdk2.parameters.Bool("merge written logs' table", default=False)

        dolbilka_param = dolbilka2.DolbilkaExecutor2.Parameters
        lunapark_param = tank_executor.LunaparkPlugin.Parameters
        offline_param = tank_executor.OfflinePlugin.Parameters

        with sdk2.parameters.Output():
            used_plan = sdk2.parameters.Resource('Used plan', resource_type=chz_resources.CHZ_PLAN_FOR_DOLBILO)
            used_daemon_resources = sdk2.parameters.Dict('Used daemon resources')
            shooting_stats = sdk2.parameters.Dict('Stats from dolbilo')

    class Requirements(sdk2.Requirements):
        disk_space = 10 * 1024
        client_tags = shooting_task2.ShootingTask2.client_tags

    def on_execute(self):
        working_dir = tempfile.mkdtemp(dir=os.getcwd())
        with sdk_helpers.ProcessLog(self, logger="chz") as process_log:
            self._process(working_dir, process_log)

    def _process(self, working_dir, process_log):
        self.Parameters.used_plan = self.choose_plan(target_port=self.Parameters.port)

        spec = spec_in_tasks_parameters.spec_object_from_parameters_with_defaults(
            parameters=self.Parameters,
            spec_type=specs.ChzRecommenderSpec,
        )

        dict_spec = attr.asdict(spec)
        logging.debug('Target spec: %s', dict_spec)
        self.Parameters.used_daemon_resources = dict_spec

        paths_spec = spec_in_tasks_resources.download_spec_resources(spec)
        deploy_steps = fs_manipulate.spec_to_deploy_steps(paths_spec)
        for step in deploy_steps:
            step(working_dir)

        with sandbox_env_instances.ChzRecommenderInSandbox(
            binary_path=os.path.join(working_dir, "recommender", "recommender_server"),
            http_port=self.Parameters.port,
            config_path=os.path.join(working_dir, "recommender", "config.pbtxt"),
            working_dir=working_dir,
            process_log=process_log
        ):
            self._init_virtualenv()
            self._dolbilo_shoot(
                port=self.Parameters.port,
                plan_resource=self.Parameters.used_plan,
                variant="chz_recommender"
            )

        self.Parameters.shooting_stats = getattr(self.Context, self.new_stats_key)

        if self.Parameters.write_access_log:
            log_path = os.path.join(working_dir, "recommender.log")
            self._write_logs_to_yt(log_path)

    def _write_logs_to_yt(self, path):
        import yt.wrapper as yt

        output_table = self.Parameters.access_log_yt_table

        if not output_table:
            return

        yt.config['proxy']['url'] = self.Parameters.access_log_yt_cluster
        yt.config['token'] = self.Parameters.yt_token.data()

        yt.create_table(output_table, recursive=True)
        output_table_with_append = yt.TablePath(output_table, append=True)

        with open(path) as log_file:
            rows_batch = []

            for row in log_file:
                rows_batch.append(json.loads(row))
                if len(rows_batch) >= ACCESS_LOG_MAX_ROWS_PER_YT_WRITE_REQUEST:
                    yt.write_table(output_table_with_append, rows_batch)
                    rows_batch = []
            else:
                if rows_batch:
                    yt.write_table(output_table_with_append, rows_batch)

        if self.Parameters.merge_log_table:
            yt.run_merge(output_table, output_table, spec=dict(combine_chunks=True))

    @sdk2.footer()
    def footer(self):
        return shooting_task2.ShootingTask2.footer(self)
