import attr
import logging
import time

from sandbox import sdk2
from sandbox.common.types import notification as ctn

from sandbox.projects.chz.common import tasks_bundle
from sandbox.projects.geosearch.AddrsBaseAutoreleaser import utils
from sandbox.projects.geosearch.tools import abc_duty
from sandbox.projects.dj.services.chz.specs import specs


CHZ_ABC_SERVICE_ID = 34113

CHZ_SANDBOX_GROUP = 'CHZ'


SLEEP_BEFORE_RELEASE = 15 * 60


class ChzReleaseNotifier(tasks_bundle.ChzBinaryTask, sdk2.Task):
    """ Task for pinging chz duty chat about release """

    class Parameters(sdk2.Parameters):
        ext_params = tasks_bundle.chz_binary_task_parameters

        ci_job_url = sdk2.parameters.String('Ci job url')
        resource_task_id = sdk2.parameters.Task('Task id to release')

        abc_token_secret = sdk2.parameters.YavSecret(
            'secret with abc token for abc duty api request like sec-****#abc_token',
            default=None,
        )

        with sdk2.parameters.RadioGroup("Release resource") as release_resource:
            for resource_attrib in attr.fields(specs.ChzRecommenderSpec):
                release_resource.values[resource_attrib.name] = release_resource.Value(resource_attrib.name)

    @staticmethod
    def wait_until_release_time():
        hours_to_sleep = utils.get_sleep_to_release_hours(morning=11, night=21)
        if hours_to_sleep:
            logging.info('Sleeping until 11 am')
            raise sdk2.WaitTime(hours_to_sleep * 60 * 60)

    def get_notify_list(self):
        if self.Parameters.abc_token_secret:
            abc_token = self.Parameters.abc_token_secret.data()['abc_token']
        else:
            abc_token = sdk2.Vault.data('GEOSEARCH_PROD', 'geosearch_abc_token')

        notify_list = abc_duty.get_service_duty_logins(
            abc_service_id=CHZ_ABC_SERVICE_ID,
            parameters=None,
            abc_token=abc_token,
            check_status=True,
        )
        logging.debug('Duty logins: %s', notify_list)
        notify_list.append(CHZ_SANDBOX_GROUP)
        return notify_list

    def get_report_message(self):
        return (
            'Task with {resource_type} is about to release '
            'in {sleep_time} minutes.\nTo cancel, stop flow in ci: '
            '{ci_job_url}'
        ).format(
            resource_type=self.Parameters.release_resource,
            sleep_time=SLEEP_BEFORE_RELEASE / 60,
            ci_job_url=self.Parameters.ci_job_url,
        )

    def on_execute(self):
        self.wait_until_release_time()

        message = self.get_report_message()
        notify_list = self.get_notify_list()

        self.server.notification(
            body=message,
            recipients=notify_list,
            transport=ctn.Transport.TELEGRAM
        )

        logging.info('Sleeping for {sleep_time} seconds'.format(sleep_time=SLEEP_BEFORE_RELEASE))
        time.sleep(SLEEP_BEFORE_RELEASE)
