import copy
import os

from sandbox import sdk2
from sandbox.common.types import resource as ctr

from sandbox.projects.chz import resources as chz_resources
from sandbox.projects.common import dolbilka2
from sandbox.projects.resource_types import PLAIN_TEXT_QUERIES


SERVICE_PORT = 15530

CHZ_AMMO_ATTR_OK_FOR_REGULAR = 'chz_ammo_ok_for_regular_shoots'
CHZ_AMMO_ATTR_REQUEST_TYPE = 'chz_ammo_requests_type'


def get_chz_specific_shoot_defaults():
    return {
        dolbilka2.DolbilkaExecutor2.Parameters.dolbilka_executor_max_simultaneous_requests.name: 10,
        dolbilka2.DolbilkaExecutor2.Parameters.dolbilka_executor_requests_limit.name: 50000,
    }


class ChzShootRequestParameters(sdk2.Task.Parameters):
    specify_plan = sdk2.parameters.Bool('Specify plan', default=False)

    with specify_plan.value[True]:
        shooting_plan = sdk2.parameters.Resource(
            'Resource with dolbilka plan',
            resource_type=chz_resources.CHZ_PLAN_FOR_DOLBILO,
        )

    with specify_plan.value[False]:
        select_plan = sdk2.parameters.Bool('Select plan', default=False)

        with select_plan.value[True]:
            with sdk2.parameters.RadioGroup("Ammo requests type") as ammo_requests_type:
                ammo_requests_type.values["chz"] = ammo_requests_type.Value("chz", default=True)
                ammo_requests_type.values["menu"] = ammo_requests_type.Value("menu")

            plan_attributes = sdk2.parameters.Dict(
                'Required attributes for plan resource', default={CHZ_AMMO_ATTR_OK_FOR_REGULAR: 1}
            )
        with select_plan.value[False]:
            text_queries = sdk2.parameters.Resource('Resource with plain text queries', resource_type=PLAIN_TEXT_QUERIES)

    use_shoot_defaults_for_chz = sdk2.parameters.Bool('Use some defaults that keep server alive', default=True)


class PlanChooserTaskMixin(object):
    def choose_plan(self, target_port=None):
        resource = None

        if self.Parameters.specify_plan:
            resource = self.Parameters.shooting_plan

        elif self.Parameters.select_plan:
            attrs = copy.deepcopy(self.Parameters.plan_attributes)
            attrs[CHZ_AMMO_ATTR_REQUEST_TYPE] = self.Parameters.ammo_requests_type

            resource = sdk2.Resource.find(
                type=chz_resources.CHZ_PLAN_FOR_DOLBILO,
                state=ctr.State.READY,
                attrs=attrs,
            ).first()

        elif self.Parameters.text_queries:
            text_queries_data = sdk2.ResourceData(self.Parameters.text_queries)

            optional_params = {}
            if self.Parameters.dolbilka_fixed_rps:
                optional_params['rps'] = self.Parameters.dolbilka_fixed_rps

            src_path = str(text_queries_data.path)
            dst_file_name = os.path.basename(src_path) + '.plan'

            resource = chz_resources.CHZ_PLAN_FOR_DOLBILO(
                self,
                "Chz shoots plan from {}".format(self.Parameters.text_queries.id),
                dst_file_name
            )
            resource.from_text_queries = self.Parameters.text_queries.id
            resource_data = sdk2.ResourceData(resource)

            planner = dolbilka2.DolbilkaPlanner2()
            planner.create_plan(
                queries_path=src_path,
                result_path=str(resource_data.path),
                host='localhost',
                port=target_port,
                **optional_params
            )

            resource_data.ready()

        return resource


def make_ui_friendly_report_pic_link(stats):
    return "//proxy.sandbox.yandex-team.ru/{}/{}".format(stats["report_resource"], stats["report_path"])
