# -*- coding: utf-8 -*-

import logging
import os
import subprocess
import time
import urllib
import urllib2
import thread

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.common.types.task import Status
from sandbox.projects.clickhouse.util.github_helper import GithubHelper
from sandbox.projects.clickhouse.util.resource_helper import ResourceHelper
from sandbox.projects.clickhouse.util.s3_helper import S3Helper
from sandbox.sandboxsdk import environments
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.projects.common.binary_task as binary_task
import sandbox.sdk2.helpers

from sandbox.projects.clickhouse.util.clickhouse_helper import ClickHouseHelper

CONFIGURE_DOCKER = """
#!/usr/bin/env bash
set -x
export DEBIAN_FRONTEND=non interactive
echo "# New path to docker root: $1"
mkdir -p "$1"

cat > /etc/docker/daemon.json << EOF
{
    "ipv6": true,
    "fixed-cidr-v6": "fd00::/8",
    "ip-forward": true,
    "data-root": "$1",
    "insecure-registries" : ["dockerhub-proxy.sas.yp-c.yandex.net:5000"],
    "registry-mirrors" : ["http://dockerhub-proxy.sas.yp-c.yandex.net:5000"]
}
EOF

systemctl restart systemd-resolved.service
systemctl daemon-reload
systemctl restart docker
docker info
"""


class PostStatuses(object):
    NEVER = 0
    ON_ERROR = 1
    ALWAYS = 2


class NeedToRunDescription(object):
    def __init__(self, need_to_run, description="", post_statuses=False):
        self.need_to_run = need_to_run
        self.description = description
        self.post_statuses = post_statuses


class BaseOnCommitTask(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("GitHub parameters") as gh_block:
            github_token_key = sdk2.parameters.String("Token for github interactions", default="clickhouse-robot-github-token")
            github_repo = sdk2.parameters.String("GitHub repo name", default="ClickHouse/ClickHouse")
            email = sdk2.parameters.String("Email for git", default="robot-clickhouse@yandex-team.ru")

        with sdk2.parameters.Group("Commit parameters") as commit_block:
            commit_sha = sdk2.parameters.String("Commit sha")
            pull_request_number = sdk2.parameters.Integer("PR number (0 means master)")

        with sdk2.parameters.Group("S3 parameters") as s3_block:
            s3_access_key_id = sdk2.parameters.String("S3 key id", default="clickhouse-robot-s3-key-id")
            s3_access_key = sdk2.parameters.String("S3 key", default="clickhouse-robot-s3-key")

        with sdk2.parameters.Group("Secrets parameters") as secrets_block:
            ssh_key = sdk2.parameters.String("SSH-key valut name", default="robot-clickhouse-ssh")
            clickhouse_url = sdk2.parameters.String("ClickHouse for test stats URL", default="clickhouse-test-stat-url")
            clickhouse_login = sdk2.parameters.String("Login for test stats ClickHouse", default="clickhouse-test-stat-login")
            clickhouse_password = sdk2.parameters.String("Password for test stats ClickHouse", default="clickhouse-test-stat-password")

        with sdk2.parameters.Group("Other parameters") as other_block:
            force_run = sdk2.parameters.Bool("Force run, ignore other checks")
            docker_images_with_versions = sdk2.parameters.JSON("Docker image version (unused for tasks without Docker)", default={})
            ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "CLICKHOUSE_SANDBOX_BINARY", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    @staticmethod
    def get_context_name():
        raise Exception("Unimplemented")

    # return state, message and report url
    def process(self, commit, repo, pull_request):
        raise Exception("Unimplemented")

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        raise Exception("Unimplemented")

    def post_statuses(self):
        raise Exception("Unimplemented")

    @staticmethod
    def need_docker():
        return False

    @staticmethod
    def docker_root_on_tmpfs():
        return False

    @staticmethod
    def require_internet():
        return False

    # Determines the order in which we try to start the checks for a PR.
    # First we start all the kinds of checks with order() = 0, then order() = 1,
    # etc. Checks with the same order() value are started in randomized relative
    # order.
    # It may be helpful to think of order() values as logarithmic buckets -- [0, 9),
    # [10, 99), [100, 999), etc. First tier are the task that are hard dependencies
    # for other tasks, such as builds, then go the very long tasks that we have
    # to start early, such as performance and integration tests, then the bulk of
    # functional tests, and then other less important things like style and docs.
    # Note that these are not priorities of tasks, but just the order in which
    # they are started.
    @staticmethod
    def order():
        return 10000

    @classmethod
    def sandbox_task_priority(cls):
        if cls.order() < 10:
            return ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH)
        elif cls.order() < 100:
            return ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)
        else:
            return ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.LOW)

    @staticmethod
    def need_to_run(pr_info):
        return NeedToRunDescription(True)

    @staticmethod
    def get_images_names():
        return []

    def get_image_with_version(self, name):
        if name in self.Parameters.docker_images_with_versions:
            return name + ":" + self.Parameters.docker_images_with_versions[name]
        # We migrating from yandex repo to clickhouse
        new_name = name.replace('yandex/clickhouse-', 'clickhouse/')
        if new_name in self.Parameters.docker_images_with_versions:
            return new_name + ":" + self.Parameters.docker_images_with_versions[new_name]

        logging.warn("Cannot find image %s (and %s) in params list %s", name, new_name, self.Parameters.docker_images_with_versions)
        if ':' not in name:
            return name + ":latest"
        return name

    def get_single_image_with_version(self):
        return self.get_image_with_version(self.get_images_names()[0])

    def get_single_image_version(self):
        name = self.get_images_names()[0]
        if name in self.Parameters.docker_images_with_versions:
            return self.Parameters.docker_images_with_versions[name]

        new_name = name.replace('yandex/clickhouse-', 'clickhouse/')
        if new_name in self.Parameters.docker_images_with_versions:
            return new_name + ":" + self.Parameters.docker_images_with_versions[new_name]

        logging.warn("Cannot find image %s (and %s) in params list %s", name, new_name, self.Parameters.docker_images_with_versions)
        return 'latest'

    def _prepare_docker(self):
        docker_script_file = os.path.join(str(self.path()), 'docker_script')
        if self.docker_root_on_tmpfs():
            if not self.ramdrive or not self.ramdrive.path:
                raise Exception("Required to place docker root on tmpfs but no ramdrive available")
            logging.info("Will configure docker root on ramdrive %s", str(self.ramdrive.path))
            docker_root = os.path.join(str(self.ramdrive.path), 'docker_root')
        else:
            docker_root = os.path.join(str(self.path()), 'docker_root')

        logging.info("Docker root %s", docker_root)
        with open(docker_script_file, 'w') as fscript:
            fscript.write(CONFIGURE_DOCKER)

        with sandbox.sdk2.helpers.ProcessLog(self, logger="docker_prepare") as pl:
            retcode = subprocess.Popen("bash {} {}".format(docker_script_file, docker_root), shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            if retcode == 0:
                logging.info("Checking internet connection from docker")
                # There is an issue with Sandbox infrastructure - sometimes network takes long time to become available.

                num_tries = 50
                for i in range(num_tries):
                    try:
                        logging.info("Pinging internet from docker num %s", i)
                        subprocess.check_call("docker run -i --rm busybox ping6 github.com -c4", shell=True)
                        break
                    except:
                        if i + 1 == num_tries:
                            raise
                        logging.info("Will retry")
                        time.sleep(0.5)
                logging.info("Docker prepared successfully")
                os.environ["DOCKER_API_VERSION"] = self.get_docker_api_version()
                os.remove(docker_script_file)
            else:
                os.remove(docker_script_file)
                raise Exception("Docker prepare failed")

    def get_docker_api_version(self):
        result = subprocess.check_output(['docker', 'version', '--format', '{{.Server.APIVersion}}'])
        return str(result).strip()

    def docker_sock(self):
        return ""

    def _wrap_url(self, task_url):
        encoded_url = urllib.quote_plus(task_url)
        fetcher = urllib2.urlopen('https://nda.ya.ru/--?url=' + encoded_url)
        return fetcher.read()

    def _get_self_url(self):
        return "https://sandbox.yandex-team.ru/task/{}/view".format(self.id)

    def _create_s3_client(self):
        s3_access_key_id = sdk2.Vault.data(self.Parameters.s3_access_key_id)
        s3_access_key = sdk2.Vault.data(self.Parameters.s3_access_key)
        return S3Helper(host='http://s3.mds.yandex.net',
                        aws_access_key_id=s3_access_key_id,
                        aws_secret_access_key=s3_access_key)

    def update_status(self, text, state="pending", url="", commit=None):
        if not url:
            url = self.task_url
        if not commit:
            commit = self.commit
        commit.create_status(state=state, context=self.get_context_name(), description=text[:140], target_url=url)

    def pull_my_docker_images(self):
        logging.info("Pulling images %s", ', '.join(self.get_images_names()))
        for image in self.get_images_names():
            image_with_version = self.get_image_with_version(image)
            exception = None
            for i in range(10):
                try:
                    logging.info("Pulling image %s", image_with_version)
                    output = subprocess.check_output("docker pull {}".format(image_with_version), shell=True)
                    logging.info("Command finished, output %s", output)
                    break
                except Exception as ex:
                    logging.info("Exception updating image %s: %s", image_with_version, str(ex))
                    time.sleep(i * 5)
                    exception = ex
            else:
                raise exception

    def on_execute(self):
        venv = environments.VirtualEnvironment()

        with venv as venv:
            if self.Parameters.binary_executor_release_type == "none":
                environments.PipEnvironment('pygithub', venv=venv).prepare()
                environments.PipEnvironment('boto3', venv=venv).prepare()
                logging.debug("Set pip environment for nonbinary release")

            thread.start_new_thread(self.on_timeout_func, ())
            logging.getLogger().setLevel(logging.INFO)
            github_token = sdk2.Vault.data(self.Parameters.github_token_key)
            self.task_owner = self.Parameters.owner
            self.user_email = self.Parameters.email
            self.ssh_key = self.Parameters.ssh_key
            self.gh_helper = GithubHelper(github_token)
            self.s3_client = self._create_s3_client()
            self.clickhouse_helper = ClickHouseHelper(
                sdk2.Vault.data(self.Parameters.clickhouse_url),
                sdk2.Vault.data(self.Parameters.clickhouse_login),
                sdk2.Vault.data(self.Parameters.clickhouse_password))

            rate_is_ok, message = self.gh_helper.check_rate_limit()
            if not rate_is_ok:
                # We should return Exception status here, so that the task is
                # restarted.
                raise Exception("[can't run :(] " + message)

            repo = self.gh_helper.get_repo(self.Parameters.github_repo)

            if self.Parameters.pull_request_number != 0:
                logging.info("Fetching pull request %ld", self.Parameters.pull_request_number)
                self.pull_request = repo.get_pull(number=self.Parameters.pull_request_number)
                logging.info("Pull request fetched")
            else:
                self.pull_request = self.gh_helper.get_fake_pr()

            logging.info("Fetching info for commit %s", self.Parameters.commit_sha)
            self.commit = repo.get_commit(self.Parameters.commit_sha)
            logging.info("Commit fetched")

            self.task_url = self._wrap_url(self._get_self_url())

            self.resource_helper = ResourceHelper(self.commit, repo, self.pull_request)

            try:
                if self.need_docker():
                    if not self.Requirements.privileged:
                        raise Exception("Docker can be used in privileged containers only")
                    self._prepare_docker()
                    self.pull_my_docker_images()

                state, description, url = self.process(self.commit, repo, self.pull_request)

                logging.info("process() returns '%s', '%s', '%s'", str(state), str(description), str(url))

                # Scroll to the first failure
                if state == "failure" and url is not None:
                    url = url + '#fail1'

                if url:
                    self.Parameters.description = '<a href="{}">Report</a>\n'.format(url) + self.Parameters.description

            except subprocess.CalledProcessError as ex:
                logging.info("Job call failed with %s", str(ex.output))
                if self.post_statuses() >= PostStatuses.ON_ERROR:
                    self.update_status(state="error", text=ex.message)
                raise
            except Exception as e:
                logging.info("Job failed with %s", str(e))
                if self.post_statuses() >= PostStatuses.ON_ERROR:
                    self._make_error_status_with_retry("Some exception occurred :(", 10)
                raise

    def _make_error_status_with_retry(self, message, retry_count):
        for i in range(retry_count):
            try:
                logging.info("Creating report status %s for commit %s", self.get_context_name(), self.commit.sha)
                self.commit.create_status(
                    state="error",
                    context=self.get_context_name(),
                    description=message[:140],
                    target_url=self.task_url)
                break
            except:
                time.sleep(i * 5)
        else:
            logging.info("GitHub api is not responding")

    def on_failure(self, prev_status):
        if self.post_statuses() >= PostStatuses.ON_ERROR:
            self._make_error_status_with_retry("Something went wrong :(", 10)

    def on_break(self, prev_status, status):
        if status in (Status.TIMEOUT, Status.EXCEPTION, Status.NO_RES) and self.post_statuses() >= PostStatuses.ON_ERROR:
            self._make_error_status_with_retry("Some exception occurred :(", 10)

    def on_timeout_func(self):
        start = time.time()
        while time.time() - start < self.Parameters.kill_timeout - 45:
            time.sleep(30)

        logging.info("Timeout received will try to post status")
        if self.post_statuses() >= PostStatuses.ON_ERROR:
            self._make_error_status_with_retry("Timeout :(", 10)
