# -*- coding: utf-8 -*-
from sandbox.projects.clickhouse.resources import CLICKHOUSE_BUILD_LXC_CONTAINER
from sandbox.projects.clickhouse.BaseOnCommitTask.test_task import BaseOnCommitTestTask
from sandbox.projects.clickhouse.BaseOnCommitTask.base import PostStatuses, NeedToRunDescription
from sandbox.projects.clickhouse.ClickhouseBuilder import get_build_parameters_as_string
from sandbox.projects.clickhouse.util.task_helper import get_ci_config
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
from sandbox import sdk2
import subprocess
import logging
import requests
import traceback
import datetime
import math


class ClickhouseFuzzer(BaseOnCommitTestTask):

    class Requirements(BaseOnCommitTestTask.Requirements):
        privileged = True
        client_tags = ctc.Tag.GENERIC

    class Parameters(BaseOnCommitTestTask.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            resource_type=CLICKHOUSE_BUILD_LXC_CONTAINER,
        )
        kill_timeout = 1.5 * 60 * 60

    def on_create(self):
        self.Parameters._container = sdk2.Resource.find(
            CLICKHOUSE_BUILD_LXC_CONTAINER,
            state=ctr.State.READY,
            attrs=dict(released="stable")
        ).order(-CLICKHOUSE_BUILD_LXC_CONTAINER.id).first().id

    @staticmethod
    def get_context_name():
        return 'AST fuzzer'

    @staticmethod
    def binary_to_download():
        return 'clang-11_debug_none_bundled_unsplitted_disable_False_binary'

    @staticmethod
    def kind():
        return 'debug'

    def process(self, commit, repo, pull_request):
        check_start_time = datetime.datetime.utcnow()
        logging.info("commit " + str(commit) + ", pr " + str(pull_request))

        subprocess.Popen('ls -lath', shell=True).wait()
        subprocess.Popen('ls -lath $(pwd)', shell=True).wait()

        ci_config = get_ci_config(pull_request, commit)
        if ci_config and self.get_context_name() in ci_config["tests_config"]:
            test_config = ci_config["tests_config"][self.get_context_name()]["required_build_properties"]
            build_path = get_build_parameters_as_string(
                test_config["compiler"], test_config["build_type"], test_config["sanitizer"],
                test_config["bundled"], test_config["splitted"], test_config["clang-tidy"],
                test_config["with_coverage"], test_config["package_type"])
            logging.info("CI config found will search build at %s", build_path)
        else:
            build_path = self.binary_to_download()

        cmd = 'docker run --network=host --volume=$(pwd)/workspace:/workspace '\
              '--cap-add syslog --cap-add sys_admin '\
              '-e PR_TO_TEST={} -e SHA_TO_TEST={} -e BINARY_TO_DOWNLOAD={} '\
              '{}'.format(
                  str(pull_request.number),
                  str(commit.sha),
                  build_path,
                  self.get_single_image_with_version())

        logging.info('run cmd {}'.format(cmd))
        retcode = subprocess.Popen(cmd, shell=True).wait()
        logging.info('retcode {}'.format(str(retcode)))

        subprocess.Popen('date', shell=True).wait()
        subprocess.Popen('pstree -aspgT', shell=True).wait()
        subprocess.Popen('free -h', shell=True).wait()
        subprocess.Popen('ps -eo pid,cmd,%cpu,%mem --sort=-%mem | head -20', shell=True).wait()
        subprocess.Popen('df -h', shell=True).wait()
        subprocess.Popen('docker volume ls', shell=True).wait()
        subprocess.Popen('ls -lath', shell=True).wait()
        subprocess.Popen('ls -lath workspace', shell=True).wait()
        subprocess.Popen('ls -lath $(pwd)', shell=True).wait()
        subprocess.Popen('tree', shell=True).wait()

        s3_prefix = '{}/{}/fuzzer_{}/'.format(str(pull_request.number), str(commit.sha), self.kind())
        paths = {
            'main.log': 'workspace/main.log',
            'server.log': 'workspace/server.log',
            'fuzzer.log': 'workspace/fuzzer.log',
            'report.html': 'workspace/report.html',
        }
        for file in paths:
            try:
                paths[file] = self.s3_client.upload_test_report_to_s3(paths[file], s3_prefix + file)
            except:
                paths[file] = ''
                traceback.print_exc()
                pass

        report_url = self.task_url
        if paths['main.log']:
            report_url = paths['main.log']
        if paths['server.log']:
            report_url = paths['server.log']
        if paths['fuzzer.log']:
            report_url = paths['fuzzer.log']
        if paths['report.html']:
            report_url = paths['report.html']

        # Try to get status message saved by the fuzzer
        try:
            status = open('workspace/status.txt').readline().rstrip('\n')
            description = open('workspace/description.txt').readline().rstrip('\n')[:140]
        except:
            status = 'failure'
            description = 'Task failed: $?=' + str(retcode)
            traceback.print_exc()
            pass

        try:
            self.clickhouse_helper.insert_event_into('gh-data', 'checks', {
                'pull_request_number': pull_request.number,
                'commit_sha': commit.sha,
                'check_name': self.get_context_name(),
                'check_status': status,
                'check_duration_ms': int(math.ceil((datetime.datetime.utcnow() - check_start_time).total_seconds() * 1000)),
                'check_start_time': str(check_start_time),
                'test_name': '',
                'test_status': description,
                'test_duration_ms': 0,
                'report_url': report_url,
                'pull_request_url': '',
                'commit_url': '',
                'task_url': self.task_url,
                'base_ref': '',
                'base_repo': '',
                'head_ref': '',
                'head_repo': ''})
        except:
            traceback.print_exc()
            pass

        logging.info("Result: '%s', '%s', '%s'", status, description, report_url)

        return status, description, report_url

    @staticmethod
    def get_images_names():
        return ["yandex/clickhouse-fuzzer"]

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        # The build is uploaded to S3 some time after the Sandbox resource is published,
        # so there is no point checking for the Sandbox resource, and we just check that
        # it did upload.
        ci_config = get_ci_config(pull_request, commit)
        if ci_config and cls.get_context_name() in ci_config["tests_config"]:
            test_config = ci_config["tests_config"][cls.get_context_name()]["required_build_properties"]
            build_path = get_build_parameters_as_string(
                test_config["compiler"], test_config["build_type"], test_config["sanitizer"],
                test_config["bundled"], test_config["splitted"], test_config["clang-tidy"],
                test_config["with_coverage"], test_config["package_type"])
            logging.info("CI config found will search build at %s", build_path)
        else:
            build_path = cls.binary_to_download()

        s3_build = 'https://clickhouse-builds.s3.yandex.net/{}/{}/clickhouse_build_check/{}/clickhouse'.format(
            pull_request.number, commit.sha, build_path)
        response = requests.head(s3_build)
        if response.ok:
            return True
        try:
            response.raise_for_status()
        except BaseException as error:
            logging.info("Exception while downloading the clickhouse binary: '{}'".format(error))
        return False

    def post_statuses(self):
        return PostStatuses.ALWAYS

    @staticmethod
    def need_docker():
        return True

    @staticmethod
    def require_internet():
        return True

    @staticmethod
    def need_to_run(pr_info):
        if 'pr-backport' in pr_info.labels or 'release' in pr_info.labels:
            return NeedToRunDescription(False, 'Not ran for backport or release PRs', False)

        return BaseOnCommitTestTask.need_to_run(pr_info)
