# -*- coding: utf-8 -*-
import json
import logging
import os
import ast
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import time
from sandbox.projects.clickhouse.util.report import create_build_html_report

from sandbox import sdk2
from sandbox.projects.clickhouse.BaseOnCommitTask.base import PostStatuses
from sandbox.projects.clickhouse.BaseOnCommitTask.base import BaseOnCommitTask
from sandbox.projects.clickhouse.ClickhouseBuilder import ClickhouseBuilder
from sandbox.projects.clickhouse.resources import CLICKHOUSE_REPO
from sandbox.projects.clickhouse.util.task_helper import get_ci_config
from sandbox.projects.clickhouse.util.task_helper import has_changes_in_code
from sandbox.projects.clickhouse.BaseOnCommitTask.base import NeedToRunDescription

# These builds are just default fallback, use file tests/ci/ci_config.json from
# clickhouse repository to configure builds
DEFAULT_BUILDS = [
    {
        "compiler": "gcc-9",
        "build-type": "",
        "sanitizer": "",
        "package-type": "deb",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "alien_pkgs": True,
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "gcc-9",
        "build-type": "",
        "sanitizer": "",
        "package-type": "performance",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "gcc-9",
        "build-type": "",
        "sanitizer": "",
        "package-type": "binary",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "",
        "sanitizer": "address",
        "package-type": "deb",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "",
        "sanitizer": "undefined",
        "package-type": "deb",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "",
        "sanitizer": "thread",
        "package-type": "deb",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "",
        "sanitizer": "memory",
        "package-type": "deb",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "",
        "sanitizer": "",
        "package-type": "deb",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "debug",
        "sanitizer": "",
        "package-type": "deb",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "",
        "sanitizer": "",
        "package-type": "binary",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10",
        "build-type": "",
        "sanitizer": "",
        "package-type": "binary",
        "bundled": "bundled",
        "splitted": "splitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10-darwin",
        "build-type": "",
        "sanitizer": "",
        "package-type": "binary",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10-aarch64",
        "build-type": "",
        "sanitizer": "",
        "package-type": "binary",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
    {
        "compiler": "clang-10-freebsd",
        "build-type": "",
        "sanitizer": "",
        "package-type": "binary",
        "bundled": "bundled",
        "splitted": "unsplitted",
        "tidy": "disable",
        "with_coverage": False,
    },
]


class BuildResult(object):
    def __init__(self, compiler, build_type, sanitizer, bundled, splitted, status, elapsed_seconds, with_coverage):
        self.compiler = compiler
        self.build_type = build_type
        self.sanitizer = sanitizer
        self.bundled = bundled
        self.splitted = splitted
        self.status = status
        self.elapsed_seconds = elapsed_seconds
        self.with_coverage = with_coverage


class ClickhouseBuildLauncher(BaseOnCommitTask):
    """
    Task build ClickHouse with different parameters
    """
    class Parameters(BaseOnCommitTask.Parameters):
        kill_timeout = 2 * 60 * 60 + 60 * 60  # 3 hour
        force_rebuild = sdk2.parameters.Bool("Force rebuild all builds")
        build_setups = sdk2.parameters.JSON(
            'Build setups is JSON of dicts [{"compiler":"gcc-9", "build-type":"debug", "sanitizer": "none", "package-type": "binary", "bundled": "bundled"}, {...}]',
            default=DEFAULT_BUILDS,
            required=True
        )

    class Requirements(BaseOnCommitTask.Requirements):
        cores = 1

    @staticmethod
    def need_to_run(pr_info):
        if not has_changes_in_code(pr_info):
            return NeedToRunDescription(False, "No diff in .cpp, .h, .py, etc...", False)

        return BaseOnCommitTask.need_to_run(pr_info)

    @staticmethod
    def get_context_name():
        return "ClickHouse build check"

    def post_statuses(self):
        return PostStatuses.ALWAYS

    @staticmethod
    def order():
        return 3

    @staticmethod
    def get_images_names():
        return ["yandex/clickhouse-deb-builder", "yandex/clickhouse-binary-builder"]

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        logging.info("Searching for CLICKHOUSE_REPO at commit %s", commit.sha)
        resources = sdk2.Resource.find(
            CLICKHOUSE_REPO,
            state=ctr.State.READY,
            attrs=dict(commit=commit.sha, pr_number=pull_request.number)
        ).order(-CLICKHOUSE_REPO.id).limit(1)
        logging.info("Search finished")
        return resources.first()

    def _get_alien_pkgs(self, setup, pull_request):
        if "alien_pkgs" in setup and setup["alien_pkgs"]:
            if pull_request.number == 0:
                return ['rpm', 'tgz']
            if 'release' in set([l.name for l in pull_request.get_labels()]):
                return ['rpm', 'tgz']
        # don't need additional builds for non release PR's and pull requests
        # except 'release'
        return []

    def _run_task(self, setup, pull_request):
        alien_packages = self._get_alien_pkgs(setup, pull_request)
        task = ClickhouseBuilder(
            self,
            description="Build {}".format(json.dumps(setup)),
            github_token_key=self.Parameters.github_token_key,
            github_repo=self.Parameters.github_repo,
            commit_sha=self.Parameters.commit_sha,
            pull_request_number=self.Parameters.pull_request_number,
            s3_access_key_id=self.Parameters.s3_access_key_id,
            s3_access_key=self.Parameters.s3_access_key,
            force_run=self.Parameters.force_run,
            compiler=setup["compiler"],
            build_type=setup["build-type"],
            package_type=setup["package-type"],
            sanitizer=setup["sanitizer"],
            bundled=setup["bundled"],
            splitted=setup["splitted"],
            tidy=setup["tidy"],
            alien_pkgs=len(alien_packages) > 0,
            with_coverage=setup["with_coverage"],
            binary_executor_release_type=self.Parameters.binary_executor_release_type,
            docker_images_with_versions=self.Parameters.docker_images_with_versions,
            path_prefix=self.get_context_name().lower().replace(' ', '_')
        )
        task.enqueue()
        return task.id, len(alien_packages)

    def _find_already_finished_task(self, build_setup):
        input_params = {k.replace('-', '_'): v for k, v in build_setup.iteritems() if v != ""}
        input_params["commit_sha"] = self.Parameters.commit_sha
        input_params["pull_request_number"] = self.Parameters.pull_request_number
        logging.info("Searching for tasks with parameters %s", json.dumps(input_params))
        try:
            tasks = sdk2.Task.find(
                ClickhouseBuilder,
                input_parameters=input_params,
                status=(ctt.Status.SUCCESS,),
                children=True,
            ).order(-ClickhouseBuilder.id).limit(len(DEFAULT_BUILDS))
        except Exception as ex:
            logging.info("Exception during tasks search %s", str(ex))
            return None

        if tasks.count == 0:
            logging.info("No tasks found")
            return None
        return tasks.first()

    def _get_build_task(self, task_id):
        tasks = sdk2.Task.find(
            ClickhouseBuilder,
            id=task_id,
            children=True,
        ).order(-ClickhouseBuilder.id).limit(len(DEFAULT_BUILDS))
        if tasks.count == 0:
            logging.info("Task with id %s not found. Looks dangerous", task_id)
            return None

        return tasks.first()

    # special builds, which came from single task (4 sanitizers, debug)
    def additional_builds(self):
        return 5

    def get_builds(self, build_setups, pull_request):
        self.Context.worker_tasks = []
        tasks_to_wait = []
        total_builds = 0
        already_finished_builds = 0
        for setup in build_setups:
            logging.info("Searching for task with setup: %s", json.dumps(setup))
            task = self._find_already_finished_task(setup)
            total_builds += 1
            if not task or self.Parameters.force_rebuild:
                task_id, additional_builds_count = self._run_task(setup, pull_request)
                tasks_to_wait.append(task_id)
                total_builds += additional_builds_count
                logging.info("Task not found, run task №%ld with that setup", task_id)
            else:
                already_finished_builds += len(task.Parameters.build_artifacts_urls)
                logging.info("Task found, number %ld", task.id)
                task_id = task.id

            self.Context.worker_tasks.append((
                task_id, setup["compiler"], setup["build-type"],
                setup["sanitizer"], setup["package-type"],
                setup["bundled"], setup["splitted"], setup["tidy"],
                setup["with_coverage"]))

        total_builds += self.additional_builds()
        if tasks_to_wait:
            logging.info("Have to wait %ld tasks", len(tasks_to_wait))
            while True:
                finished_builds = 0
                finished_tasks = 0
                unfinished_task_ids = []
                for task_id in tasks_to_wait:
                    task = self._get_build_task(task_id)
                    if task and (
                        task.status in ctt.Status.Group.SUCCEED or task.status in ctt.Status.Group.BREAK
                    ):
                        if task.Parameters.build_artifacts_urls:
                            finished_builds += len(task.Parameters.build_artifacts_urls)
                        finished_tasks += 1
                    else:
                        logging.debug('Running task {} is in status {}'.format(task_id, task.Parameters.status))
                        unfinished_task_ids.append(str(task_id) + ": '" +
                                                   (task.Parameters.status if task else "no task") + "'")

                if finished_tasks == len(tasks_to_wait):
                    return
                else:
                    msg = "{}/{} builds finished".format(already_finished_builds + finished_builds, total_builds)
                    logging.info(msg)
                    self.update_status(msg)
                    logging.info('Task ids to wait: ' + ', '.join([str(i) for i in unfinished_task_ids]))
                    time.sleep(120)

    def _process_task(self, task_info):
        task_id = task_info[0]
        build_result = BuildResult(
            compiler=task_info[1],
            build_type=task_info[2],
            sanitizer=task_info[3],
            bundled=task_info[5],
            splitted=task_info[6],
            status=sdk2.Task[task_id].Parameters.status,
            elapsed_seconds=sdk2.Task[task_id].Parameters.elapsed_seconds,
            with_coverage=task_info[8])

        tmp_log_name = os.path.join(str(self.path()) + "/build_log_{}_{}.log".format(task_id, int(time.time())))

        if sdk2.Task[task_id].Parameters.build_artifacts_urls:
            build_artifacts_urls = sdk2.Task[task_id].Parameters.build_artifacts_urls
            build_log_url = sdk2.Task[task_id].Parameters.build_log_url
            logging.info("Downloaded log resource at url %s and builds %s", build_log_url, build_artifacts_urls)
        else:
            build_artifacts_urls = {}
            build_log_url = sdk2.Task[task_id].Parameters.build_log_url
            with open(tmp_log_name, 'w') as f:
                f.write("Build failed without build log. It's inner CI problem. Maybe something wrong with docker or host. Just wait for build restart.")

        build_results = []
        build_urls = []
        build_logs_urls = []
        if not build_artifacts_urls:
            logging.info("No artifacts for task")
            build_results.append(build_result)
            build_logs_urls.append(build_log_url)
            build_urls.append("")
        else:
            for _, artifacts_list in build_artifacts_urls.items():
                logging.info("Got artifacts list %s type %s", artifacts_list, type(artifacts_list))
                if not isinstance(artifacts_list, list):
                    artifacts_list = ast.literal_eval(artifacts_list)

                build_results.append(build_result)
                build_urls.append(artifacts_list)
                build_logs_urls.append(build_log_url)

        return build_results, build_urls, build_logs_urls

    def build_config_name(self):
        return "build_config"

    def get_build_setup_config(self, commit, pull_request):
        ci_config = get_ci_config(pull_request, commit)
        if not ci_config:
            logging.info("Build config not found :(, will use default from params")
            return self.Parameters.build_setups

        logging.info("Build config found, will use it!")
        return ci_config[self.build_config_name()]

    def process(self, commit, repo, pull_request):
        logging.info("Start building clickhouse in different environments")

        build_setup = self.get_build_setup_config(commit, pull_request)
        self.get_builds(build_setup, pull_request)

        build_results = []
        build_artifacts = []
        build_logs = []

        for task_info in self.Context.worker_tasks:
            build_result, build_artifacts_url, build_logs_url = self._process_task(task_info)
            build_results += build_result
            build_artifacts += build_artifacts_url
            build_logs += build_logs_url

        branch_url = repo.html_url
        branch_name = "master"
        if pull_request.number != 0:
            branch_name = "PR #{}".format(pull_request.number)
            branch_url = pull_request.html_url
        commit_url = commit.html_url

        report = create_build_html_report(
            self.get_context_name(),
            build_results,
            build_logs,
            build_artifacts,
            self.task_url,
            branch_url,
            branch_name,
            commit_url
        )
        with open('report.html', 'w') as f:
            f.write(report)

        context_name_for_path = self.get_context_name().lower().replace(' ', '_')
        s3_path_prefix = str(pull_request.number) + "/" + self.commit.sha + "/" + context_name_for_path

        url = self.s3_client.upload_build_file_to_s3('./report.html', s3_path_prefix + "/report.html")

        total_builds = len(build_results)
        ok_builds = 0
        summary_status = "success"
        for build_result in build_results:
            if build_result.status == "failure" and summary_status != "error":
                summary_status = "failure"
            if build_result.status == "error" or not build_result.status:
                summary_status = "error"

            if build_result.status == "success":
                ok_builds += 1

        description = "{}/{} builds are OK".format(ok_builds, total_builds)

        logging.info("Result placed in %s", url)

        return summary_status, description, url
