# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.projects.clickhouse.BaseOnCommitTask.base import BaseOnCommitTask, PostStatuses
from sandbox.projects.clickhouse.resources import CLICKHOUSE_REPO, CLICKHOUSE_BUILD, CLICKHOUSE_BUILD_LXC_CONTAINER, CLICKHOUSE_BUILD_LOGS, CLICKHOUSE_BUILD_CACHE
from sandbox.sdk2.path import Path
import datetime
import logging
import os
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.misc as ctm
import sandbox.sdk2.helpers
import subprocess
import time
import traceback

from sandbox.projects.clickhouse.util.task_helper import compress_fast, decompress_fast


def get_build_parameters_as_string(compiler, build_type, sanitizer,
                                   bundled, splitted, tidy, with_coverage, pkg_type):

    return "_".join([compiler, build_type, sanitizer, bundled, splitted, tidy, str(with_coverage), pkg_type])


class BuildException(Exception):
    pass


def execute_with_retry(function, name, retry_count):
    result_ex = None
    for i in range(retry_count):
        try:
            function()
            break
        except Exception as ex:
            logging.warn("Function '%s' failed with exception %s retry %s", name, str(ex), i + 1)
            result_ex = ex
            time.sleep(i * 2 + 1)
    else:
        raise result_ex


def _get_alien_pkgs(pull_request):
    if pull_request.number == 0:
        return ['rpm', 'tgz']
    if 'release' in set([l.name for l in pull_request.get_labels()]):
        return ['rpm', 'tgz']
    # don't need additional builds for non release PR's and pull requests
    # except 'release'
    return []


def static_uploads(compiler):
    if compiler == 'clang-11-darwin' or compiler == 'clang-12-darwin' or compiler == 'clang-13-darwin':
        return 'master/macos'
    elif compiler == 'clang-11-darwin-aarch64' or compiler == 'clang-12-darwin-aarch64' or compiler == 'clang-13-darwin-aarch64':
        return 'master/macos-aarch64'
    elif compiler == 'clang-11-aarch64' or compiler == 'clang-12-aarch64' or compiler == 'clang-13-aarch64':
        return 'master/aarch64'
    elif compiler == 'clang-13-ppc64le':
        return 'master/ppc64le'
    elif compiler == 'clang-13-riscv64':
        return 'master/riscv64'
    elif compiler == 'clang-11-freebsd' or compiler == 'clang-12-freebsd' or compiler == 'clang-13-freebsd':
        return 'master/freebsd'
    elif compiler == 'clang-11' or compiler == 'clang-12' or compiler == 'clang-13':
        return 'master/amd64'
    return None


class ClickhouseBuilder(BaseOnCommitTask):
    """
    Task can build ClickHouse from resource in different environments
    """

    REL_CCACHE_DIR = "local_ccache"

    class Parameters(BaseOnCommitTask.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            resource_type=CLICKHOUSE_BUILD_LXC_CONTAINER,
        )
        kill_timeout = 3 * 60 * 60  # 3 hours
        with sdk2.parameters.RadioGroup('Package type', required=True) as package_type:
            package_type.values['binary'] = package_type.Value(value='binary', default=True)
            package_type.values['deb'] = package_type.Value(value='deb')
            package_type.values['performance'] = package_type.Value(value='performance')

        with sdk2.parameters.RadioGroup('Compiler', required=True) as compiler:
            compiler.values['gcc-9'] = compiler.Value(value='gcc-9')
            compiler.values['gcc-10'] = compiler.Value(value='gcc-10')
            compiler.values['gcc-11'] = compiler.Value(value='gcc-11')

            compiler.values['clang-13'] = compiler.Value(value='clang-13')
            compiler.values['clang-12'] = compiler.Value(value='clang-12')
            compiler.values['clang-11'] = compiler.Value(value='clang-11')
            compiler.values['clang-10'] = compiler.Value(value='clang-10', default=True)

            compiler.values['clang-10-darwin'] = compiler.Value(value='clang-10-darwin')
            compiler.values['clang-10-aarch64'] = compiler.Value(value='clang-10-aarch64')
            compiler.values['clang-10-freebsd'] = compiler.Value(value='clang-10-freebsd')

            compiler.values['clang-11-darwin'] = compiler.Value(value='clang-11-darwin')
            compiler.values['clang-11-darwin-aarch64'] = compiler.Value(value='clang-11-darwin-aarch64')
            compiler.values['clang-11-aarch64'] = compiler.Value(value='clang-11-aarch64')
            compiler.values['clang-11-freebsd'] = compiler.Value(value='clang-11-freebsd')
            compiler.values['clang-11-ppc64le'] = compiler.Value(value='clang-11-ppc64le')

            compiler.values['clang-12-darwin'] = compiler.Value(value='clang-12-darwin')
            compiler.values['clang-12-darwin-aarch64'] = compiler.Value(value='clang-12-darwin-aarch64')
            compiler.values['clang-12-aarch64'] = compiler.Value(value='clang-12-aarch64')
            compiler.values['clang-12-freebsd'] = compiler.Value(value='clang-12-freebsd')
            compiler.values['clang-12-ppc64le'] = compiler.Value(value='clang-12-ppc64le')

            compiler.values['clang-13-darwin'] = compiler.Value(value='clang-13-darwin')
            compiler.values['clang-13-darwin-aarch64'] = compiler.Value(value='clang-13-darwin-aarch64')
            compiler.values['clang-13-aarch64'] = compiler.Value(value='clang-13-aarch64')
            compiler.values['clang-13-freebsd'] = compiler.Value(value='clang-13-freebsd')
            compiler.values['clang-13-ppc64le'] = compiler.Value(value='clang-13-ppc64le')
            compiler.values['clang-13-riscv64'] = compiler.Value(value='clang-13-riscv64')

        with sdk2.parameters.RadioGroup('Build type') as build_type:
            build_type.values[''] = build_type.Value(value='relwithdebuginfo', default=True)
            build_type.values['debug'] = build_type.Value(value='debug')

        with sdk2.parameters.RadioGroup('Sanitizer') as sanitizer:
            sanitizer.values[''] = sanitizer.Value(value='none', default=True)
            sanitizer.values['address'] = sanitizer.Value(value='address')
            sanitizer.values['thread'] = sanitizer.Value(value='thread')
            sanitizer.values['memory'] = sanitizer.Value(value='memory')
            sanitizer.values['undefined'] = sanitizer.Value(value='undefined')

        with sdk2.parameters.RadioGroup('Bundled') as bundled:
            bundled.values['bundled'] = bundled.Value(value='bundled', default=True)
            bundled.values['unbundled'] = bundled.Value(value='unbundled')

        with sdk2.parameters.RadioGroup('Splitted') as splitted:
            splitted.values['unsplitted'] = splitted.Value(value='unsplitted', default=True)
            splitted.values['splitted'] = splitted.Value(value='splitted')

        with sdk2.parameters.RadioGroup('Clang-Tidy') as tidy:
            tidy.values['disable'] = tidy.Value(value='disable', default=True)
            tidy.values['enable'] = tidy.Value(value='enable')

        with sdk2.parameters.Output:
            build_resource_ids = sdk2.parameters.List(
                "Directories with build artifacts",
                value_type=sdk2.parameters.Integer,
                default=[]
            )
            build_log_resource_id = sdk2.parameters.Integer(
                "Text file with build log"
            )
            status = sdk2.parameters.String(
                "Text status",
                required=True,
            )
            elapsed_seconds = sdk2.parameters.Integer(
                "Elapsed time required to build resource",
                required=True,
                default=0,
            )
            build_artifacts_urls = sdk2.parameters.Dict(
                "Files with build artifacts",
                default={}
            )
            build_log_url = sdk2.parameters.String(
                "Text file with build log"
            )

        alien_pkgs = sdk2.parameters.Bool("Additional packages", default=False)
        with_coverage = sdk2.parameters.Bool("With coverage", default=False)
        path_prefix = sdk2.parameters.String("Prefix of path in S3", default="clickhouse_build_check")

    class Requirements(BaseOnCommitTask.Requirements):
        # Don't require a particular CPU, but require at least 32 cores so that
        # it's not too slow.
        client_tags = ctc.Tag.GENERIC
        cores = 32

        privileged = True

        # We need about 150G to enable ccache debug. Without it, 80GB should
        # suffice. The RAM drive is not reserved, so we can always request a large
        # one. Change the RAM requirement in the next line instead.
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 150 * 1024, None)
        # How much RAM we need: the RAM drive for build and at least a couple GB
        # per each of 32 (probably) compiler processes.
        ram = (80 + 2 * 32) * 1024

        # Need a couple of GB to pack ccache and build logs, everything else is
        # on the RAM disk. Reducing disk requirement from the default 30G is
        # beneficial because many hosts are often out of disk space but otherwise
        # free.
        disk_space = 5 * 1024

    def on_create(self):
        self.Parameters._container = sdk2.Resource.find(
            CLICKHOUSE_BUILD_LXC_CONTAINER,
            state=ctr.State.READY,
            attrs=dict(released="stable")
        ).order(-CLICKHOUSE_BUILD_LXC_CONTAINER.id).first().id

        # hack for Alexey PR (TODO Remove)
        if self.Parameters.pull_request_number == 20001:
            logging.info("Running for special PR")
            self.Requirements.cores = 40
            self.Requirements.client_tags = ctc.Tag.GENERIC
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 200 * 1024, None)
            self.Parameters.kill_timeout = 3 * 60 * 60

    @staticmethod
    def need_docker():
        return True

    @staticmethod
    def get_context_name():
        return "Build clickhouse binary"

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        logging.info("Searching for CLICKHOUSE_REPO at commit %s", commit.sha)
        resources = sdk2.Resource.find(
            CLICKHOUSE_REPO,
            state=ctr.State.READY,
            attrs=dict(commit=commit.sha, pr_number=pull_request.number)
        ).order(-CLICKHOUSE_REPO.id).limit(1)
        logging.info("Search finished")
        return resources.first()

    def post_statuses(self):
        return PostStatuses.NEVER

    def _save_repo(self, commit, repo, pull_request):
        repo_resource = ClickhouseBuilder.get_resources(commit, repo, pull_request)
        self.version = repo_resource.version
        self.revision = repo_resource.revision
        logging.info("Downloading resource: {}".format(repo_resource))
        repo_data = sdk2.ResourceData(repo_resource)
        repo_path = str(repo_data.path)
        logging.info("Download finished, archive path %s", repo_path)

        logging.info("Is file exists:" + str(os.path.exists(repo_path)))
        logging.info("Unpacking dowloaded repo fast")
        decompress_fast(repo_path, self.ramdrive.path)

        logging.info("Unpack finished")
        return os.path.join(str(self.ramdrive.path), "ClickHouse")

    def _can_build_with(self, path, opt):
        with open(path, 'r') as script:
            for line in script:
                if opt in line:
                    return True
        return False

    def _find_cache_resource(self, pr_number, pr_base_branch):
        # Look for some variants of suitable ccache resource, starting with the
        # best one -- a resource for the same PR. Note that we don't check for
        # the base branch in this variant, because it doesn't change, and because
        # it's not yet set for most resources when we are introducing this feature.
        attr_variants = [{
            "compiler": self.Parameters.compiler,
            "build_type": self.Parameters.build_type if self.Parameters.build_type else "relwithdebuginfo",
            "sanitizer": self.Parameters.sanitizer if self.Parameters.sanitizer else "none",
            # Different packages have subtly different compilation flags, e.g.
            # "binary" builds unit tests and can't use ThinLTO, but "performance"
            # doesn't build tests and uses ThinLTO. This means we can't mix ccache
            # between them. Just distinguish on package type, although it might
            # be somewhat suboptimal, e.g. a "deb" ccache would probably be OK
            # for "performance" build.
            "package_type": self.Parameters.package_type,
            "bundled": self.Parameters.bundled,
            "splitted": self.Parameters.splitted,
            "tidy": self.Parameters.tidy,
            "with_coverage": self.Parameters.with_coverage,
            "pr_number": pr_number,
        }]

        # Next, any PR with same base branch.
        attr_variants.append(attr_variants[-1].copy())
        attr_variants[-1].pop("pr_number")
        attr_variants[-1]["pr_base_branch"] = pr_base_branch

        # Next, at least something with same compiler options.
        attr_variants.append(attr_variants[-1].copy())
        attr_variants[-1].pop("pr_base_branch")

        res = None
        for attrs in attr_variants:
            res = sdk2.Resource.find(
                CLICKHOUSE_BUILD_CACHE,
                state=ctr.State.READY,
                attrs=attrs).order(-CLICKHOUSE_BUILD_CACHE.id).limit(1).first()
            if res:
                logging.info("Found ccache resource id %s from PR %s @ %s by attrs filter %s", res.id, res.pr_number, res.commit, attrs)
                return res
        else:
            logging.info("Not found ccache resource for attribute variants %s", attr_variants)
            return None

    def _get_cache_path(self, commit, pull_request):
        cache_path = os.path.join(str(self.ramdrive.path), self.REL_CCACHE_DIR)
        pr_number = pull_request.number
        pr_base_branch = pull_request.raw_data['base']['ref'] if pr_number != 0 else 'master'
        result = self._find_cache_resource(pr_number, pr_base_branch)
        if not result:
            logging.info("No cache found at all")
            os.mkdir(cache_path)
            return cache_path

        logging.info("Extracting cache data")
        cache_data = sdk2.ResourceData(result)
        decompress_fast(cache_data.path, str(self.ramdrive.path))
        logging.info("Finished cache data extraction to %s", str(self.ramdrive.path))
        return cache_path

    def _pack_ccache(self, ccache_path, commit, pull_request):
        logging.info("Packing ccache from path %s", ccache_path)
        compress_fast(ccache_path, "clickhouse_ccache.tar.gz")
        ccache_res = CLICKHOUSE_BUILD_CACHE(
            self,
            "Clickhouse build cache with commit {} ".format(commit.sha),
            "./clickhouse_local_ccache.tar.gz",
            commit=commit.sha,
            date=datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            pr_number=pull_request.number,
            pr_base_branch=pull_request.raw_data['base']['ref'] if pull_request.number != 0 else 'master',
            compiler=self.Parameters.compiler,
            build_type=self.Parameters.build_type if self.Parameters.build_type else "relwithdebuginfo",
            sanitizer=self.Parameters.sanitizer if self.Parameters.sanitizer else "none",
            package_type=self.Parameters.package_type,
            bundled=self.Parameters.bundled,
            splitted=self.Parameters.splitted,
            tidy=self.Parameters.tidy,
            with_coverage=self.Parameters.with_coverage,
        )
        ccache_data = sdk2.ResourceData(ccache_res)

        def cache_resource_upload():
            ccache_data.path.write_bytes(Path("clickhouse_ccache.tar.gz").read_bytes())
            ccache_data.ready()
        execute_with_retry(cache_resource_upload, "Upload cache resource", 5)
        logging.info("Write finished")

    def _can_export_binaries(self):
        if self.Parameters.package_type != 'deb':
            return False
        if self.Parameters.bundled and self.Parameters.bundled != "bundled":
            return False
        if self.Parameters.splitted and self.Parameters.splitted == 'splitted':
            return False
        if self.Parameters.sanitizer and self.Parameters.sanitizer != 'none':
            return True
        if self.Parameters.build_type and self.Parameters.build_type != 'relwithdebuginfo':
            return True
        return False

    def get_build_image_and_version(self):
        if self.Parameters.package_type != 'deb':
            if 'clickhouse/binary-builder' in self.Parameters.docker_images_with_versions:
                return 'clickhouse/binary-builder', self.Parameters.docker_images_with_versions['clickhouse/binary-builder']
            return 'yandex/clickhouse-binary-builder', self.Parameters.docker_images_with_versions['yandex/clickhouse-binary-builder']
        else:
            if 'clickhouse/deb-builder' in self.Parameters.docker_images_with_versions:
                return 'clickhouse/deb-builder', self.Parameters.docker_images_with_versions['clickhouse/deb-builder']
            return 'yandex/clickhouse-deb-builder', self.Parameters.docker_images_with_versions['yandex/clickhouse-deb-builder']

    def _build(self, repo, repo_path, pull_request, ccache_path):
        my_env = os.environ.copy()
        logging.info("Start build")
        log_name = "build_" + str(int(time.time()))
        packager_path = os.path.join(repo_path, "docker/packager")
        output_path = os.path.join(repo_path, "build_output_folder")
        logging.info("Will place output to path %s", output_path)
        if not os.path.exists(output_path):
            os.makedirs(output_path, mode=0o0777)
        cmd = "cd {ppath} && ./packager --output-dir={odir} --package-type={package_type} --compiler={comp}".format(
            ppath=packager_path,
            odir=output_path,
            package_type=self.Parameters.package_type,
            comp=self.Parameters.compiler
        )
        if self.Parameters.build_type:
            cmd += ' --build-type={}'.format(self.Parameters.build_type)
        if self.Parameters.sanitizer:
            cmd += ' --sanitizer={}'.format(self.Parameters.sanitizer)

        if self.Parameters.splitted == "splitted":
            if self._can_build_with(os.path.join(packager_path, "packager"), '--split-binary'):
                cmd += ' --split-binary'
            else:
                logging.info("Can't build split-binary :(, not ready for this")

        if self.Parameters.tidy == "enable":
            cmd += ' --clang-tidy'

        if self._can_build_with(os.path.join(packager_path, "packager"), '--version'):
            logging.info("Will use version as argument")
            cmd += ' --version={}'.format(self.version)
        else:
            logging.info("Can't use version argument :(")

        self.can_build_with_alien = self._can_build_with(os.path.join(packager_path, "packager"), '--alien-pkgs')
        if self.can_build_with_alien and self.Parameters.alien_pkgs:
            cmd += ' --alien-pkgs ' + ' '.join(_get_alien_pkgs(pull_request))

        self.can_build_with_coverage = self._can_build_with(os.path.join(packager_path, "packager"), '--with-coverage')
        if self.can_build_with_coverage and self.Parameters.with_coverage:
            cmd += ' --with-coverage'

        self.can_use_docker_images_with_versions = self._can_build_with(os.path.join(packager_path, "packager"), '--docker-image-version')
        if self.can_use_docker_images_with_versions:
            build_image, version = self.get_build_image_and_version()
            image_with_version = build_image + ':' + version
            for i in range(10):
                try:
                    logging.info("Pulling image %s", image_with_version)
                    output = subprocess.check_output("docker pull {}".format(image_with_version), shell=True)
                    logging.info("Command finished, output %s", output)
                    break
                except Exception as ex:
                    logging.info("Exception updating image %s: %s", image_with_version, str(ex))
                    time.sleep(i * 5)
                    exception = ex
            else:
                raise exception

            logging.info("Will build with docker image version %s:%s", build_image, version)
            cmd += ' --docker-image-version=' + version

        logging.info("Hooray, can use ccache for distributed build")
        cmd += ' --cache=ccache'
        cmd += ' --ccache_dir={}'.format(ccache_path)

        if self._can_export_binaries():
            logging.info("Will export unit tests and binaries from deb build")
            cmd += ' --with-binaries=tests'

        official_build = pull_request.number == 0 or 'release' in [l.name for l in pull_request.get_labels()]
        if self._can_build_with(os.path.join(packager_path, "packager"), '--official') and official_build:
            cmd += ' --official'
            logging.info("Will build official because we on master and packager can do it")
        else:
            logging.info("Not official build, pr number %s", pull_request.number)

        logging.info("Will build ClickHouse with cmd '%s'", cmd)
        with sandbox.sdk2.helpers.ProcessLog(self, logger=log_name) as pl:
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout, env=my_env).wait()
            logging.info("Build finished with exit code {}".format(retcode))

            subprocess.Popen('lsblk', shell=True).wait()
            subprocess.Popen('df -h', shell=True).wait()
            # This one can take tens of minutes.
            # subprocess.Popen('du --max-depth=1 /', shell=True).wait()

            description = '<a href="https://proxy.sandbox.yandex-team.ru/task/{}/log1/{}.out.log">Build log</a>\n'.format(self.id, log_name)
            if pull_request.number:
                description = '<a href="https://github.com/ClickHouse/ClickHouse/pull/{}">#{}</a>\n'.format(pull_request.number, pull_request.number)
            description += '{}, {}, {}, {}, '\
                           'Sanitizer: {}, Coverage: {}, Clang-Tidy: {}'.format(self.Parameters.package_type, self.Parameters.compiler, self.Parameters.bundled,
                                                                                self.Parameters.splitted, self.Parameters.sanitizer,
                                                                                self.Parameters.with_coverage, self.Parameters.tidy)
            self.Parameters.description = description
            self.build_log_path = str(pl.stdout.path)
            if retcode == 0:
                logging.info("Build successfully")
            else:
                raise BuildException("Build failed")

        return output_path

    def _upload_log(self, build_log_path, commit, pull_request):
        s3_path_prefix = str(pull_request.number) + "/" + commit.sha + "/" + self.Parameters.path_prefix

        logging.info("Creating resource with build log from %s", build_log_path)
        log_resource = CLICKHOUSE_BUILD_LOGS(
            self,
            "Task logs for clickhouse build with commit {}".format(commit.sha),
            "build_logs_" + str(int(time.time())) + ".txt",
            date=datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            compiler=self.Parameters.compiler,
            build_type=self.Parameters.build_type,
            sanitizer=self.Parameters.sanitizer,
            bundled=self.Parameters.bundled,
            splitted=self.Parameters.splitted,
            tidy=self.Parameters.tidy,
            pr_number=pull_request.number,
            package_type=self.Parameters.package_type,
            version=self.version,
            revision=self.revision,
            with_coverage=self.Parameters.with_coverage
        )
        log_data = sdk2.ResourceData(log_resource)

        def log_resource_upload():
            log_data.path.write_bytes(Path(build_log_path).read_bytes())
            log_data.path.chmod(0o0777)
            log_data.ready()

        execute_with_retry(log_resource_upload, "Upload log resource", 5)
        logging.info("Log resource uploaded")

        return self.s3_client.upload_build_file_to_s3(build_log_path, s3_path_prefix + "/" + os.path.basename(build_log_path))

    def move_alien_packages(self, build_result_path, alien_packages):
        result = [(build_result_path, self.Parameters.package_type)]
        for alien_pkg in alien_packages:
            package_path = os.path.join(str(self.path()), alien_pkg + "_directory")
            os.mkdir(package_path)
            try:
                subprocess.check_call("cd {} && mv *.{} {}".format(build_result_path, alien_pkg, package_path), shell=True)
            except Exception as ex:
                raise Exception("Cannot move result for {} to {}".format(alien_pkg, package_path) + str(ex))
            result.append((package_path, alien_pkg))
        return result

    def move_binary_results(self, build_result_paths, build_result_path):
        package_path = os.path.join(str(self.path()), "bin_directory")
        os.mkdir(package_path)
        try:
            subprocess.check_call("cd {} && mv binary/* {} && rmdir binary".format(build_result_path, package_path), shell=True)
        except Exception as ex:
            raise Exception("Cannot move result to {}".format(package_path) + str(ex))
        return (package_path, "binary")

    def process(self, commit, repo, pull_request):
        logging.info("Start fetching resource with repo")
        repo_path = self._save_repo(commit, repo, pull_request)

        build_urls = {}
        ccache_path = self._get_cache_path(commit, pull_request)
        try:
            try:
                start = time.time()
                build_path = self._build(repo, repo_path, pull_request, ccache_path)
                self.Parameters.elapsed_seconds = int(time.time() - start)
                alien_pkgs = []
                if self.can_build_with_alien and self.Parameters.alien_pkgs:
                    alien_pkgs = _get_alien_pkgs(pull_request)
                build_result_paths = self.move_alien_packages(build_path, alien_pkgs)
                if self._can_export_binaries():
                    build_result_paths.append(self.move_binary_results(build_result_paths, build_path))

                log_url = self._upload_log(self.build_log_path, commit, pull_request)
            except BuildException as ex:
                log_url = self._upload_log(self.build_log_path, commit, pull_request)
                logging.info("Build failed")
                raise ex
            logging.info("Log url %s", log_url)

            self.Parameters.build_log_url = log_url
            logging.info("Creating resource from %s", ' '.join([path[0] for path in build_result_paths]))
            build_type = self.Parameters.build_type
            if not build_type:
                build_type = "relwithdebuginfo"

            sanitizer = self.Parameters.sanitizer
            if not sanitizer:
                sanitizer = "none"

            logging.info("Non splitted mode, will upload build resource")
            for build_result_path, pkg_type in build_result_paths:
                logging.info("Uploading %s of type %s", build_result_path, pkg_type)

                # Publish the package for performance tests at a predictable URL.
                if pkg_type == "performance":
                    prefix = "./performance"
                else:
                    prefix = "./" + get_build_parameters_as_string(self.Parameters.compiler, build_type, sanitizer, self.Parameters.bundled,
                                                                   self.Parameters.splitted, self.Parameters.tidy, self.Parameters.with_coverage,
                                                                   pkg_type)
                resource = CLICKHOUSE_BUILD(
                    self,
                    "ClickHouse build with commit {}".format(commit.sha),
                    prefix,
                    commit=commit.sha,
                    date=datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                    compiler=self.Parameters.compiler,
                    build_type=build_type,
                    sanitizer=sanitizer,
                    bundled=self.Parameters.bundled,
                    pr_number=pull_request.number,
                    splitted=self.Parameters.splitted,
                    tidy=self.Parameters.tidy,
                    package_type=pkg_type,
                    version=self.version,
                    revision=self.revision,
                    with_coverage=self.Parameters.with_coverage,
                    elapsed_seconds=self.Parameters.elapsed_seconds
                )
                resource_data = sdk2.ResourceData(resource)
                files_in_result = os.listdir(build_result_path)
                logging.debug("Files in {}: {}".format(build_result_path, files_in_result))
                resource_data.path.mkdir(parents=True, exist_ok=True)
                for build_artifact in files_in_result:
                    full_path = os.path.join(build_result_path, build_artifact)
                    if not os.path.islink(full_path):
                        artifact_path = resource_data.path.joinpath(build_artifact)
                        logging.info("Writing artifact %s into %s", str(full_path), str(artifact_path))

                        def write_bytes():
                            artifact_path.write_bytes(Path(full_path).read_bytes())

                        execute_with_retry(write_bytes, "Write bytes for {}".format(artifact_path), 5)
                        logging.info("Artifact successfuly written")
                    else:
                        logging.info("Path %s is symlink, will not add it to result image", full_path)
                resource_data.path.chmod(0o0777)

                def ready_resource():
                    resource_data.ready()

                execute_with_retry(ready_resource, "Mark resource ready", 5)

                logging.info("Resource uploaded")

                s3_path_prefix = str(pull_request.number) + "/" + self.commit.sha + "/" + self.Parameters.path_prefix + "/" + os.path.basename(prefix)

                urls = self.s3_client.upload_build_folder_to_s3(build_result_path, s3_path_prefix, keep_dirs_in_s3_path=False, upload_symlinks=False)
                build_urls[pkg_type] = urls
                if (pull_request.number == 0 and build_type == "relwithdebuginfo" and sanitizer == "none" and
                        self.Parameters.bundled == "bundled" and self.Parameters.splitted == "unsplitted" and pkg_type == "binary"):
                    static_path = static_uploads(self.Parameters.compiler)
                    logging.info("Result path %s, directories %s", build_result_path, os.listdir(build_result_path))
                    if static_path and os.path.exists(os.path.join(build_result_path, 'clickhouse')):
                        logging.info("Static path found: %s", static_path)
                        static_upload_paths = self.s3_client.upload_build_folder_to_s3(build_result_path, static_path, keep_dirs_in_s3_path=False, upload_symlinks=False)
                        logging.info("Uploaded to: %s", static_upload_paths)
                    else:
                        logging.info("Static path not found")

            self.Parameters.build_artifacts_urls = build_urls
            self.Parameters.status = "success"

            logging.info("Got urls %s log url %s", build_urls, log_url)

            try:
                self._pack_ccache(ccache_path, commit, pull_request)
            except Exception as ex:
                logging.info("Cannot pack ccache %s", str(ex))

            return "success", "Build finished", None
        except BuildException as build_ex:
            logging.info("Build exception '%s'\n%s", str(build_ex), traceback.format_exc())
            self.Parameters.build_artifacts_urls = {}
            self.Parameters.build_log_url = log_url
            self.Parameters.status = "failure"
            return "failure", "Build failed, see logs in task", None
        except Exception as ex:
            logging.info("Task exception '%s'\n%s", str(ex), traceback.format_exc())
            if build_urls.keys() and not self.Parameters.build_artifacts_urls:
                self.Parameters.build_artifacts_urls = build_urls
            else:
                self.Parameters.build_artifacts_urls = {}

            self.Parameters.build_log_url = log_url
            self.Parameters.status = "error"
            return "error", "Task exception during build, see logs in task", None
