# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.projects.clickhouse.BaseOnCommitTask.base import PostStatuses
from sandbox.projects.clickhouse.BaseOnCommitTask.test_task import BaseOnCommitTestTask
from sandbox.projects.clickhouse.util.resource_helper import ResourceHelper
from sandbox.projects.clickhouse.resources import CLICKHOUSE_BUILD_LXC_CONTAINER
import subprocess
import requests
import sandbox.common.types.resource as ctr
import logging
import os
import time
import sandbox

from sandbox.projects.clickhouse.util.docker_image_helper import DockerImageHelper


VERY_OLD_VERSION_PREFIX = '1.1.54'
RETRY_COUNT = 10


def is_ok_version(version):
    return VERY_OLD_VERSION_PREFIX not in version and version != '1' and version != '1.1' and 'alpine' not in version


def get_version_components(version):
    current_array = version.split('.')
    if len(current_array) == 4:
        cur_major, cur_minor, cur_patch, cur_tweak = current_array
        cur_patch += '.' + cur_tweak
    elif len(current_array) == 3:
        cur_major, cur_minor, cur_patch = current_array
    elif len(current_array) == 2:
        cur_major, cur_minor = current_array
        cur_patch = None
    elif len(current_array) == 1:
        cur_major = current_array[0]
        cur_minor = None
        cur_patch = None
    else:
        raise Exception("Can't release version {}, not ready for this".format(version))

    return cur_major, cur_minor, cur_patch


class ClickhouseDockerHubPush(BaseOnCommitTestTask):

    class Parameters(BaseOnCommitTestTask.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            resource_type=CLICKHOUSE_BUILD_LXC_CONTAINER,
        )
        dockerfiles = sdk2.parameters.Dict("Relative paths to Dockerfiles and image names")
        dockerfiles_from_repo = sdk2.parameters.String("Relative path to json with dockerfiles mapping", default="docker/images.json")
        dockerhub_password = sdk2.parameters.String("Dockerhub password", default="clickhouse-robot-dockerhub-password")
        with_version = sdk2.parameters.Bool("User version from downloaded resource", default=False)
        version_from_pr = sdk2.parameters.Bool("Use PR number as version", default=True)
        post_statuses = sdk2.parameters.Bool("Post status to commit", default=True)

    class Requirements(BaseOnCommitTestTask.Requirements):
        privileged = True

    def on_create(self):
        self.Parameters._container = sdk2.Resource.find(
            CLICKHOUSE_BUILD_LXC_CONTAINER,
            state=ctr.State.READY,
            attrs=dict(released="stable")
        ).order(-CLICKHOUSE_BUILD_LXC_CONTAINER.id).first().id

    def post_statuses(self):
        if self.Parameters.post_statuses:
            return PostStatuses.ALWAYS
        else:
            return PostStatuses.NEVER

    @staticmethod
    def order():
        return 2

    @staticmethod
    def get_context_name():
        return "Push to dockerhub"

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        helper = ResourceHelper(commit, repo, pull_request)
        return helper.get_repo_no_submodules_resource()

    @staticmethod
    def require_internet():
        return True

    @staticmethod
    def need_docker():
        return True

    def build_and_push_alpine_package(self, path_to_dockerfile_folder, image_name, version_string, version_postfix, full_version):
        logging.info("Building alpine docker image %s-alpine with version %s from path %s", image_name, version_string, path_to_dockerfile_folder)
        build_log = None
        push_log = None
        with sandbox.sdk2.helpers.ProcessLog(self, logger="dockerbuild") as pl:
            cmd = "cd {path} && VERSION={ver} DOCKER_IMAGE='{im}' ./alpine-build.sh".format(im=image_name, ver=full_version, path=path_to_dockerfile_folder)
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            build_log = str(pl.stdout.path)
            if retcode != 0:
                return False, build_log, None

        with sandbox.sdk2.helpers.ProcessLog(self, logger="dockertag") as pl:
            cmd = "docker tag {im}:{full_ver}{postfix} {im}:{ver}{postfix}".format(im=image_name, full_ver=full_version, ver=version_string, postfix=version_postfix)
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            build_log = str(pl.stdout.path)
            if retcode != 0:
                return False, build_log, None

        result_version = version_string + version_postfix
        logging.info("Pushing image %s-alpine to dockerhub with version %s", image_name, result_version)

        with sandbox.sdk2.helpers.ProcessLog(self, logger="dockerpush") as pl:
            cmd = "docker push {im}:{ver}".format(im=image_name, ver=result_version)
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            push_log = str(pl.stdout.path)
            if retcode != 0:
                return False, build_log, push_log

        logging.info("Processing of %s successfully finished", image_name)
        return True, build_log, push_log

    def build_and_push_one_package(self, path_to_dockerfile_folder, image_name, version_string):
        logging.info("Building docker image %s with version %s from path %s", image_name, version_string, path_to_dockerfile_folder)
        build_log = None
        push_log = None
        with sandbox.sdk2.helpers.ProcessLog(self, logger="dockerbuild") as pl:
            cmd = "docker build --network=host -t {im}:{ver} {path}".format(im=image_name, ver=version_string, path=path_to_dockerfile_folder)
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            build_log = str(pl.stdout.path)
            if retcode != 0:
                return False, build_log, None

        with sandbox.sdk2.helpers.ProcessLog(self, logger="taglatest") as pl:
            cmd = "docker build --network=host -t {im} {path}".format(im=image_name, path=path_to_dockerfile_folder)
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            build_log = str(pl.stdout.path)
            if retcode != 0:
                return False, build_log, None

        logging.info("Pushing image %s to dockerhub", image_name)

        with sandbox.sdk2.helpers.ProcessLog(self, logger="dockerpush") as pl:
            cmd = "docker push {im}:{ver}".format(im=image_name, ver=version_string)
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            push_log = str(pl.stdout.path)
            if retcode != 0:
                return False, build_log, push_log

        logging.info("Processing of %s successfully finished", image_name)
        return True, build_log, push_log

    # return versions structure in format
    # major => minor1 => patch1
    #                 => patch2
    #                 => patch3
    #       => minor2
    #                 => major.minor2.patch1
    #       => minor3
    def _prepare_versions_structure(self, all_versions):
        result = {}
        for ver in all_versions:
            major, minor, patch = get_version_components(ver)
            if not minor or not patch:
                continue
            major = float(major)
            minor = float(minor)
            patch = float(patch)
            if major not in result:
                result[major] = {}

            if minor not in result[major]:
                result[major][minor] = []

            result[major][minor].append(patch)

        return result

    def get_versions_for_image(self, current_version, image_name):
        response = requests.get("https://registry.hub.docker.com/v1/repositories/{}/tags".format(image_name))
        try:
            response.raise_for_status()
            all_versions = set([v['name'] for v in response.json() if is_ok_version(v['name'])])
            versions_structure = self._prepare_versions_structure(all_versions)
        except Exception as ex:
            logging.info("Versions not found %s", str(ex))
            versions_structure = {}

        if versions_structure:
            max_major = max(versions_structure.keys())
        else:
            max_major = 0

        major_str, minor_str, patch_str = get_version_components(current_version)
        if minor_str is None or patch_str is None:
            raise Exception("Cannot work with version {}, no patch or minor component".format(current_version))

        patch_without_tweak_str = ''
        if '.' in patch_str:
            patch_without_tweak_str = patch_str.split('.')[0]

        major = float(major_str)
        minor = float(minor_str)
        patch = float(patch_str)
        result = []
        # ugly
        if major in versions_structure:
            max_minor = max(versions_structure[major].keys())
        else:
            max_minor = 0

        if major in versions_structure and minor in versions_structure[major]:
            max_patch = max(versions_structure[major][minor])
        else:
            max_patch = 0

        if major > max_major or (major == max_major and minor > max_minor) or (major == max_major and minor == max_minor and patch >= max_patch):
            result.append('latest')
            result.append(major_str)
            result.append(major_str + '.' + minor_str)
            if patch_without_tweak_str:
                result.append(major_str + '.' + minor_str + '.' + patch_without_tweak_str)
            result.append(major_str + '.' + minor_str + '.' + patch_str)
        elif minor > max_minor or (minor == max_minor and patch >= max_patch):
            result.append(major_str)
            result.append(major_str + '.' + minor_str)
            if patch_without_tweak_str:
                result.append(major_str + '.' + minor_str + '.' + patch_without_tweak_str)
            result.append(major_str + '.' + minor_str + '.' + patch_str)
        elif patch >= max_patch:
            result.append(major_str + '.' + minor_str)
            if patch_without_tweak_str:
                result.append(major_str + '.' + minor_str + '.' + patch_without_tweak_str)
            result.append(major_str + '.' + minor_str + '.' + patch_str)
        else:
            if patch_without_tweak_str:
                result.append(major_str + '.' + minor_str + '.' + patch_without_tweak_str)
            result.append(major_str + '.' + minor_str + '.' + patch_str)

        return result

    def process_single_image(self, version, path_to_dockerfile_folder, image_name, version_from_pr):
        logging.info("Will process image %s with version %s from path %s", image_name, version, path_to_dockerfile_folder)
        if not version_from_pr:
            all_for_this_image = self.get_versions_for_image(version, image_name)
        else:
            all_for_this_image = version
        logging.info("Image will be pushed with versions %s", ', '.join(all_for_this_image))
        result = []
        for ver in all_for_this_image:
            for i in range(RETRY_COUNT):
                success, build_log, push_log = self.build_and_push_one_package(path_to_dockerfile_folder, image_name, ver)
                if success:
                    result.append((image_name + ":" + ver, build_log, push_log, 'OK'))
                    break
                logging.info("Got error will retry %s time and sleep for %s seconds", i, i * 5)
                time.sleep(i * 5)
            else:
                result.append((image_name + ":" + ver, build_log, push_log, 'FAIL'))

        logging.info("Processing finished")
        return result

    def process_alpine_image(self, version, path_to_dockerfile_folder, image_name):
        logging.info("Will process alpine image %s with version %s from path %s", image_name, version, path_to_dockerfile_folder)
        all_for_this_image = self.get_versions_for_image(version, image_name)
        result = []
        for ver in all_for_this_image:
            for i in range(RETRY_COUNT):
                success, build_log, push_log = self.build_and_push_alpine_package(os.path.dirname(path_to_dockerfile_folder), image_name, ver, '-alpine', version)
                if success:
                    result.append((image_name + ":" + ver + '-alpine', build_log, push_log, 'OK'))
                    break
                logging.info("Got error will retry %s time and sleep for %s seconds", i, i * 5)
                time.sleep(i * 5)
            else:
                result.append((image_name + ":" + ver, build_log, push_log, 'FAIL'))
        logging.info("Processing finished")
        return result

    def _get_images(self, repo_path, commit, pull_request):
        if self.Parameters.dockerfiles:
            logging.info("Running task for concrete images, will rebuild them unconditionally")
            return self.Parameters.dockerfiles.items()

        if self.Parameters.dockerfiles_from_repo:
            return self.docker_image_helper.get_changed_docker_images(commit, pull_request, repo_path, self.Parameters.dockerfiles_from_repo)[0]

        return []

    def _process_test_results(self, test_results, s3_path_prefix):
        overall_status = 'success'
        processed_test_results = []
        for image, build_log, push_log, status in test_results:
            if status != 'OK':
                overall_status = 'failure'
            url_part = ''
            if build_log is not None and os.path.exists(build_log):
                build_url = self.s3_client.upload_test_report_to_s3(
                    build_log,
                    s3_path_prefix + "/" + os.path.basename(build_log))
                url_part += '<a href="{}">build_log</a>'.format(build_url)
            if push_log is not None and os.path.exists(push_log):
                push_url = self.s3_client.upload_test_report_to_s3(
                    push_log,
                    s3_path_prefix + "/" + os.path.basename(push_log))
                if url_part:
                    url_part += ', '
                url_part += '<a href="{}">push_log</a>'.format(push_url)
            if url_part:
                test_name = image + ' (' + url_part + ')'
            else:
                test_name = image
            processed_test_results.append((test_name, status))
        return overall_status, processed_test_results

    def _get_additional_images(self, image_name):
        if image_name.startswith('yandex/') and not self.Parameters.version_from_pr:
            return [image_name, image_name.replace('yandex/', 'clickhouse/')]

        return [image_name]

    def run(self, commit, repo, pull_request):
        dh_pass = sdk2.Vault.data(self.Parameters.dockerhub_password)
        repo_resource = self.get_resources(commit, repo, pull_request)
        repo_path = self.resource_helper.save_any_repo_resource(repo_resource)
        self.docker_image_helper = DockerImageHelper(self.gh_helper)
        if self.Parameters.with_version:
            version = repo_resource.version
        elif self.Parameters.version_from_pr:
            pr_commit_version = str(pull_request.number) + '-' + commit.sha
            if pull_request.number == 0:
                version = ['latest', '0', pr_commit_version]
            else:
                version = [str(pull_request.number), pr_commit_version]
                if pull_request.number == 28656:
                    version = ['latest'] + version
                logging.info("Will use version %s for all images", version)
        else:
            version = 'latest'
        logging.info("Logging with docker")
        for i in range(10):
            try:
                subprocess.check_output("docker login --username 'robotclickhouse' --password '{}'".format(dh_pass), shell=True)
                break
            except Exception as ex:
                logging.info("Login to dockerhub failed %s", str(ex))
                time.sleep(i * 3)
        else:
            raise Exception("Cannot login to dockerhub")

        logging.info("Start pushing images")
        images = self._get_images(repo_path, commit, pull_request)
        if not images:
            logging.info("No images found")
        else:
            logging.info("Found images %s", str(images))

        images_processing_result = []
        # The order is important, because the dependent images go later than base
        # images, so they will use the updated versions of base images.
        for rel_path, image_name in images:
            full_path = os.path.join(repo_path, rel_path)
            if os.path.exists(full_path):
                if os.path.isdir(full_path):
                    logging.info("Found normal image, will and push")
                    additional_images = self._get_additional_images(image_name)
                    for image in additional_images:
                        images_processing_result += self.process_single_image(version, full_path, image, self.Parameters.version_from_pr)
                elif os.path.isfile(full_path) and 'alpine' in full_path:
                    logging.info("Found alpine image, will build it with custom command")
                    additional_images = self._get_additional_images(image_name)
                    for image in additional_images:
                        images_processing_result += self.process_alpine_image(version, full_path, image)
            else:
                logging.info("Path %s doesn't exists", full_path)

        logging.info("Finished")
        if len(images):
            description = "Updated " + ','.join([im[1] for im in images])
        else:
            description = "Nothing to update"

        s3_path_prefix = str(pull_request.number) + "/" + self.commit.sha + "/" + self.get_context_name().lower().replace(' ', '_')
        status, test_results = self._process_test_results(images_processing_result, s3_path_prefix)

        if len(description) >= 140:
            description = description[:136] + "..."

        if not test_results:
            test_results = [('Nothig to update', 'OK')]

        return status, description, test_results, 'https://hub.docker.com/u/clickhouse'
