# -*- coding: utf-8 -*-
import logging
import sandbox.sdk2.helpers
import sandbox.sdk2.ssh
import subprocess

import sandbox.common.types.task as ctt
import os
from sandbox import sdk2
import sandbox.common.types.resource as ctr
import sandbox.common.types.client as ctc
from sandbox.projects.clickhouse.BaseOnCommitTask.base import NeedToRunDescription
from sandbox.projects.clickhouse.BaseOnCommitTask.base_text_test_task import BaseTextTestTask
from sandbox.projects.clickhouse.resources import CLICKHOUSE_DOCS_LXC_CONTAINER, CLICKHOUSE_REPO_NO_SUBMODULES
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.clickhouse.util.task_helper import has_changes_in_documentation


class ClickhouseDocsRelease(BaseTextTestTask):
    class Requirements(BaseTextTestTask.Requirements):
        privileged = True
        client_tags = ctc.Tag.GENERIC

    class Parameters(BaseTextTestTask.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            resource_type=CLICKHOUSE_DOCS_LXC_CONTAINER,
        )

        cloudflare_token = sdk2.parameters.String("Cloudflare token", default="clickhouse-robot-cloudflare-token")

    def on_create(self):
        self.Parameters._container = sdk2.Resource.find(
            CLICKHOUSE_DOCS_LXC_CONTAINER,
            state=ctr.State.READY,
            attrs=dict(released="stable")
        ).order(-CLICKHOUSE_DOCS_LXC_CONTAINER.id).first().id

    @staticmethod
    def need_to_run(pr_info):
        if not has_changes_in_documentation(pr_info):
            return NeedToRunDescription(False, "No diff in .md, .html, etc...", False)

        return NeedToRunDescription(True)

    @staticmethod
    def get_context_name():
        return "Docs release"

    @staticmethod
    def need_docker():
        return False

    def on_enqueue(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name="CLICKHOUSE_DOCS_RELEASE_SEMAPHORE"),
            ]
        )

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        logging.info("Searching for CLICKHOUSE_REPO_NO_SUBMODULES at commit %s", commit.sha)
        resources = sdk2.Resource.find(
            CLICKHOUSE_REPO_NO_SUBMODULES,
            state=ctr.State.READY,
            attrs=dict(commit=commit.sha, pr_number=pull_request.number, docs_change=True)
        ).order(-CLICKHOUSE_REPO_NO_SUBMODULES.id).limit(1)
        res = resources.first()
        logging.info("Search finished: %s", res.id if res else 'not found')
        return res

    def install_newer_python(self):
        with sandbox.sdk2.helpers.ProcessLog(self, logger="insall_python") as pl:
            cmd = "apt-get update && apt-get -o DPkg::Options::=\"--force-confold\" install python3.7 python3.7-venv python3-pip --yes --force-yes && "\
                  "python3.7 -m pip install pip && "\
                  "update-alternatives --install /usr/bin/python3 python3 /usr/bin/python3.7 1"
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            if retcode != 0:
                raise Exception("Cannot prepare new python")

    def send_mail(self, sha, state):
        try:
            purl = 'https://dash.cloudflare.com/d4fd593eebab2e3a58a599400c4cd64d/clickhouse.tech'
            surl = 'https://clickhouse.tech/'
            header = 'ClickHouse docs release with state {}'.format(state)
            body = '''Version sha {s} was released with state {st}
            Task: {turl}
            Cloudflare url: {purl}
            Site url: {surl}
            '''.format(s=sha, st=state, turl=self._get_self_url(), purl=purl, surl=surl)
            channel.sandbox.send_email("clickhouse-docs-releases@yandex-team.ru", '', header, body)
        except Exception as ex:
            logging.info("Exception happened sending mail", ex)

    def check(self, repo_path, commit, repo, pull_request):
        cloudflare_token = sdk2.Vault.data(self.Parameters.cloudflare_token)
        release_output = os.path.join(str(self.path()), "release_output.txt")
        my_env = os.environ.copy()
        my_env["CLOUDFLARE_TOKEN"] = cloudflare_token
        current_sha = commit.sha

        self.install_newer_python()
        with sandbox.sdk2.helpers.ProcessLog(self, logger="prepare_venv") as pl:
            cmd = "/bin/bash -c \"set -o pipefail && cd {}/docs/tools && python3 -m pip install --ignore-installed --upgrade setuptools pip virtualenv \
                && mkdir venv && virtualenv -p $(which python3) venv && source venv/bin/activate \
                && python3 -m pip install --ignore-installed -r requirements.txt | tee {}\"".format(repo_path, release_output)
            logging.info("Executing cmd '%s'", cmd)
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout, env=my_env).wait()
            if retcode != 0:
                logging.info("Cannot prepare env")
                return [("FAIL", "Cannot prepare env")], release_output

        lines = []
        with sandbox.sdk2.ssh.Key(self, self.task_owner, self.Parameters.ssh_key):
            with sandbox.sdk2.helpers.ProcessLog(self, logger="docs_release") as pl:
                cmd = "/bin/bash -c \"set -o pipefail && cd {}/docs/tools && virtualenv -p $(which python3) venv && source venv/bin/activate \
                        && ./release.sh 2>&1 | tee {}\"".format(repo_path, release_output)
                logging.info("Executing cmd '%s'", cmd)
                retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout, env=my_env).wait()
                logging.info("Cmd finished with exit code %s", retcode)
                if retcode != 0:
                    logging.info("Error happened, sending sad mail :(")
                    self.send_mail(current_sha, 'failure')
                    lines = [("FAIL", "Info in logs")]
                else:
                    logging.info("No errors, trying to send happy mail")
                    self.send_mail(current_sha, 'success')

        return lines, release_output
