# -*- coding: utf-8 -*-
import os
from sandbox.projects.clickhouse.BaseOnCommitTask.test_task import BaseOnCommitTestTask
from sandbox.projects.clickhouse.resources import CLICKHOUSE_BUILD_LXC_CONTAINER, CLICKHOUSE_BUILD_CACHE
from sandbox import sdk2
from sandbox.sdk2.path import Path
import logging
import subprocess
import datetime
import time

import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
from sandbox.projects.clickhouse.BaseOnCommitTask.base import PostStatuses, NeedToRunDescription
import sandbox.common.types.misc as ctm
from sandbox.projects.clickhouse.util.deprecated import FunctionalTestDeprecated

from sandbox.projects.clickhouse.util.task_helper import compress_fast, decompress_fast


def execute_with_retry(function, name, retry_count):
    result_ex = None
    for i in range(retry_count):
        try:
            function()
            break
        except Exception as ex:
            logging.warn("Function '%s' failed with exception %s retry %s", name, str(ex), i + 1)
            result_ex = ex
            time.sleep(i * 2 + 1)
    else:
        raise result_ex


def replace_duplicates_test_results(test_results):
    filtered_results = {}
    for test_name, result, test_time in test_results:
        if test_name not in filtered_results:
            filtered_results[test_name] = (result, test_time)
        elif filtered_results[test_name][0] != 'OK':
            filtered_results[test_name] = (result, test_time)

    total = 0
    skipped = 0
    failed = 0
    success = 0
    flatten_result = [(key, value[0], value[1]) for key, value in filtered_results.items()]
    result = list(reversed(sorted(flatten_result, key=lambda x: x[0])))
    for test_name, test_result, _ in result:
        total += 1
        if test_result == 'Timeout':
            failed += 1
        elif test_result == 'FAIL':
            failed += 1
        elif test_result == 'SKIPPED':
            skipped += 1
        else:
            success += 1
    return total, skipped, failed, success, list(result)


class ClickhouseFastTest(BaseOnCommitTestTask):

    class Requirements(BaseOnCommitTestTask.Requirements):
        client_tags = ctc.Tag.GENERIC & (ctc.Tag.INTEL_E5_2660 | ctc.Tag.INTEL_E5_2660V1 | ctc.Tag.INTEL_E5_2660V4 | ctc.Tag.INTEL_E5_2683V4 | ctc.Tag.INTEL_E5_2683)
        privileged = True
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 80 * 1024, None)
        ram = 32 * 1024  # 32GB

    def with_raw_logs(self):
        return True

    @staticmethod
    def need_docker():
        return True

    @staticmethod
    def get_context_name():
        return "Fast test"

    @staticmethod
    def get_images_names():
        return ["clickhouse/fasttest"]

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        # always ready to run
        return True

    @staticmethod
    def order():
        return 0

    @staticmethod
    def require_internet():
        return False

    @staticmethod
    def need_to_run(pr_info):
        if 'pr-backport' in pr_info.labels or 'release' in pr_info.labels:
            return NeedToRunDescription(False, 'Not ran for backport or release PRs', False)

        # We can't check for changes in code here, because the list of changed
        # files is built by the repo cloner task, and we don't wait for it, to
        # reduce latency. But we can skip the documentation PRs by tag.

        if 'pr-documentation' in pr_info.labels:
            return NeedToRunDescription(False, 'Not ran for pr-documentation PRs', False)

        return NeedToRunDescription(True)

    class Parameters(BaseOnCommitTestTask.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            resource_type=CLICKHOUSE_BUILD_LXC_CONTAINER,
        )

    def on_create(self):
        self.Parameters._container = sdk2.Resource.find(
            CLICKHOUSE_BUILD_LXC_CONTAINER,
            state=ctr.State.READY,
            attrs=dict(released="stable")
        ).order(-CLICKHOUSE_BUILD_LXC_CONTAINER.id).limit(1).first().id

    def post_statuses(self):
        return PostStatuses.ALWAYS

    def fetch_cache_resource(self, pr_number, commit_sha, target_dir):
        attrs_variants = [
            {"build_type": "fasttest", "pr_number": pr_number},
            {"build_type": "fasttest"},
        ]

        res = None
        for attrs in attrs_variants:
            res = sdk2.Resource.find(
                CLICKHOUSE_BUILD_CACHE,
                state=ctr.State.READY,
                attrs=attrs).order(-CLICKHOUSE_BUILD_CACHE.id).limit(1).first()
            if res:
                logging.info("Found ccache resource id %s from PR %s @ %s by attrs filter %s", res.id, res.pr_number, res.commit, attrs)
                break

        if not res:
            logging.info("Not found ccache resource")
            return

        cache_data = sdk2.ResourceData(res)
        decompress_fast(cache_data.path, target_dir)
        logging.info("Extracted ccache resource from %s to %s", cache_data.path, target_dir)

    def upload_cache_resource(self, pr_number, commit_sha, cache_dir):
        logging.info("Packing ccache for PR %s @ %s from %s", pr_number, commit_sha, cache_dir)
        compress_fast(cache_dir, "clickhouse_ccache.tar.gz")
        ccache_res = CLICKHOUSE_BUILD_CACHE(
            self,
            "ClickHouse build cache for PR {} @ {}".format(pr_number, commit_sha),
            "./clickhouse_local_ccache.tar.gz",
            build_type="fasttest",
            pr_number=pr_number,
            commit=commit_sha,
            date=datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        )
        ccache_data = sdk2.ResourceData(ccache_res)
        def cache_resource_upload():
            ccache_data.path.write_bytes(Path("clickhouse_ccache.tar.gz").read_bytes())
            ccache_data.ready()
        execute_with_retry(cache_resource_upload, "Upload cache resource", 5)
        logging.info("Write finished")

    # return state, description, test_result, raw_log_path
    def run(self, commit, repo, pull_request):

        test_output = os.path.join(str(self.path()), 'test_output')
        os.mkdir(test_output)

        # For newer versions of fast test.
        workspace = os.path.join(str(self.path()), 'fasttest-workspace')
        os.mkdir(workspace)

        # Note that the resulting dir will be <workspace>/ccache, where the
        # 'ccache' dir comes from the tar archive. This is because the
        # decompress_fast/compress_fast functions have a weird interface,
        # where the former has to be passed <workspace>, but the latter
        # has to be passed <workspace>/ccache. Probably decompress_fast
        # should have specified `tar --strip-component=1` for symmetry.
        self.fetch_cache_resource(pull_request.number, commit.sha, workspace)

        cmd = "docker run " \
            "-e FASTTEST_WORKSPACE=/fasttest-workspace -e FASTTEST_OUTPUT=/test_output " \
            "-e FASTTEST_SOURCE=/ClickHouse --cap-add=SYS_PTRACE " \
            "-e PULL_REQUEST_NUMBER={} -e COMMIT_SHA={} -e COPY_CLICKHOUSE_BINARY_TO_OUTPUT=1 " \
            "--volume={}:/fasttest-workspace --volume={}:/ClickHouse --volume={}:/test_output {}".format(
                pull_request.number, commit.sha, workspace, str(self.ramdrive.path), test_output, self.get_single_image_with_version())

        state = "success"

        with sdk2.helpers.ProcessLog(self, logger='runlog') as pl:
            with open(str(pl.stdout.path), 'a') as f:
                f.write("Executing cmd: '{}'\n".format(cmd))
            logging.info("Executing cmd: %s", cmd)
            process = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout)
            retcode = process.wait()  # no timeout in python2
            if retcode == 0:
                logging.info("Run successfully")
            else:
                logging.info("Run failed")

            run_log_path = str(pl.stdout.path)

        test_output_files = os.listdir(test_output)
        additional_logs = []
        for f in test_output_files:
            additional_logs.append(os.path.join(test_output, f))

        test_log_exists = 'test_log.txt' in test_output_files or 'test_result.txt' in test_output_files
        test_result_exists = 'test_results.tsv' in test_output_files
        test_results = []
        if 'submodule_log.txt' not in test_output_files:
            description = "Cannot clone repository"
            state = "failure"
        elif 'cmake_log.txt' not in test_output_files:
            description = "Cannot fetch submodules"
            state = "failure"
        elif 'build_log.txt' not in test_output_files:
            description = "Cannot finish cmake"
            state = "failure"
        elif 'install_log.txt' not in test_output_files:
            description = "Cannot build ClickHouse"
            state = "failure"
        elif not test_log_exists and not test_result_exists:
            description = "Cannot install or start ClickHouse"
            state = "failure"
        else:
            self.upload_cache_resource(
                pull_request.number, commit.sha,
                os.path.join(workspace, 'ccache'))

            if test_result_exists:
                state, description, test_results, additional_logs = self.process_result_simple(test_output, None, None, commit, repo, pull_request)
            else:
                if 'test_log.txt' in test_output_files:
                    raw_log = 'test_log.txt'
                else:
                    raw_log = 'test_result.txt'
                total, skipped, unknown, failed, success, _, test_results = FunctionalTestDeprecated.process_test_log(os.path.join(test_output, raw_log))

                total, skipped, failed, success, test_results = replace_duplicates_test_results(test_results)

                tail = ""
                if failed != 0 or success == 0:
                    state = "failure"
                    tail = ". Other tests won't be run before fast test will be fixed."

                description = "fail: {}, passed: {}".format(failed, success)
                if skipped != 0:
                    description += ", skipped: {}".format(skipped)

                description += tail
        return state, description, test_results, run_log_path, additional_logs
