# -*- coding: utf-8 -*-
import os
import sandbox.common.types.client as ctc
from sandbox.projects.clickhouse.BaseOnCommitTask.simple_test_task import SimpleDockerBuildTestTask
from distutils.version import StrictVersion

MAX_GLIBC_VERSION = '2.4'


class ClickhouseGlibcCompatibilityCheck(SimpleDockerBuildTestTask):

    class Requirements(SimpleDockerBuildTestTask.Requirements):
        client_tags = ctc.Tag.INTEL_E5645

    @staticmethod
    def get_context_name():
        return "Compatibility check"

    @staticmethod
    def get_images_names():
        return ["yandex/clickhouse-test-old-ubuntu", "yandex/clickhouse-test-old-centos"]

    @staticmethod
    def required_build_properties():
        return ('gcc-9', 'deb', 'relwithdebuginfo', 'none', 'bundled', 'unsplitted', 'disable', False)

    @staticmethod
    def order():
        return 1000

    def get_prepare_cmd(self, build_path, result_folder, server_log_folder, _):
        return "dpkg -x {bp}/clickhouse-common-static_* {rf} && \
                dpkg -x {bp}/clickhouse-server_* {rf}".format(bp=build_path, rf=result_folder)

    def get_run_cmd(self, build_path, result_folder, server_log_folder, repo_path, perfraw_path):
        return [
            "readelf -s {}/usr/bin/clickhouse | grep '@GLIBC_' > {}/glibc.log".format(result_folder, result_folder),
            "readelf -s {}/usr/bin/clickhouse-odbc-bridge | grep '@GLIBC_' >> {}/glibc.log".format(result_folder, result_folder),
            "docker run --network=host --volume={}/usr/bin/clickhouse:/clickhouse \
            --volume={}/etc/clickhouse-server:/config \
            --volume={}:/var/log/clickhouse-server {} > {}/ubuntu:12.04".format(
                result_folder, result_folder, server_log_folder, self.get_image_with_version("yandex/clickhouse-test-old-ubuntu"), result_folder),
            "docker run --network=host --volume={}/usr/bin/clickhouse:/clickhouse \
            --volume={}/etc/clickhouse-server:/config \
            --volume={}:/var/log/clickhouse-server {} > {}/centos:5".format(
                result_folder, result_folder, server_log_folder, self.get_image_with_version("yandex/clickhouse-test-old-centos"), result_folder),
        ]

    def _process_os_check(self, log_path):
        name = os.path.basename(log_path)
        with open(log_path, 'r') as log:
            line = log.read().split('\n')[0].strip()
            if line != 'OK':
                return (name, "FAIL")
            else:
                return (name, "OK")

    def _process_glibc_check(self, log_path):
        bad_lines = []
        with open(log_path, 'r') as log:
            for line in log:
                if line.strip():
                    columns = line.strip().split(' ')
                    symbol_with_glibc = columns[-2]  # sysconf@GLIBC_2.2.5
                    symbol, version = symbol_with_glibc.split('@GLIBC_')
                    if version == 'PRIVATE':
                        bad_lines.append((symbol_with_glibc, "FAIL"))
                    elif StrictVersion(version) > MAX_GLIBC_VERSION:
                        bad_lines.append((symbol_with_glibc, "FAIL"))
        if not bad_lines:
            bad_lines.append(("glibc check", "OK"))
        return bad_lines

    # TODO Move it to github (it's not trivial, because we run two dockers)
    def process_result(self, result_folder, server_log_folder, perfraw_path,
                       commit, repo, pull_request):

        summary = self._process_glibc_check(os.path.join(result_folder, "glibc.log"))

        status = "success"
        description = "Compatibility check passed"
        if len(summary) > 1 or summary[0][1] != "OK":
            status = "failure"
            description = "glibc check failed"

        if status == "success":
            for operating_system in ("ubuntu:12.04", "centos:5"):
                result = self._process_os_check(os.path.join(result_folder, operating_system))
                if result[1] != "OK":
                    status = "failure"
                    description = "Old {} failed".format(operating_system)
                    summary += [result]
                    break
                else:
                    summary += [result]

        server_log_path = os.path.join(server_log_folder, "clickhouse-server.log")
        stderr_log_path = os.path.join(server_log_folder, "stderr.log")
        client_stderr_log_path = os.path.join(server_log_folder, "clientstderr.log")
        result_logs = []
        if os.path.exists(server_log_path):
            result_logs.append(server_log_path)

        if os.path.exists(stderr_log_path):
            result_logs.append(stderr_log_path)

        if os.path.exists(client_stderr_log_path):
            result_logs.append(client_stderr_log_path)

        return status, description, summary, result_logs
