# -*- coding: utf-8 -*-
import requests
import logging
from sandbox.projects.clickhouse.resources import CLICKHOUSE_BUILD_LXC_CONTAINER
from sandbox.projects.clickhouse.BaseOnCommitTask.test_task import BaseOnCommitTestTask
from sandbox.projects.clickhouse.BaseOnCommitTask.base import PostStatuses, NeedToRunDescription
from sandbox.sandboxsdk.process import run_process
import sandbox.common.types.client as ctc
from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.common.types.task import Semaphores
from sandbox.projects.clickhouse.ClickhouseBuilder import get_build_parameters_as_string
from sandbox.projects.clickhouse.util.task_helper import get_ci_config, compress_fast
from sandbox.sdk2.ssh import Key as SSHKey
import os

NODES_FILE = """ch-iva.iva.yp-c.yandex.net
ch-sas.sas.yp-c.yandex.net
ch-man.man.yp-c.yandex.net
"""


SUCCESSFUL_TESTS_ANCHOR = "# Successful tests"
INTERMINATE_TESTS_ANCHOR = "# Indeterminate tests"
CRASHED_TESTS_ANCHOR = "# Crashed tests"
FAILED_TESTS_ANCHOR = "# Failed tests"


def _parse_jepsen_output(path):
    test_results = []
    current_type = ''
    with open(path, 'r') as f:
        for line in f:
            if SUCCESSFUL_TESTS_ANCHOR in line:
                current_type = 'OK'
            elif INTERMINATE_TESTS_ANCHOR in line or CRASHED_TESTS_ANCHOR in line:
                current_type = 'ERROR'
            elif FAILED_TESTS_ANCHOR in line:
                current_type = 'FAIL'

            if (line.startswith('store/clickhouse-keeper') or line.startswith('clickhouse-keeper')) and current_type:
                test_results.append((line.strip(), current_type))

    return test_results


class ClickhouseKeeperJepsen(BaseOnCommitTestTask):

    class Requirements(BaseOnCommitTestTask.Requirements):
        privileged = True
        client_tags = ctc.Tag.GENERIC

    class Parameters(BaseOnCommitTestTask.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            resource_type=CLICKHOUSE_BUILD_LXC_CONTAINER,
        )
        kill_timeout = 3 * 60 * 60

    def on_create(self):
        self.Parameters._container = sdk2.Resource.find(
            CLICKHOUSE_BUILD_LXC_CONTAINER,
            state=ctr.State.READY,
            attrs=dict(released="stable")
        ).order(-CLICKHOUSE_BUILD_LXC_CONTAINER.id).first().id

    def on_enqueue(self):
        self.Requirements.semaphores = Semaphores(acquires=[
            Semaphores.Acquire(name="clickhouse_keeper_jepsen")
        ])

    @staticmethod
    def get_context_name():
        return 'ClickHouse Keeper Jepsen'

    @staticmethod
    def binary_to_download():
        return 'clang-13_relwithdebuginfo_none_bundled_unsplitted_disable_False_binary'

    @staticmethod
    def get_images_names():
        return ["clickhouse/keeper-jepsen-test"]

    def post_statuses(self):
        return PostStatuses.ALWAYS

    @staticmethod
    def need_docker():
        return True

    @staticmethod
    def require_internet():
        return True

    @staticmethod
    def need_to_run(pr_info):
        if 'jepsen-test' in pr_info.labels:
            return NeedToRunDescription(True, 'jepsen-test label on PR', True)

        return NeedToRunDescription(False, 'Not ran for PR without jepsen-test label', False)

    @classmethod
    def get_build_path(cls, commit, pull_request):
        # The build is uploaded to S3 some time after the Sandbox resource is published,
        # so there is no point checking for the Sandbox resource, and we just check that
        # it did upload.
        ci_config = get_ci_config(pull_request, commit)
        if ci_config and cls.get_context_name() in ci_config["tests_config"]:
            test_config = ci_config["tests_config"][cls.get_context_name()]["required_build_properties"]
            build_path = get_build_parameters_as_string(
                test_config["compiler"], test_config["build_type"], test_config["sanitizer"],
                test_config["bundled"], test_config["splitted"], test_config["clang-tidy"],
                test_config["with_coverage"], test_config["package_type"])
            logging.info("CI config found will search build at %s", build_path)
        else:
            build_path = cls.binary_to_download()

        return 'https://clickhouse-builds.s3.yandex.net/{}/{}/clickhouse_build_check/{}/clickhouse'.format(
            pull_request.number, commit.sha, build_path)

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        s3_build = cls.get_build_path(commit, pull_request)
        response = requests.head(s3_build)
        if response.ok:
            return True
        try:
            response.raise_for_status()
        except BaseException as error:
            logging.info("Exception while downloading the clickhouse binary: '{}'".format(error))
        return False

    def run(self, commit, repo, pull_request):
        logging.info("commit " + str(commit) + ", pr " + str(pull_request))
        nodes_path = os.path.join(str(self.path()), "nodes.txt")
        with open(nodes_path, 'w') as f:
            f.write(NODES_FILE)
            f.flush()

        output_path = os.path.join(str(self.path()), "test_output")
        os.mkdir(output_path)

        with SSHKey(self, self.task_owner, self.ssh_key):
            ssh_auth_sock = os.environ['SSH_AUTH_SOCK']
            auth_sock_dir = os.path.dirname(ssh_auth_sock)
            cmd = ("docker run --network=host -v '{}:{}' -e SSH_AUTH_SOCK={} "
                   "-e PR_TO_TEST={} -e SHA_TO_TEST={} -v '{}:/nodes.txt' -v '{}:/test_output' "
                   "{}").format(
                       auth_sock_dir,
                       auth_sock_dir,
                       ssh_auth_sock,
                       pull_request.number,
                       commit.sha,
                       nodes_path,
                       output_path,
                       self.get_single_image_with_version())

            logging.info('run cmd "{}"'.format(cmd))
            retcode = run_process(cmd, shell=True, log_prefix="run_docker_cmd.log")

            logging.info('retcode {}'.format(str(retcode)))

        status = 'success'
        description = 'No invalid analysis found ヽ(‘ー`)ノ'
        jepsen_log_path = os.path.join(output_path, 'jepsen_run_all_tests.log')
        additional_data = []
        try:
            test_result = _parse_jepsen_output(jepsen_log_path)
            if any(r[1] == 'FAIL' for r in test_result):
                status = 'failure'
                description = 'Found invalid analysis (ﾉಥ益ಥ）ﾉ ┻━┻'
            compress_fast(os.path.join(output_path, 'store'), 'jepsen_store.tar.gz')
            additional_data.append(os.path.join(str(self.path()), 'jepsen_store.tar.gz'))
        except:
            status = 'failure'
            description = 'No Jepsen output log'
            test_result = [('No Jepsen output log', 'FAIL')]

        return status, description, test_result, jepsen_log_path, additional_data
