# -*- coding: utf-8 -*-
from sandbox.projects.clickhouse.BaseOnCommitTask.base import PostStatuses, NeedToRunDescription
from sandbox.projects.clickhouse.BaseOnCommitTask.test_task import BaseOnCommitTestTask
from sandbox.projects.clickhouse.resources import CLICKHOUSE_BUILD_LXC_CONTAINER
from sandbox.projects.clickhouse.util.task_helper import get_nda_task_url
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.misc as ctm
from sandbox import sdk2
import subprocess
import logging
import os
import re
import requests
import time
import traceback


class ClickhousePerformanceComparison(BaseOnCommitTestTask):

    class Requirements(BaseOnCommitTestTask.Requirements):
        privileged = True

        # 250 GiB RAM total -- 150 GiB for tmpfs and the rest for servers.
        ram = 250 * 1024

        # 150 GiB ram drive. The datasets are about 65 GiB, and some
        # additional space is needed for logs and insert tests.
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 150 * 1024, None)

        client_tags = ctc.Tag.GENERIC & ctc.Tag.INTEL_GOLD_6230

        # Require even less than the default 30 GB, because with 30 GB
        # we still don't fit some hosts. We only need a couple GB to
        # download the docker container, most of the test files are on
        # tmpfs.
        disk_space = 2 * 1024

    class Parameters(BaseOnCommitTestTask.Parameters):
        _container = sdk2.parameters.Container(
            "Environment container resource",
            resource_type=CLICKHOUSE_BUILD_LXC_CONTAINER,
        )
        kill_timeout = 8 * 60 * 60

    def on_create(self):
        self.Parameters._container = sdk2.Resource.find(
            CLICKHOUSE_BUILD_LXC_CONTAINER,
            state=ctr.State.READY,
            attrs=dict(released="stable")
        ).order(-CLICKHOUSE_BUILD_LXC_CONTAINER.id).first().id

        # Try other machines besides Gold 6230. The tests require 20
        # cores w/o HyperThreading, so we probably should request twice as much.
        # This is changed here as a reminder that you can easily put it under `if`
        # and, for example, only change it for master, like we did before.
        self.Requirements.cores = 40
        self.Requirements.client_tags = ctc.Tag.GENERIC
        self.Parameters.kill_timeout = 12 * 60 * 60

    @staticmethod
    def get_context_name():
        return 'Performance'

    @staticmethod
    def get_images_names():
        return ["yandex/clickhouse-performance-comparison"]

    # Start early, because this test is long.
    @staticmethod
    def order():
        return 10

    def process(self, commit, repo, pull_request):
        logging.info("commit " + str(commit) + ", pr " + str(pull_request))

        docker_env = ''
        if pull_request.number != 0 and 'force tests' in set([label.name for label in pull_request.labels]):
            # Run all perf tests if labeled 'force tests'.
            docker_env += ' -e CHPC_MAX_QUERIES=0 '

        # The comparison script can upload the results into a test database. Set
        # up the environment if specified.
        clickhouse_url = sdk2.Vault.data("clickhouse-test-stat-url-native")
        if clickhouse_url:
            docker_env += ' -e CHPC_DATABASE_URL={} -e CHPC_DATABASE_USER={} -e CHPC_DATABASE_PASSWORD={} '.format(
                clickhouse_url, sdk2.Vault.data(self.Parameters.clickhouse_login),
                sdk2.Vault.data(self.Parameters.clickhouse_password))

        # Inject task and PR link to the report.
        if pull_request.number == 0:
            pr_link = commit.html_url
        else:
            pr_link = pull_request.html_url
        docker_env += ' -e CHPC_ADD_REPORT_LINKS="<a href={}>Task (internal network)</a> <a href={}>Tested commit</a>"'.format(
            get_nda_task_url(self.id), pr_link)

        # For some reason we don't have acces to /dev/mem anymore, so dmidecode
        # won't work. Add  '--device=/dev/mem:/dev/mem ' to docker invocation to
        # see if it's fixed.
        cmd = 'docker run --network=host ' \
              '--privileged ' \
              '--volume={}:/workspace --volume={}:/output ' \
              '--cap-add syslog --cap-add sys_admin --cap-add sys_rawio '\
              '-e PR_TO_TEST={} -e SHA_TO_TEST={} '\
              '{} {}'.format(
                  self.ramdrive.path,
                  self.ramdrive.path,
                  str(pull_request.number),
                  str(commit.sha),
                  docker_env,
                  self.get_single_image_with_version())

        logging.info('run cmd {}'.format(cmd))
        process = subprocess.Popen(cmd, shell=True)
        status = ''
        status_file_path = os.path.join(str(self.ramdrive.path), 'status.txt')
        while process.poll() is None:
            try:
                new_status = status
                new_status = open(status_file_path, 'r').read().strip()
            except Exception:
                traceback.print_exc()
                pass
            if status != new_status:
                try:
                    status = new_status
                    self.update_status(text=new_status, url=get_nda_task_url(self.id), commit=commit)
                except Exception:
                    traceback.print_exc()
                    pass
            time.sleep(120)
        logging.info('Return code {}'.format(str(process.returncode)))

        subprocess.Popen('free -h', shell=True).wait()
        subprocess.Popen('ps -eo pid,cmd,%cpu,%mem --sort=-%mem | head -20', shell=True).wait()
        subprocess.Popen('df -h', shell=True).wait()
        subprocess.Popen('docker volume ls', shell=True).wait()
        subprocess.Popen('ls -lath', shell=True).wait()
        subprocess.Popen('tree', shell=True).wait()

        s3_prefix = '{}/{}/performance_comparison/'.format(str(pull_request.number), str(commit.sha))
        paths = {
            'compare.log': 'compare.log',
            'output.7z': 'output.7z',
            'report.html': 'report.html',
            'all-queries.html': 'all-queries.html',
            'queries.rep': 'queries.rep',
            'all-query-metrics.tsv': 'report/all-query-metrics.tsv',
        }
        for file in paths:
            try:
                paths[file] = self.s3_client.upload_test_report_to_s3(
                    os.path.join(str(self.ramdrive.path), paths[file]),
                    s3_prefix + file)
            except Exception:
                paths[file] = ''
                traceback.print_exc()
                pass

        # Upload all images and flamegraphs to S3
        try:
            self.s3_client.upload_test_folder_to_s3(
                os.path.join(str(self.ramdrive.path), 'images'),
                s3_prefix + 'images'
            )
        except Exception:
            traceback.print_exc()
            pass

        # Try to fetch status from the report.
        status = ''
        message = ''
        try:
            report_text = open(os.path.join(str(self.ramdrive.path), 'report.html'), 'r').read()
            status_match = re.search('<!--[ ]*status:(.*)-->', report_text)
            message_match = re.search('<!--[ ]*message:(.*)-->', report_text)
            if status_match:
                status = status_match.group(1).strip()
            if message_match:
                message = message_match.group(1).strip()
        except Exception:
            traceback.print_exc()
            status = 'failure'
            message = 'Failed to parse the report.'
            pass

        if not status:
            status = 'failure'
            message = 'No status in report.'
        elif not message:
            status = 'failure'
            message = 'No message in report.'

        report_url = self.task_url

        if paths['compare.log']:
            report_url = paths['compare.log']

        if paths['output.7z']:
            report_url = paths['output.7z']

        if paths['report.html']:
            report_url = paths['report.html']

        return status, message, report_url

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        # The build is uploaded to S3 some time after the Sandbox resource is published,
        # so there is no point checking for the Sandbox resource, and we just check that
        # it did upload.
        s3_build = 'https://clickhouse-builds.s3.yandex.net/{}/{}/clickhouse_build_check/performance/performance.tgz'.format(pull_request.number, commit.sha)
        response = requests.head(s3_build)
        if not response.ok:
            # logging.info(response.raise_for_status())
            return False
        return True

    def post_statuses(self):
        return PostStatuses.ALWAYS

    @staticmethod
    def need_docker():
        return True

    @staticmethod
    def require_internet():
        return True

    @staticmethod
    def need_to_run(pr_info):
        if 'pr-backport' in pr_info.labels or 'release' in pr_info.labels:
            return NeedToRunDescription(False, 'Not ran for backport or release PRs', False)

        return BaseOnCommitTestTask.need_to_run(pr_info)
