# -*- coding: utf-8 -*-
import copy
import logging
import os
import requests
import sandbox.common.types.resource as ctr
import sandbox.common.types.client as ctc
import shutil
import time
import sys

from sandbox import sdk2
from sandbox.projects.clickhouse.util.misc import retry
from sandbox.projects.clickhouse.BaseOnCommitTask.base import BaseOnCommitTask, PostStatuses
from sandbox.projects.clickhouse.ClickhouseDockerHubPush import ClickhouseDockerHubPush
from sandbox.projects.clickhouse.resources import CLICKHOUSE_BUILD, CLICKHOUSE_REPO
from sandbox.projects.clickhouse.util.git_helper import GitHelper, ClickHouseVersion, build_version_description
from sandbox.projects.common.conductor import Api as ConductorAPI, OauthAuth as ConductorOAuth
from sandbox.projects.common.gnupg import GpgKey2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.ssh import Key as SSHKey
from sandbox.projects.clickhouse.util.task_helper import get_ci_config, compress_fast, decompress_fast

TEST_URLS = [
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-client_22.1.1.2542_all.deb",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-common-static-dbg_22.1.1.2542_amd64.deb",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-common-static_22.1.1.2542_amd64.deb",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-server_22.1.1.2542_all.deb",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-test_22.1.1.2542_all.deb",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse_22.1.1.2542_amd64.buildinfo",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse_22.1.1.2542_amd64.changes",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-client-22.1.1.2542.tgz",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-common-static-22.1.1.2542.tgz",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-common-static-dbg-22.1.1.2542.tgz",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-server-22.1.1.2542.tgz",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-test-22.1.1.2542.tgz",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-client-22.1.1.2542-2.noarch.rpm",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-common-static-22.1.1.2542-2.x86_64.rpm",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-common-static-dbg-22.1.1.2542-2.x86_64.rpm",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-server-22.1.1.2542-2.noarch.rpm",
    "https://s3.amazonaws.com/clickhouse-builds/22.1/dfe64a2789bbf51046bb6b5476f874f7b59d124c/package_release/clickhouse-test-22.1.1.2542-2.noarch.rpm",
]

PR_BODY_TEXT = """
This PullRequest is part of ClickHouse release cycle. It's used by CI system only. Don't perform any changes with it.
"""

RELEASE_LABEL = "release"

REPOS_TO_UPLOAD = {
    "yandex-xenial.dupload.dist.yandex.ru": "robot-clickhouse",
    "yandex-trusty.dupload.dist.yandex.ru": "robot-clickhouse",
    "yandex-precise.dupload.dist.yandex.ru": "robot-clickhouse",
    #  "metrika.dupload.dist.yandex.ru": "robot-clickhouse",
    #  "metrika-trusty.dupload.dist.yandex.ru": "robot-clickhouse",
    "metrika-xenial.dupload.dist.yandex.ru": "robot-clickhouse",
    "repo.mirror.yandex.net": "buildfarm",
}

DEPRECATED_PACKAGES = [
    "clickhouse-server-base",
    "clickhouse-server-common",
]

DOCKER_PACKAGES = {
    "docker/server": "yandex/clickhouse-server",
    "docker/server/alpine-build.sh": "yandex/clickhouse-server",
    "docker/client": "yandex/clickhouse-client",
}


@retry(tries=5, delay=5)
def get_person_logins_from_group(group_name, any_token):
    headers = {
        'Authorization': 'OAuth {}'.format(any_token)
    }
    response = requests.get(
        "https://staff-api.yandex-team.ru/v3/groupmembership",
        headers=headers,
        params={
            "_query": "group.url=='{}' and person.official.is_dismissed==False".format(group_name),
            "_fields": "person.login"
        }
    ).json()
    return [p['person']['login'] for p in response['result']]


def dowload_build_with_progress(url, path):
    logging.info("Downloading from %s to temp path %s", url, path)
    for i in range(5):
        try:
            with open(path, 'wb') as f:
                response = requests.get(url, stream=True)
                response.raise_for_status()
                total_length = response.headers.get('content-length')
                if total_length is None or int(total_length) == 0:
                    logging.info("No content-length, will download file without progress")
                    f.write(response.content)
                else:
                    dl = 0
                    total_length = int(total_length)
                    logging.info("Content length is %ld bytes", total_length)
                    for data in response.iter_content(chunk_size=4096):
                        dl += len(data)
                        f.write(data)
            break
        except Exception as ex:
            sys.stdout.write("\n")
            time.sleep(3)
            logging.info("Exception while downloading %s, retry %s", ex, i + 1)
            if os.path.exists(path):
                os.remove(path)
    else:
        raise Exception("Cannot download dataset from {}, all retries exceeded".format(url))

    sys.stdout.write("\n")
    logging.info("Downloading finished")


class ClickhouseRelease(BaseOnCommitTask):

    class Parameters(BaseOnCommitTask.Parameters):

        with sdk2.parameters.Group("Release parameters") as release_block:
            with sdk2.parameters.RadioGroup('Release type', required=True) as release_type:
                release_type.values['testing'] = release_type.Value(value='testing', default=True)
                release_type.values['prestable'] = release_type.Value(value='prestable')
                release_type.values['stable'] = release_type.Value(value='stable')
                release_type.values['lts'] = release_type.Value(value='lts')

        with sdk2.parameters.Group("Publication parameters") as pub_block:
            remote_repos = sdk2.parameters.List("Repos list to push", default=list(REPOS_TO_UPLOAD.keys()))
            push_to_git = sdk2.parameters.Bool("Push to git", default=True)
            push_to_repos = sdk2.parameters.Bool("Push to repos", default=True)
            push_to_dockerhub = sdk2.parameters.Bool("Push to dockerhub", default=True)
            update_startrek = sdk2.parameters.Bool("Update startrek", default=True)
            add_asset_to_github = sdk2.parameters.Bool("Add asset to GitHub", default=True)
            packages_from_s3 = sdk2.parameters.List("Path to s3 prefix with packages", default=[])

        with sdk2.parameters.Group("Secrets parameters") as secrets_block:
            gpg_pub_key_vault = sdk2.parameters.String("GPG public key vault key", default="robot-clickhouse-gpg-public")
            gpg_priv_key_vault = sdk2.parameters.String("GPG private key vault key", default="robot-clickhouse-gpg-private")
            gpg_passphrase_vault = sdk2.parameters.String("GPG passphrase vault key", default="robot-clickhouse-gpg-passphrase")
            gpg_key_user = sdk2.parameters.String("GPG key user", default="robot-clickhouse")
            conductor_token_vault = sdk2.parameters.String("Token for conductor", default="clickhouse-robot-coductor-token")
            startrek_token = sdk2.parameters.String("Startrek token", default="clickhouse-robot-st-token")
            dockerhub_password = sdk2.parameters.String("Dockerhub password", default="clickhouse-robot-dockerhub-password")

        with sdk2.parameters.Group("Other parameters") as other_block:
            force_run = sdk2.parameters.Bool("Force run, ignore other checks", default=True)

    class Requirements(BaseOnCommitTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            environments.PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"]),
        )
        # LINUX_BIONIC has newer gnupg that requires changes in push_packages script
        client_tags = ctc.Tag.GENERIC & (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL)

        privileged = True  # to run apt-get installs in build-all script

    def post_statuses(self):
        return PostStatuses.ALWAYS

    def _install_debian_packages(self):
        run_process(["apt-get", "update"], log_prefix="apt-get")
        run_process(
            ["apt-get", "install", "-y", "rpm"],
            log_prefix="apt-get"
        )

    @staticmethod
    def get_context_name():
        return "Release"

    @classmethod
    def get_sources(cls, commit, repo, pull_request):
        logging.info("Searching for CLICKHOUSE_REPO at commit %s", commit.sha)

        attrs = dict(
            commit=commit.sha,
            pr_number=pull_request.number
        )

        sresources = sdk2.Resource.find(
            CLICKHOUSE_REPO,
            attrs=attrs,
            state=ctr.State.READY
        ).order(-CLICKHOUSE_REPO.id).limit(1)
        logging.info("Search finished")

        return sresources.first()

    @classmethod
    def get_resources(cls, commit, repo, pull_request, pkg='deb'):
        logging.info("Searching for CLICKHOUSE_BUILD at commit %s", commit.sha)
        ci_config = get_ci_config(pull_request, commit)
        if not ci_config or cls.get_context_name() not in ci_config["tests_config"]:
            logging.info("Build config not found :(, will use default from params")
            build_type = 'relwithdebuginfo'
            sanitizer = 'none'
            bundled = 'bundled'
            compiler = 'gcc-9'
            splitted = "unsplitted"
        else:
            logging.info("Build config found, will take info from repository")
            tests_config = ci_config["tests_config"]
            test_config = tests_config[cls.get_context_name()]["required_build_properties"]

            build_type = test_config["build_type"]
            sanitizer = test_config["sanitizer"]
            compiler = test_config["compiler"]
            bundled = test_config["bundled"]
            splitted = test_config["splitted"]

        attrs = dict(
            commit=commit.sha,
            pr_number=pull_request.number,
            build_type=build_type,
            package_type=pkg,
            sanitizer=sanitizer,
            bundled=bundled,
            compiler=compiler,
            splitted=splitted,
        )
        logging.info("Searching for CLICKHOUSE_BUILD at commit %s attrs %s", commit.sha, attrs)

        bresources = sdk2.Resource.find(
            CLICKHOUSE_BUILD,
            attrs=attrs,
            state=ctr.State.READY
        ).order(-CLICKHOUSE_BUILD.id).limit(1)
        logging.info("Search finished")

        return bresources.first()

    def _save_src(self, resource):
        sources_dir = str(self.path())
        logging.info("Downloading CLICKHOUSE_REPO resource")
        resource_data = sdk2.ResourceData(resource)
        src_path = str(resource_data.path)

        decompress_fast(src_path, sources_dir)
        tar_path = os.path.join(sources_dir, "clickhouse_with_submobules.tar.gz")
        # tar fails to unpack crc files from avro - exclude them
        # .git is too large for GitHub Release artifacts.
        compress_fast(os.path.join(sources_dir, "ClickHouse"), tar_path, ["ClickHouse/.git", "ClickHouse/contrib/avro/lang/java/mapred/src/test"])

        os.chmod(tar_path, 0o0777)
        logging.info("Repack finished, sources path %s", tar_path)
        return tar_path

    def _save_build(self, build_resource, pkg):
        copy_to = "./packages_" + pkg
        logging.info("Downloading CLICKHOUSE_BUILD resource")
        build_data = sdk2.ResourceData(build_resource)
        build_path = str(build_data.path)  # deb package
        shutil.copytree(build_path, copy_to)
        os.chmod(copy_to, 0o0777)
        for f in os.listdir(copy_to):
            logging.info("Changing permissions of %s", f)
            os.chmod(os.path.join(copy_to, f), 0o0777)
        logging.info("Download finished, build path %s", copy_to)
        return os.path.join(str(self.path()), copy_to)

    def _remove_deprecated_packages(self, packages_path):
        for package in os.listdir(packages_path):
            for name in DEPRECATED_PACKAGES:
                if name in package:
                    logging.info("Removing deprecated package %s", package)
                    os.remove(os.path.join(packages_path, package))

    @staticmethod
    def require_internet():
        return True

    @staticmethod
    def run_labels():
        return {"release"}

    def s3_push_packages(self, packages_urls, remote_repo_path):
        groups = {'deb': [], 'binary': [], 'tgz': [], 'rpm': [], 'performance': []}
        for url in packages_urls:
            if url.endswith('performance.tgz'):
                groups['performance'].append(url)
            elif url.endswith('.deb') or url.endswith('.buildinfo') or url.endswith('.changes') or url.endswith('.tar.gz'):
                groups['deb'].append(url)
            elif url.endswith('.rpm'):
                groups['rpm'].append(url)
            elif url.endswith('.tgz'):
                groups['tgz'].append(url)
            else:
                groups['binary'].append(url)

        logging.info("Got packages groups %s", groups)

        debs = groups['deb']
        debs_path = os.path.join(str(self.path()), "deb_packages")
        os.makedirs(debs_path)
        for url in debs:
            fname = os.path.basename(url)
            logging.info("Downloading %s to %s", url, os.path.join(debs_path, fname))
            dowload_build_with_progress(url, os.path.join(debs_path, fname))

        rpms = groups['rpm']
        rpms_path = os.path.join(str(self.path()), "rpm_packages")
        os.makedirs(rpms_path)
        for url in rpms:
            fname = os.path.basename(url)
            logging.info("Downloading %s to %s", url, os.path.join(rpms_path, fname))
            dowload_build_with_progress(url, os.path.join(rpms_path, fname))

        tgzs = groups['tgz']
        tgzs_path = os.path.join(str(self.path()), "tgz_packages")
        os.makedirs(tgzs_path)
        for url in tgzs:
            fname = os.path.basename(url)
            logging.info("Downloading %s to %s", url, os.path.join(tgzs_path, fname))
            dowload_build_with_progress(url, os.path.join(tgzs_path, fname))

        logging.info("Pushing packages to repositories")
        results = self.push_packages(debs_path, remote_repo_path, self.Parameters.remote_repos, rpms_path, tgzs_path)
        for repo_url, retcode, log in results:
            if retcode != 0:
                raise Exception("Push to repo {} failed, see info in log {}".format(repo_url, log))

    def push_package(self, deb_path, repo_path, repo_url, passphrase, gpg_user, release_type, rpm_path='', tgz_path=''):
        log_name = "push_to{}.log".format(repo_url.replace('.', '_'))
        rpm_opt = ''
        tgz_opt = ''
        if rpm_path:
            rpm_opt = '--rpm-directory ' + rpm_path
        if tgz_path:
            tgz_opt = '--tgz-directory ' + tgz_path

        with environments.VirtualEnvironment() as venv:
            venv.pip('pexpect')
            venv.pip('ptyprocess')
            cmd = "cd {path}/utils/release && {ex} ./push_packages "\
                  "--gpg-key-user '{key_user}' --deb-directory {deb_path} {rpm_opt} {tgz_opt} "\
                  "--repo-url {url} --repo-user {user} --gpg-passphrase \"{passphrase}\" --release-type {reltype}".format(
                      ex=venv.executable,
                      path=repo_path, deb_path=deb_path, rpm_opt=rpm_opt, tgz_opt=tgz_opt,
                      url=repo_url, user=REPOS_TO_UPLOAD[repo_url], passphrase=passphrase,
                      reltype=release_type, key_user=gpg_user)
            logging.info("Will execute cmd '%s'", cmd)
            result = run_process(cmd, shell=True, log_prefix=log_name)
            logging.info("Execute finished")
            retcode = result.returncode
            if retcode != 0:
                logging.info("Run failed with code %s", retcode)
            return log_name, retcode

    def push_packages(self, deb_path, repo_path, remote_repos, rpm_path='', tgz_path=''):
        result = []
        if self.Parameters.release_type in ('lts', 'stable'):  # don't push testing/prestable packages in yandex repos
            with GpgKey2(self.task_owner, self.Parameters.gpg_priv_key_vault, self.Parameters.gpg_pub_key_vault):
                with SSHKey(self, self.task_owner, self.ssh_key):
                    for repo_url in remote_repos:
                        logging.info("Pushing to repo %s", repo_url)
                        if repo_url != 'repo.mirror.yandex.net':
                            log, retcode = self.push_package(deb_path, repo_path, repo_url, self.gpg_passphrase, self.Parameters.gpg_key_user, self.Parameters.release_type)
                            logging.info("Push finised with retcode %s", retcode)
                            result.append((repo_url, retcode, log))

        with GpgKey2(self.task_owner, "repo-clickhouse-gpg-private", "repo-clickhouse-gpg-public"):
            with SSHKey(self, self.task_owner, self.ssh_key):
                release_types_to_push = [self.Parameters.release_type]
                if self.Parameters.release_type == 'lts':
                    release_types_to_push.append('stable')
                for release_type in release_types_to_push:
                    repo_url = 'repo.mirror.yandex.net'
                    logging.info("Pushing to repo %s with %s", repo_url, release_type)
                    log, retcode = self.push_package(deb_path, repo_path, repo_url, "''", "<milovidov@yandex-team.ru>", release_type, rpm_path, tgz_path)
                    logging.info("Push finised with retcode %s", retcode)
                    result.append((repo_url, retcode, log))

        return result

    def wait_packages_in_repo(self, package_files, repo_url):
        repo_name = repo_url.split('.')[0]
        logging.info("Will try to wait package for repo %s", repo_name)
        packages_dist_urls = [
            "http://dist.yandex.ru/{}/unstable/all/Packages".format(repo_name),
            "http://dist.yandex.ru/{}/unstable/amd64/Packages".format(repo_name),
        ]
        found_dct = {name: False for name in package_files}

        start = time.time()
        while time.time() - start < 120:
            for package_dist_url in packages_dist_urls:
                for line in requests.get(package_dist_url).iter_lines():
                    for name in found_dct:
                        if name in line:
                            found_dct[name] = True
            if all(found_dct.values()):
                logging.info("All packages were found")
                break
        else:
            logging.info("Some packages were not found: %s", ', '.join([name for name, v in found_dct.iteritems() if not v]))

    @retry(tries=5, delay=5)
    def create_conductor_ticket(self, version, packages):
        api = ConductorAPI(task=self, auth=ConductorOAuth(sdk2.Vault.data(self.Parameters.conductor_token_vault)))
        packages = {p: version.get_version_string() for p in packages}
        ticket_id = api.ticket_add(
            packages=packages,
            branch="testing",
        )
        logging.info("tikect id %s", ticket_id)
        return ticket_id

    @retry(tries=5, delay=5)
    def get_issue_for_version(self, st_client, version):
        query = 'Tags: {} AND Queue: ClickHouse AND Components: Release AND Status: open'.format(str(version.major) + "." + str(version.minor))
        logging.info("Will search st issues with query '%s'", query)
        issues = st_client.issues.find(query)
        if len(issues) > 0:
            return issues[0]
        return None

    @retry(tries=5, delay=5)
    def create_startrek_ticket(self, st_client, version, pull_request, token):
        version_string = str(version.major) + "." + str(version.minor)
        if pull_request.number == 0:
            pr_desc = "Release from master: https://github.com/yandex/ClickHouse"
        else:
            pr_desc = "Release from pull request {}".format(pull_request.html_url)

        return st_client.issues.create(
            queue="ClickHouse",
            summary="Release v{}".format(version_string),
            description=pr_desc,
            tags=version_string,
            components=['Release'],
            followers=get_person_logins_from_group('yandex_monetize_metrica_dev_stat', token),
        )

    @retry(tries=5, delay=1)
    def check_prestable_tag_exists(self, version, repo):
        tag_name = build_version_description(version, "prestable")
        for tag in repo.get_tags():
            if tag.name == tag_name:
                return True
        return False

    def update_docker(self, commit, pull_request):
        logging.info("Start task for docker push")
        docker_task = ClickhouseDockerHubPush(
            self,
            description="Push to docker for release",
            dockerfiles=DOCKER_PACKAGES,
            with_version=True,
            commit_sha=commit.sha,
            pull_request_number=pull_request.number,
            force_run=True,
            version_from_pr=False,
            post_statuses=False,
        )
        docker_task.enqueue()

    def process(self, commit, repo, pull_request):
        # Check that the release type is allowed for the branch.
        if pull_request.number == 0:
            if self.Parameters.release_type not in ['testing', 'prestable']:
                raise Exception("Release type '{}' is not allowed for master branch".format(self.Parameters.release_type))
        else:
            if self.Parameters.release_type not in ['prestable', 'stable', 'lts']:
                raise Exception("Release type '{}' is not allowed for release branch".format(self.Parameters.release_type))

        logging.info("Installing required package")
        self._install_debian_packages()
        logging.info("Creating release for repo %s", repo.full_name)
        self.gpg_passphrase = sdk2.Vault.data(self.Parameters.gpg_passphrase_vault)
        if self.gpg_passphrase == 'empty':
            self.gpg_passphrase = "''"
        st_token = sdk2.Vault.data(self.Parameters.startrek_token)
        import startrek_client
        self.st_client = startrek_client.Startrek(token=st_token, useragent='sandbox-task')

        remote_repo_path = os.path.join(str(self.path()), "./ClickHouseRemote")
        os.mkdir(remote_repo_path)
        githelper_remote = GitHelper(self, repo, remote_repo_path, self.task_owner, self.user_email, self.ssh_key)
        logging.info("Cloning remote repo")
        githelper_remote.clone_repo(pull_request, commit, with_key=True)
        logging.info("Cloned")

        if len(self.Parameters.packages_from_s3) > 0:
            logging.info("S3 packages not empty, pushing them")
            self.s3_push_packages(TEST_URLS, remote_repo_path)
            logging.info("Packages pushed, exiting")
            return "success", "Packages pushed from s3", ""

        source_tar = self.get_sources(commit, repo, pull_request)
        if source_tar is None:
            logging.info("Cannot find sources resource for commit %s and pr %s", commit.sha, pull_request.number)
            return "failure", "Cannot find sources for commit and pr", ""

        src_path = self._save_src(source_tar)
        logging.info("Downloaded sources with path %s", src_path)

        deb_resource = self.get_resources(commit, repo, pull_request, 'deb')
        if deb_resource is None:
            logging.info("Cannot find build resource for commit %s and pr %s", commit.sha, pull_request.number)
            return "failure", "Cannot find build for commit and pr", ""

        deb_path = self._save_build(deb_resource, 'deb')
        logging.info("Downloaded deb packages with path %s", deb_path)
        deb_names = [name for name in os.listdir(deb_path) if name.endswith('deb')]
        logging.info("Str version %s", deb_resource.version)
        major, minor, patch, tweak = (int(v) for v in deb_resource.version.split('.'))
        revision = int(deb_resource.revision)
        version = ClickHouseVersion(major, minor, patch, tweak, revision)

        publishing_version = copy.deepcopy(version)

        rpm_resource = self.get_resources(commit, repo, pull_request, 'rpm')
        if rpm_resource is not None:
            rpm_path = self._save_build(rpm_resource, 'rpm')
            logging.info("Dowloaded rpm packages with path %s", rpm_path)
        else:
            rpm_path = ''

        tgz_resource = self.get_resources(commit, repo, pull_request, 'tgz')
        if tgz_resource is not None:
            tgz_path = self._save_build(tgz_resource, 'tgz')
            logging.info("Dowloaded tgz packages with path %s", tgz_path)
        else:
            tgz_path = ''

        if self.Parameters.push_to_repos:
            logging.info("Pushing packages to repositories")
            results = self.push_packages(deb_path, remote_repo_path, self.Parameters.remote_repos, rpm_path, tgz_path)
            for repo_url, retcode, log in results:
                if retcode != 0:
                    return "failure", "Push to repo {} failed, see info in log {}".format(repo_url, log), ""
            for repo_url, _, _ in results:
                if repo_url != 'repo.mirror.yandex.net':
                    self.wait_packages_in_repo(deb_names, repo_url)
        else:
            logging.info("Push to repositories disabled manually")

        try:
            logging.info("Tagging version and pushing")
            tag_name = githelper_remote.tag_version_and_push(commit, version, self.Parameters.release_type, pull_request, self.Parameters.push_to_git, self.Parameters.release_type != "testing")
            logging.info("Version tagged")
            if self.Parameters.release_type in {"lts", "stable"}:
                logging.info("Pushing version_date.tsv update")
                githelper_remote.update_versions_list_in_master(version, True)
                logging.info("Update pushed")
        except Exception as ex:
            logging.info("Version update failure %s", str(ex))

        if self.Parameters.add_asset_to_github:
            prerelease = False if self.Parameters.release_type == "lts" or self.Parameters.release_type == "stable" else True
            logging.info("Pushing assets to GitHub, %s", "prerelease" if prerelease else "release")
            message = "ClickHouse {type} release {tag}".format(type=self.Parameters.release_type, tag=tag_name)

            gh_release = self.gh_helper.get_repo('ClickHouse/ClickHouse').create_git_release(
                tag=tag_name, name=tag_name, message=message, draft=True, prerelease=prerelease)
            logging.info("GitHub Release created")

            for deb_name in deb_names:
                logging.info("GitHub Release uploading asset {}".format(deb_name))
                gh_release.upload_asset(content_type='application/binary', name=deb_name, path=os.path.join(deb_path, deb_name), label="Linux deb package {}".format(deb_name))

            if tgz_path != '':
                tgz_names = [name for name in os.listdir(tgz_path) if name.endswith('tgz')]
                for tgz_name in tgz_names:
                    logging.info("GitHub Release uploading asset {}".format(tgz_name))
                    gh_release.upload_asset(content_type='application/binary', name=tgz_name, path=os.path.join(tgz_path, tgz_name), label="Linux tgz package {}".format(tgz_name))

            if rpm_path != '':
                rpm_names = [name for name in os.listdir(rpm_path) if name.endswith('rpm')]
                for rpm_name in rpm_names:
                    logging.info("GitHub Release uploading asset {}".format(rpm_names))
                    gh_release.upload_asset(content_type='application/binary', name=rpm_name, path=os.path.join(rpm_path, rpm_name), label="Linux rpm package {}".format(rpm_name))

            if src_path != '':
                logging.info("GitHub Release uploading asset {}".format(src_path))
                gh_release.upload_asset(content_type='application/binary', name='ClickHouse_sources_with_submodules.tar.gz', path=os.path.join(src_path), label="Source files with submodules")

            gh_release.update_release(name=tag_name, message=message, draft=False, prerelease=prerelease)

        if self.Parameters.update_startrek and self.Parameters.release_type != "testing":
            st_issue = self.get_issue_for_version(self.st_client, version)
            if st_issue is None:
                st_issue = self.create_startrek_ticket(self.st_client, version, pull_request, st_token)
            if isinstance(st_issue, bool):
                logging.info("Cannot create startrek issue")
            else:
                st_issue.comments.create(text="New version is released: {}".format(tag_name))

            if st_issue:
                self.Parameters.description += "\nStartrek ticket: https://st.yandex-team.ru/{}\n".format(st_issue.key)

        if self.Parameters.release_type != "testing":
            if pull_request.number == 0 and self.Parameters.push_to_git:
                logging.info("It's master, going to create additional branch")
                try:
                    branch_name = githelper_remote.create_version_patch_branch(version, pull_request, commit, self.Parameters.push_to_git)
                    logging.info("Additional branch %s created", branch_name)
                    branch_version = version.patch_update()

                    logging.info("Commiting new version %s to branch", branch_version.get_version_string())
                    githelper_remote.update_version_local(commit, branch_version, self.Parameters.release_type)
                    githelper_remote.commit_all(branch_version, pull_request, commit, self.Parameters.push_to_git, to_branch=branch_name)
                except Exception as ex:
                    logging.info("Cannot create patch-branch %s", str(ex))
                    return "failure", "Cannot create patch-branch {}".format(str(ex)), ""
                version = version.minor_update()
            else:
                version = version.patch_update()

            githelper_remote.checkout_branch(pull_request)
            logging.info("Updating local version")
            githelper_remote.update_version_local(commit, version, self.Parameters.release_type)
            logging.info("Update finished")

            logging.info("Commiting new version %s to repo", version.get_version_string())
            githelper_remote.commit_all(version, pull_request, commit, self.Parameters.push_to_git)

            if pull_request.number == 0 and self.Parameters.push_to_git:
                logging.info("Will create pull request for new branch")
                pull = repo.create_pull("Release pull request for branch {}".format(branch_name), PR_BODY_TEXT,  "master", branch_name, True)
                pull.add_to_labels(RELEASE_LABEL)
            else:
                pull = None

            packages = ["clickhouse-common-static", "clickhouse-client", "clickhouse-common-static-dbg", "clickhouse-server"]
            conductor_id = self.create_conductor_ticket(publishing_version, packages)
            self.Parameters.description += "\nConductor ticket: {}\n".format(conductor_id)
            if self.Parameters.update_startrek and st_issue:
                if pull:
                    st_issue.comments.create(text="Release pull request is {}".format(pull.html_url))
                st_issue.comments.create(text="Created conductor ticket {}".format(conductor_id))

        if self.Parameters.release_type == "stable" or self.Parameters.release_type == "lts":  # stable versions are pushed to dockerhub
            if self.Parameters.push_to_dockerhub:
                try:
                    self.update_docker(commit, pull_request)
                except Exception as ex:
                    logging.info("Exception while pushing %s", ex)
                    return "failure", "Push to dockerhub failed", ""
            else:
                logging.info("Push to dockerhub manually disabed")

        return "success", self.Parameters.release_type + ": " + publishing_version.get_version_string(), "https://repo.yandex.ru/clickhouse/deb/{}/main/".format(self.Parameters.release_type)
