# -*- coding: utf-8 -*-
import logging
from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.clickhouse.BaseOnCommitTask.base import NeedToRunDescription
from sandbox.projects.clickhouse.BaseOnCommitTask.simple_test_task import SimpleDockerBuildTestTask
from sandbox.projects.clickhouse.util.task_helper import get_ci_config

from sandbox.projects.clickhouse.resources import CLICKHOUSE_BUILD


class ClickhouseSanitizerCoverage(SimpleDockerBuildTestTask):

    class Parameters(SimpleDockerBuildTestTask.Parameters):
        kill_timeout = 4 * 60 * 60

    class Requirements(SimpleDockerBuildTestTask.Requirements):
        cores = 56

    @staticmethod
    def need_to_run(pr_info):
        if pr_info.number != 20539:
            return NeedToRunDescription(False, "Task work in experimental mode", False)

        return NeedToRunDescription(True)

    @staticmethod
    def required_build_properties():
        return ('clang-11', 'deb', 'debug', 'none', 'bundled', 'unsplitted', 'enable', True)

    @staticmethod
    def get_context_name():
        return "Code coverage (sanitizer)"

    @classmethod
    def get_resources(cls, commit, repo, pull_request):
        ci_config = get_ci_config(pull_request, commit)
        if not ci_config or cls.get_context_name() not in ci_config["tests_config"]:
            logging.info("Build config not found :(, will use default from params")
            compiler, package_type, build_type, sanitizer, bundled, splitted, _, with_coverage = cls.required_build_properties()
        else:
            logging.info("Build config found, will take info from repository")
            tests_config = ci_config["tests_config"]
            test_config = tests_config[cls.get_context_name()]["required_build_properties"]
            compiler = test_config["compiler"]
            package_type = test_config["package_type"]
            build_type = test_config["build_type"]
            sanitizer = test_config["sanitizer"]
            bundled = test_config["bundled"]
            splitted = test_config["splitted"]
            with_coverage = test_config["with_coverage"]

        bresources = sdk2.Resource.find(
            CLICKHOUSE_BUILD,
            attrs=dict(
                commit=commit.sha,
                pr_number=pull_request.number,
                package_type=package_type,
                build_type=build_type,
                sanitizer=sanitizer,
                bundled=bundled,
                compiler=compiler,
                splitted=splitted,
                with_coverage=with_coverage,
            ),
            state=ctr.State.READY
        ).order(-CLICKHOUSE_BUILD.id).limit(1)
        logging.info("Search finished")

        build_resource = bresources.first()
        if not build_resource:
            return None

        return build_resource

    def _save_resources(self, commit, repo, pull_request):
        build_resource = self.get_resources(commit, repo, pull_request)
        if not self.build_path:
            logging.info("Downloading CLICKHOUSE_BUILD resource")
            build_data = sdk2.ResourceData(build_resource)
            self.build_path = str(build_data.path)  # deb package
            logging.info("Download finished, build path %s", self.build_path)
        else:
            logging.info("Resource already downloaded %s", self.build_path)

        return self.build_path

    @staticmethod
    def get_images_names():
        return ["yandex/clickhouse-test-coverage"]

    def get_run_cmd(self, build_path, result_folder, server_log, repo_path, perfraw_path):
        return "docker run --volume={build_path}:/package_folder \
            --volume={repo_path}:/build \
            --volume={result_folder}:/output \
            {image}".format(build_path=build_path, repo_path=repo_path, result_folder=result_folder, image=self.get_single_image_with_version())

    def process_result(self, result_folder, server_log_path, perfraw_path, commit, repo, pull_request):
        s3_path_prefix_commit = str(pull_request.number) + "/" + commit.sha + "/sanitizer_coverage_report"
        logging.info("Uploading to commit directory")
        html_urls = self.s3_client.upload_test_folder_to_s3(result_folder, s3_path_prefix_commit)
        index_html = None
        for url in html_urls:
            if 'test_output/index.html' in url:
                index_html = '<a href="{}">HTML report</a>'.format(url)
                break
        else:
            index_html = html_urls[0]
            logging.info("Cannot find html report in result urls: " + ', '.join(html_urls))

        description = "Coverage prepared"
        test_results = [(index_html, "Report ready")]

        return "success", description, test_results, []
