# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common.types import resource as ctr
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
import datetime
import logging
import sandbox.common.types.misc as ctm
import sandbox.projects.common.binary_task as binary_task
import subprocess
import time
import dateutil
import pytz

from sandbox.projects.clickhouse.ClickhouseCoverage import ClickhouseCoverage
from sandbox.projects.clickhouse.ClickhouseDockerHubPush import ClickhouseDockerHubPush
from sandbox.projects.clickhouse.ClickhouseDocsCheck import ClickhouseDocsCheck
from sandbox.projects.clickhouse.ClickhouseFunctionalStatelessTest import ClickhouseStatelessFunctionalTestDebug
from sandbox.projects.clickhouse.ClickhouseFuzzer import ClickhouseFuzzerDebug
from sandbox.projects.clickhouse.ClickhouseKeeperJepsen import ClickhouseKeeperJepsen
from sandbox.projects.clickhouse.ClickhousePerformanceComparison import ClickhousePerformanceComparison
from sandbox.projects.clickhouse.ClickhousePullRequestTrigger import MARKER_CHECK_TEXT
from sandbox.projects.clickhouse.ClickhouseRelease import ClickhouseRelease
# from sandbox.projects.clickhouse.ClickhouseSqlancerTest import ClickhouseSqlancerTest
from sandbox.projects.clickhouse.ClickhouseWoboqBrowser import ClickhouseWoboqBuild
from sandbox.projects.clickhouse.util.github_helper import GithubHelper
from sandbox.projects.clickhouse.util.pr_info import PRInfo
from sandbox.projects.clickhouse.util.task_helper import get_nda_task_url


TIMER_TASKS = {
    ClickhouseWoboqBuild.get_context_name(): (ClickhouseWoboqBuild, 86400),
    ClickhouseCoverage.get_context_name(): (ClickhouseCoverage, 86400),
    ClickhouseKeeperJepsen.get_context_name(): (ClickhouseKeeperJepsen, 86400),
}

CHECKS_TO_IGNORE = [
    ClickhouseDockerHubPush.get_context_name(),
    ClickhouseDocsCheck.get_context_name(),
    ClickhousePerformanceComparison.get_context_name(),  # Performance test is slow and unreliable - not a criteria for testing release.
    ClickhouseFuzzerDebug.get_context_name(),  # Fuzzer usually finds old issues and it is not 100% clean today.
    'Integration tests (thread)',  # Soft timeouts in 21.4.
    MARKER_CHECK_TEXT,
    ClickhouseWoboqBuild.get_context_name(),
    ClickhouseCoverage.get_context_name(),
    ClickhouseKeeperJepsen.get_context_name(),
    # ClickhouseSqlancerTest.get_context_name(),
]


class ClickhouseTimerTrigger(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Task polling github api and run subtasks for commits by timer
    """
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("GitHub parameters") as gh_block:
            github_token_key = sdk2.parameters.String("Token for github interactions", default='clickhouse-poll-robot-github-token3')
            github_repo = sdk2.parameters.String("GitHub repo name", default='ClickHouse/ClickHouse')

        with sdk2.parameters.Group("S3 parameters") as s3_block:
            s3_access_key_id = sdk2.parameters.String("S3 key id", default='clickhouse-robot-s3-key-id')
            s3_access_key = sdk2.parameters.String("S3 key", default='clickhouse-robot-s3-key')

        with sdk2.parameters.Group("Other parameters") as other_block:
            ignore_existing = sdk2.parameters.Bool("Ignore already check commit in time")
            ext_params = binary_task.binary_release_parameters(stable=True)
            periodic_release = sdk2.parameters.Bool("Run periodic release for supported branches", default=False)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            # A test of old behavior for 'none' release type.
            # https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/clickhouse/ClickhousePullRequestTrigger/__init__.py?rev=r5801457
            # Match version to contrib: https://a.yandex-team.ru/arc/trunk/arcadia/contrib/python/PyGithub/github
            environments.PipEnvironment('pygithub', version='1.45'),
        )
        dns = ctm.DnsType.DNS64
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "CLICKHOUSE_SANDBOX_BINARY", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def has_failures(self, statuses):
        for status in statuses:
            if status.state == "failure" or status.state == "exception":
                return status.context
        return ""

    def filter_ignored_checks(self, statuses, ignored_checks):
        # Yandex synchronization checks are not of our interest.
        return [status for status in statuses if (status.context not in ignored_checks) and (not status.context.startswith('Yandex'))]

    def green_count(self, statuses):
        return sum(1 for status in statuses if status.state == "success")

    def get_last_tag_release_type(self, branch):
        logging.info("Looking for previous release for branch '%s'", branch)

        # Fetch the tags again -- maybe the previous release task published some
        # relevant release, e.g. if we published 'prestable', it doesn't make sense
        # to publish 'testing' at the same commit.
        subprocess.check_call("cd ch && git fetch --tags", shell=True)
        subprocess.check_call("cd ch && git fetch origin {}".format(branch), shell=True)

        prev_release = str(subprocess.check_output("cd ch && git describe --abbrev=0 --match='v*-*' origin/{}".format(branch), shell=True)).rstrip()
        if not prev_release:
            raise Exception("Previous release is not found for branch '{}'".format(branch))

        known_release_types = ['testing', 'prestable', 'stable', 'lts']
        for x in known_release_types:
            if prev_release.endswith("-" + x):
                logging.info("The previous release '%s' for branch '%s' is '%s'", prev_release, branch, x)
                return x

        raise Exception("The tag '{}' for branch '{}' does not match the known release types '{}'".format(prev_release, branch, str(known_release_types)))

    def periodic_release(self, repo, pr_number, branch, release_type):
        logging.info("Looking for periodic '%s' release candidate for branch '%s'", release_type, branch)

        # Fetch the tags again -- maybe the previous release task published some
        # relevant release, e.g. if we published 'prestable', it doesn't make sense
        # to publish 'testing' at the same commit.
        subprocess.check_call("cd ch && git fetch --tags", shell=True)

        # Find the release PR on GitHub.
        if pr_number != 0:
            pygithub_pr = repo.get_pull(pr_number)
            pr_head_branch = pygithub_pr.raw_data['head']['ref']
        else:
            pygithub_pr = self.gh_helper.get_fake_pr()
            pr_head_branch = 'master'
        # Sanity check. Should switch to just using the branch and determining the
        # PR from it.
        if pr_head_branch != branch:
            raise Exception("Base branch for pull request '{}': '{}' doesn't match the branch we want to release: '{}'".format(
                pr_number, pr_head_branch, branch))

        subprocess.check_call("cd ch && git fetch origin {}".format(branch), shell=True)

        # Another sanity check. In a branch, first we can have testing releases,
        # then prestable, then stable, then lts. A testing release can be preceded
        # by prestable, if it is the first one after making a prestable branch
        # (because the first prestable tag points to commit also visible from master).
        # We can also have multiple consecutive releases of the same kind.
        # Any other sequence is suspicious.
        # Releasing from prestable to stable requires manual confirmation,
        # because the "release checklist" issue must be completed before.
        allowed_prev_release_types = {
            'testing': ['testing', 'prestable'],
            'prestable': ['testing', 'prestable'],
            'stable': ['stable'],
            'lts': ['stable', 'lts']
        }

        prev_release = str(subprocess.check_output("cd ch && git describe --abbrev=0 --match='v*-*' origin/{}".format(branch), shell=True)).rstrip()
        if not prev_release:
            raise Exception("Previous release is not found for branch '{}'".format(branch))

        if not any([prev_release.endswith("-" + x) for x in allowed_prev_release_types[release_type]]):
            # Not a hard error. Can happen if we released something wrong manually,
            # or for a prestable branch -- the prestable -> stable transition
            # is only done manually.
            logging.info(
                "Previous release '%s' does not match allowed previous release types '%s'. Will not release automatically",
                prev_release, allowed_prev_release_types[release_type])
            return None

        # Next, find the commits since the previous release. Different rules for
        # master and release branches.
        if pr_number != 0:
            # For release branches, try to release something older than the last
            # release.
            closest_tag = str(subprocess.check_output("cd ch && git describe --abbrev=0 --match='v*-*' origin/{}".format(branch), shell=True)).rstrip()
            logging.info("Closest tag for branch '%s' is '%s'", branch, closest_tag)
            if not closest_tag:
                raise Exception("Failed to find closest tag for branch '%s'", branch)

            # We want to use the tag date here, because it can be significantly
            # later than the commit date. For example, if we used commit date,
            # we'd release testing releases twice daily instead of daily, because
            # the checks for a commit take half a day to complete.
            # This could have been as simple as 'git tag --format="%(taggerdate:unix)"',
            # but git in Sandbox is ancient and doesn't support this.
            # The newer git version shows the date like '1621462514 +0300',
            # and the older one like 'Fri May 21 16:22:03 2021 +0300'. The former
            # can't be parsed by dateutil parser, so this will break when Sandbox
            # updates git.
            release_time_str = str(subprocess.check_output('cd ch && git cat-file -p $(git rev-parse "{}") | sed -n "s/^tagger.*> \\(.\\+$\\)/\\1/p"'.format(closest_tag), shell=True)).strip()
            release_time = dateutil.parser.parse(release_time_str)
            days = (datetime.datetime.utcnow().replace(tzinfo=pytz.utc) - release_time).days
            logging.info("Date of previous release '%s' is '%s', '%s' days ago", closest_tag, release_time, days)

            if days < 7:
                logging.info("Will not release because the last release is still too recent")
                return None

            # Probably it's best to only release the last commit to avoid confusion.
            # Else we could release non-last commit multiple times, adding version
            # update commits to branch and releasing from the middle of it. Only the
            # last number of release would change (21.4.3.{10,20,15000}), and then
            # it would suddenly jump to 21.4.7.something. This would look weird.
            # I'm keeping a loop here because I'm still undecided on this.
            new_commits = str(subprocess.check_output('set -x && cd ch && git rev-list -1 --first-parent "^{}" origin/{}'.format(closest_tag, branch), shell=True)).rstrip().splitlines()
            logging.info("Commits after closest tag '%s' in branch '%s': '%s'", closest_tag, branch, new_commits)
        else:
            # We are working with master branch. We make daily testing releases,
            # and split off a new prestable branch every two weeks.
            if branch != 'master':
                raise Exception("Wrong branch '%s' specified for PR #0, must be 'master'", branch)

            if release_type not in ('prestable', 'testing'):
                raise Exception("Unsupported release type '%s' specified for master branch. Only 'prestable' and 'testing' is supported", release_type)

            if release_type == 'prestable':
                # Make a new prestable branch from master. We only try to make a
                # new prestable branch when there are none of them. The prestable
                # branches are transitioned to stable manually, so we don't
                # really have to have any time-based logic here. Require at least
                # a week of delay as a sanity check.
                min_days = 7
                release_filter = '--match="*-prestable"'
            else:
                # Make a new testing release every day. A prestable release also
                # counts, and we shouldn't have other release types on master.
                min_days = 1
                release_filter = '--match="*-prestable" --match="*-testing"'

            # Check when we last did a prestable/testing release.
            closest_tag = str(subprocess.check_output('cd ch && git describe {} --abbrev=0 origin/master'.format(release_filter), shell=True)).rstrip()
            logging.info("Closest tag for branch '%s' is '%s'", branch, closest_tag)
            if not closest_tag:
                raise Exception("Failed to find closest tag for branch '%s'", branch)

            # See the comment for the same code above.
            release_time_str = str(subprocess.check_output('cd ch && git cat-file -p $(git rev-parse "{}") | sed -n "s/^tagger.*> \\(.\\+$\\)/\\1/p"'.format(closest_tag), shell=True)).strip()
            release_time = dateutil.parser.parse(release_time_str)
            days = (datetime.datetime.utcnow().replace(tzinfo=pytz.utc) - release_time).days
            logging.info("Date of previous release '%s' is '%s', '%s' days ago", closest_tag, release_time, days)

            if days < min_days:
                logging.info("Will not release to '%s' because the last release is still too recent (%s < %s days)", release_type, days, min_days)
                return None

            # It probably doesn't make sense to make a prestable or testing
            # release that is too far away in the past. Let's decide that not
            # having green commits for a day is "normal" and can happen, so we
            # probably should find something releasable in a last couple of days.
            # The "^closest_tag" filter is mostly a sanity check.
            new_commits = str(subprocess.check_output('set -x && cd ch && git rev-list --first-parent --after=2.days.ago "^{}" origin/{}'.format(
                closest_tag, branch), shell=True)).rstrip().splitlines()
            logging.info("Commits after closest tag '%s' in branch '%s': '%s'", closest_tag, branch, new_commits)

        for sha in new_commits:
            pr_info = PRInfo(pygithub_pr, repo.get_commit(sha))

            significant_checks = 0
            checks_by_state = {'failure': [], 'error': [], 'success': [], 'pending': [], 'ignored': []}
            for check, status in pr_info.statuses.items():
                if check in CHECKS_TO_IGNORE or check.startswith('Yandex') or check.startswith('AST '):
                    checks_by_state['ignored'].append(check)
                else:
                    checks_by_state[status.state].append(check)
                    significant_checks += 1

            logging.info("Checks for commit %s: %s significant", sha, significant_checks)
            for state, checks in sorted(checks_by_state.items(), key=lambda x: x[0]):
                logging.info("%s: %s: %s", sha, state, checks)

            if len(checks_by_state['success']) != significant_checks:
                logging.info("Commit %s has some significant checks that are not successful", sha)
                continue

            # Require Marker Check as a sign that all checks were started.
            if MARKER_CHECK_TEXT not in pr_info.statuses:
                logging.info("Commit '%s' does not have the marker check '%s'", sha, MARKER_CHECK_TEXT)
                continue

            # We check that it's not failed separately because for some reason it
            # is in the list of ignored checks.
            marker_check_status = pr_info.statuses[MARKER_CHECK_TEXT]
            if marker_check_status.state != 'success':
                logging.info("Marker check state for commit '%s' is '%s', not 'success'", sha, marker_check_status.state)
                continue

            # As a sanity check, also verify that the functional stateless tests
            # have finished. Don't have to check its status, because we now know
            # that it is succeeded, checked above.
            ft_name = ClickhouseStatelessFunctionalTestDebug.get_context_name()
            if ft_name not in pr_info.statuses:
                logging.info("No '%s' check for commit '%s'", ft_name, sha)
                continue
            assert(pr_info.statuses[ft_name].state == 'success')

            # Ignore commits with auto version update by robot-clickhouse{1234},
            # or we would endlessly publish releases differing only by version.
            git_commit = pr_info.pygithub_commit.commit
            if git_commit.author.name.startswith('robot-clickhouse'):
                logging.info("Ignoring commit '%s' authored by robot '%s'", sha, git_commit.author.name)
                continue
            if git_commit.message.startswith('Auto version update'):
                logging.info("Ignoring auto version update commit '%s' with message '%s'", sha, git_commit.message)
                continue

            # Try to get the release task resources for this commit.
            if not ClickhouseRelease.get_resources(pr_info.pygithub_commit, repo, pr_info.pygithub_pr):
                logging.info("Cannot get task resources for commit '%s'", sha)
                continue

            # All conditions fulfilled, start the task.
            logging.info("Will release commit '%s'", sha)
            task = ClickhouseRelease(
                self,
                description="Child of ClickhouseTimerTrigger task {}".format(self.id),
                github_token_key=self.Parameters.github_token_key,
                github_repo=self.Parameters.github_repo,
                s3_access_key_id=self.Parameters.s3_access_key_id,
                s3_access_key=self.Parameters.s3_access_key,
                commit_sha=sha,
                pull_request_number=pr_number,
                release_type=release_type,
                priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.LOW)
            )
            task.enqueue()
            return task

    def on_execute(self):
        venv = environments.VirtualEnvironment()

        with venv as venv:
            if self.Parameters.binary_executor_release_type == "none":
                # https://github.com/yandex/yandex-tank/issues/845
                # https://sandbox.yandex-team.ru/docs/html/sandboxsdk.html?highlight=pipenvironment#sandboxsdk.environments.PipEnvironment
                environments.PipEnvironment('pip', version='9.0.3', venv=venv).prepare()
                environments.PipEnvironment('setuptools', version='20.7.0', venv=venv).prepare()
                environments.PipEnvironment('pygithub', version='1.45', venv=venv).prepare()
                environments.PipEnvironment('boto3', venv=venv).prepare()
                logging.debug("Set pip environment for nonbinary release")

            logging.getLogger().setLevel(logging.INFO)
            github_token = sdk2.Vault.data(self.Parameters.github_token_key)

            self.gh_helper = GithubHelper(github_token)

            rate_is_ok, message = self.gh_helper.check_rate_limit()
            if not rate_is_ok:
                self.Parameters.description = "[can't run :(] " + message
                return

            repo = self.gh_helper.get_repo(self.Parameters.github_repo)

            if self.Parameters.periodic_release:
                subprocess.check_call("git --version", shell=True)
                subprocess.check_call("git clone https://github.com/ClickHouse/ClickHouse ch", shell=True)

                # We will build an array of Release objects describing which
                # release branches we have, determine what we want to release,
                # and perform some sanity checks.
                class Release:
                    def __init__(self, release_branch, release_pr_number):
                        self.release_branch = release_branch
                        self.release_pr_number = release_pr_number
                        self.existing_release_type = ''
                        self.next_release_type = ''

                    def __str__(self):
                        return "release PR '{}' branch '{}' existing '{}' next '{}'".format(self.release_pr_number, self.release_branch, self.existing_release_type, self.next_release_type)

                    def __repr__(self):
                        return self.__str__()

                # First, the master branch is always released to testing.
                releases = [Release('master', 0)]
                releases[0].existing_release_type = self.get_last_tag_release_type('master')
                if releases[0].existing_release_type not in ['testing', 'prestable']:
                    # We might see 'prestable' tag on master if we just split off
                    # a new prestable branch. 'testing' is normal, and others
                    # probably signify an error.
                    raise Exception("Unexpected existing release type '{}' for 'master'".format(releases[0].existing_release_type))
                releases[0].next_release_type = 'testing'

                # Next, determine the list of supported branches from the release PRs.
                release_prs = self.gh_helper.gh.search_issues(query='type:pr repo:ClickHouse/ClickHouse is:open label:release')
                lts_pr_numbers = set([pr.number for pr in self.gh_helper.gh.search_issues(query='type:pr repo:ClickHouse/ClickHouse is:open label:release-lts')])
                for pr in sorted(release_prs, key=lambda x: x.created_at, reverse=True):
                    release_branch = repo.get_pull(pr.number).raw_data['head']['ref']
                    release = Release(release_branch, pr.number)
                    release.existing_release_type = self.get_last_tag_release_type(release_branch)
                    # The transition from 'prestable' to 'lts' must be performed manually.
                    release.next_release_type = 'lts' if (pr.number in lts_pr_numbers and release.existing_release_type in ['stable', 'lts']) else release.existing_release_type
                    # The last release in a back branch should be at least
                    # 'prestable' that is released when the branch is created. A
                    # 'testing' indicates error.
                    if release.existing_release_type not in ['prestable', 'stable', 'lts']:
                        raise Exception("Unexpected existing release type '{}' for PR #{}".format(release.existing_release_type, pr.number))
                    # Testing is only for master which doesn't have a release PR.
                    # This code runs for branches, that should not be released as
                    # testing.
                    if release.next_release_type not in ['prestable', 'stable', 'lts']:
                        raise Exception("Unexpected next release type '{}' for PR #{}".format(release.next_release_type, pr.number))
                    releases.append(release)
                logging.info("Releases: '%s'", releases)

                # First of all, if there is no prestable branch, try to make a
                # new one. We do this even before publishing a testing release,
                # because there is no point in releasing a testing right after a
                # prestable.
                if not any([x.next_release_type == 'prestable' for x in releases]):
                    task = self.periodic_release(repo, 0, 'master', 'prestable')
                    if not task:
                        logging.info("Nothing to release for branch 'master' type 'prestable'")
                    else:
                        while not (task.status in ctt.Status.Group.SUCCEED or task.status in ctt.Status.Group.BREAK):
                            logging.info("Waiting for task id %s with status %s", task.id, task.status)
                            time.sleep(120)
                            task.reload()

                # Next, try to release something for all the release branches we found.
                released_pr_numbers = set()
                for x in releases:
                    task = self.periodic_release(repo, x.release_pr_number, x.release_branch, x.next_release_type)
                    if not task:
                        logging.info("Nothing to release for branch '%s' type '%s'", x.release_branch, x.next_release_type)
                        continue
                    while not (task.status in ctt.Status.Group.SUCCEED or task.status in ctt.Status.Group.BREAK):
                        logging.info("Waiting for task id %s with status %s", task.id, task.status)
                        time.sleep(120)
                        task.reload()
                    if task.status in ctt.Status.Group.SUCCEED:
                        released_pr_numbers.add(x.release_pr_number)

                # Close the older PRs. We want to have one prestable and two
                # stable branches, not counting the LTS ones.
                # We don't close the PR if we didn't release anything in this
                # run. Probably something was added to the branch since the last
                # release, but we were unable to release it now because the CI
                # was red or the last release was too recent or something. These
                # new commits are important, so we should try later. The branch
                # might live longer than it should, but it's not a big deal.
                # Not closing w/o a release also serves as a protection against
                # annoingly closing a manually reopened release PR.
                # Note that we have to enumerate the release PRs again here,
                # because we might have opened a new prestable PR, and we want
                # to account for it as well.
                release_prs = self.gh_helper.gh.search_issues(query='type:pr repo:ClickHouse/ClickHouse is:open label:release')
                no_lts = [repo.get_pull(pr.number) for pr in release_prs if pr.number not in lts_pr_numbers]
                to_close = sorted(no_lts, key=lambda x: x.created_at, reverse=True)[3:]
                for pr in to_close:
                    if pr.number in released_pr_numbers:
                        logging.info("Closing old release PR %s because we just made a new release for it", pr.number)
                        pr.edit(state="closed")
                    else:
                        logging.info("Not closing old release PR %s because we were unable to make a release for it", pr.number)

            # Run other periodic checks.
            fake_pr = self.gh_helper.get_fake_pr()
            for check_context, (TaskType, task_period) in TIMER_TASKS.iteritems():
                logging.info("Processing check %s", check_context)
                date_from = datetime.datetime.now() - datetime.timedelta(seconds=task_period)
                commits = list(sorted(repo.get_commits(since=date_from), key=lambda x: x.commit.author.date))
                str_date = date_from.strftime("%Y-%m-%d %H:%M:%S")
                best_commit = None
                best_statuses_count = 0
                already_checked = False
                for commit in commits:
                    total_statuses = self.gh_helper.get_filtered_statuses(commit)
                    statuses = self.filter_ignored_checks(total_statuses, CHECKS_TO_IGNORE)
                    logging.info("%s: total statuses %s, filtered %s", commit.sha, len(total_statuses), len(statuses))
                    if self.gh_helper.is_statuses_checked(total_statuses, check_context, datetime.timedelta(seconds=task_period)):
                        logging.info("Has already processed commit %s since %s by check %s", commit.sha, str_date, check_context)
                        already_checked = True
                        if not self.Parameters.ignore_existing:
                            break
                    else:
                        check_failures = self.has_failures(statuses)
                        if check_failures:
                            logging.info("Commit %s failured on %s", commit.sha, check_failures)
                            continue
                        gc = self.green_count(statuses)
                        if gc > best_statuses_count:
                            best_statuses_count = gc
                            best_commit = commit
                if already_checked and not self.Parameters.ignore_existing:
                    continue
                elif not best_commit:
                    logging.info("All %s commits have failures, nothing to release", len(commits))
                elif best_statuses_count < 10:
                    logging.info("Best commit %s has %s < 10 statuses, nothing to release", best_commit.sha, best_statuses_count)
                else:
                    logging.info("Will run on %s commit", best_commit.sha)
                    if TaskType.get_resources(best_commit, repo, fake_pr):
                        task = TaskType(
                            self,
                            description="Child of ClickhouseTimerTrigger task {}".format(self.id),
                            github_token_key="clickhouse-robot-github-token",
                            github_repo=self.Parameters.github_repo,
                            s3_access_key_id=self.Parameters.s3_access_key_id,
                            s3_access_key=self.Parameters.s3_access_key,
                            commit_sha=best_commit.sha,
                            priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.LOW)
                        )
                        task.enqueue()
                        task.update_status(text="Started", url=get_nda_task_url(task.id), commit=best_commit)
                    else:
                        self.Parameters.description += "\n Will not run task {} for commit {}".format(check_context, commit.sha)
                        logging.info("Task dependencies are not satisfied")
