# -*- coding: utf-8 -*-
import json
import requests
import time
import logging


class ClickHouseHelper(object):
    def __init__(self, url, user, password):
        self.url = url
        self.auth = {
            'X-ClickHouse-User': user,
            'X-ClickHouse-Key': password
        }

    def _insert_json_str_info(self, db, table, json_str):
        params = {
            'database': db,
            'query': 'INSERT INTO {table} FORMAT JSONEachRow'.format(table=table),
            'date_time_input_format': 'best_effort',
            'send_logs_level': 'warning',
        }

        for i in range(5):
            response = requests.post(self.url, params=params, data=json_str, headers=self.auth, verify=False)

            logging.info("Response content '%s'", response.content)

            if response.ok:
                break

            error = (
                "Cannot insert data into clickhouse at try " + str(i)
                + ": HTTP code " + str(response.status_code) + ": '"
                + str(response.text) + "'")

            if response.status_code >= 500:
                # A retriable error
                time.sleep(1)
                continue

            logging.info("Request headers '%s', body '%s'", response.request.headers, response.request.body)

            raise Exception(error)
        else:
            raise Exception(error)

    def insert_event_into(self, db, table, event):
        event_str = json.dumps(event)
        self._insert_json_str_info(db, table, event_str)

    def insert_events_into(self, db, table, events):
        jsons = []
        for event in events:
            jsons.append(json.dumps(event))

        self._insert_json_str_info(db, table, ','.join(jsons))

    def _select_and_get_json_each_row(self, db, query):
        params = {
            'database': db,
            'query': query,
            'default_format': 'JSONEachRow',
        }
        for i in range(5):
            response = None
            try:
                response = requests.get(self.url, params=params, headers=self.auth, verify=False)
                response.raise_for_status()
                return response.text
            except Exception as ex:
                logging.warn("Cannot insert with exception %s", str(ex))
                if response:
                    logging.warn("Reponse text %s", response.text)
                time.sleep(0.1)
        else:
            raise Exception("Cannot insert data into clickhouse")

    def select_json_each_row(self, db, query):
        text = self._select_and_get_json_each_row(db, query)
        result = []
        for line in text.split('\n'):
            if line:
                result.append(json.loads(line))
        return result
