# -*- coding: utf-8 -*-
import os
import logging
import subprocess
import time
import shutil
import random
import json
from collections import defaultdict

from sandbox.projects.clickhouse.util.task_helper import compress_fast
import sandbox.sdk2.helpers
from sandbox.projects.clickhouse.util.pr_info import PRInfo


# Do not modify code in this file. It's append-only and
# it exists for compatibility with old branches. It can be removed after 21.3 EOL.
# Use scripts from GitHub repository instead.
# See also: BaseOnCommitTask.process_result_simple(...) and ClickHouse/tests/integration/ci-runner.py

class StressTestDeprecated:
    SIGNAL_KILL_SIGN = "########################################"
    SANITIZER_SIGN1 = "=================="
    SANITIZER_SIGN2 = "WARNING"
    IGNORE_SIGN_LIST = ["ASan doesn't fully support makecontext/swapcontext functions"]

    @staticmethod
    def check_sign(task, result_log_path, sign):
        with open(result_log_path, 'r') as result_log:
            for line in result_log:
                if sign in line:
                    if not [ignore_sign for ignore_sign in StressTestDeprecated.IGNORE_SIGN_LIST if ignore_sign in line]:
                        return True
        return False

    @staticmethod
    def process_result(task, result_folder, server_log_folder, perfraw_path, commit, repo, pull_request):
        state = "success"
        description = "No errors found"
        test_results = []
        alive_check_path = os.path.join(result_folder, "alive_check.txt")
        if os.path.exists(alive_check_path):
            with open(alive_check_path, 'r') as alive_check_file:
                line = alive_check_file.readline()
                if 'Server failed to start' in line:
                    description = 'Server failed to start'
                    state = "failure"
                    test_results.append(('Server failed to start', 'FAIL'))
                else:
                    test_results.append(('Server successfuly started', 'OK'))
        else:
            test_results.append(('No aliveness check', 'SKIPPED'))

        test_files = []
        if os.path.exists(result_folder):
            test_files = [f for f in os.listdir(result_folder) if os.path.isfile(os.path.join(result_folder, f))]

        server_log_path = os.path.join(server_log_folder, "clickhouse-server.log")
        stderr_log_path = os.path.join(server_log_folder, "stderr.log")
        logging.info("Will search for signals in server log %s", server_log_path)
        logging.info("Will search for sanitizer asserts in stderr log %s", stderr_log_path)
        additional_files = []
        additional_files += [os.path.join(result_folder, f) for f in test_files]
        if not os.path.exists(server_log_path) or not os.path.exists(stderr_log_path):
            # Don't throw an exception -- this will restart the task, but the
            # failure to start w/o logs is probably persistent -- e.g. we can't
            # open some shared library.
            state = "failure"
            test_results.append(("Path with logs doesn't exist: {} or {}".format(server_log_path, stderr_log_path), 'FAIL'))
        else:
            additional_files += [server_log_path, stderr_log_path]
            signal_in_server_log = StressTestDeprecated.check_sign(task, server_log_path, StressTestDeprecated.SIGNAL_KILL_SIGN)
            sanitizer_assert = StressTestDeprecated.check_sign(task, stderr_log_path, StressTestDeprecated.SANITIZER_SIGN1) or \
                               StressTestDeprecated.check_sign(task, stderr_log_path, StressTestDeprecated.SANITIZER_SIGN2)
            if sanitizer_assert:
                logging.info("Found sanitizer assert in stderr log: %s", stderr_log_path)
                description = "Sanitizer assert (in stderr.log)"
                state = "failure"
            if signal_in_server_log:
                logging.info("Found kill by signal in server log: %s", server_log_path)
                description = "Killed by signal (in clickhouse-server.log)"
                state = "failure"

        status = "OK"

        for file in test_files:
            if StressTestDeprecated.check_sign(task, os.path.join(result_folder, file), StressTestDeprecated.SIGNAL_KILL_SIGN):
                status = "FAIL"
                break

        test_results.append(('stress_test_run', status))
        test_results.append(('exit_signal', "FAIL" if signal_in_server_log or sanitizer_assert else "OK"))

        return state, description, test_results, additional_files


class FunctionalTestDeprecated:

    @staticmethod
    def process_test_log(log_path):
        OK_SIGN = "[ OK "
        FAIL_SING = "[ FAIL "
        TIMEOUT_SING = "[ Timeout! "
        UNKNOWN_SIGN = "[ UNKNOWN "
        SKIPPED_SIGN = "[ SKIPPED "
        HUNG_SIGN = "Found hung queries in processlist"

        total = 0
        skipped = 0
        unknown = 0
        failed = 0
        success = 0
        hung = False
        test_results = []
        with open(log_path, 'r') as test_file:
            for line in test_file:
                line = line.strip()
                if HUNG_SIGN in line:
                    hung = True
                if any(sign in line for sign in (OK_SIGN, FAIL_SING, UNKNOWN_SIGN, SKIPPED_SIGN)):
                    test_name = line.split(' ')[2].split(':')[0]

                    test_time = ''
                    try:
                        time_token = line.split(']')[1].strip().split()[0]
                        float(time_token)
                        test_time = time_token
                    except:
                        pass

                    total += 1
                    if TIMEOUT_SING in line:
                        failed += 1
                        test_results.append((test_name, "Timeout", test_time))
                    elif FAIL_SING in line:
                        failed += 1
                        test_results.append((test_name, "FAIL", test_time))
                    elif UNKNOWN_SIGN in line:
                        unknown += 1
                        test_results.append((test_name, "FAIL", test_time))
                    elif SKIPPED_SIGN in line:
                        skipped += 1
                        test_results.append((test_name, "SKIPPED", test_time))
                    else:
                        success += int(OK_SIGN in line)
                        test_results.append((test_name, "OK", test_time))
        return total, skipped, unknown, failed, success, hung, test_results

    @staticmethod
    def process_result(task, result_path, server_log_path, perfraw_path, commit, repo, pull_request):
        test_results = []
        state = "success"
        description = ""
        text_log_path = os.path.join(result_path, 'text_log_dump.tar')
        query_log_path = os.path.join(result_path, 'query_log_dump.tar')
        files = os.listdir(result_path)
        if files:
            logging.info("Find files in result folder %s", ','.join(files))
            result_path = os.path.join(result_path, 'test_result.txt')
        else:
            result_path = None
            description = "No output log"
            state = "error"

        if os.path.exists(perfraw_path) and os.listdir(perfraw_path):
            logging.info("Create perfraw part resource for path %s", perfraw_path)
            compress_fast(perfraw_path, "clickhouse_coverage.tar.gz")
            task.make_coverage_resource("clickhouse_coverage.tar.gz", commit, pull_request)
            logging.info("Resource created")

        if result_path and os.path.exists(result_path):
            total, skipped, unknown, failed, success, hung, test_results = FunctionalTestDeprecated.process_test_log(result_path)
            # If no tests were run (success == 0) it indicates an error (e.g. server did not start or crashed immediately)
            # But it's Ok for "flaky checks" - they can contain just one test for check which is marked as skipped.
            if failed != 0 or unknown != 0 or (success == 0 and (not ('flaky' in task.get_context_name()))):
                state = "failure"

            if hung:
                description = "Some queries hung, "
                state = "failure"
            else:
                description = ""

            description += "fail: {}, passed: {}".format(failed, success)
            if skipped != 0:
                description += ", skipped: {}".format(skipped)
            if unknown != 0:
                description += ", unknown: {}".format(unknown)
        else:
            state = "failure"
            description = "Output log doesn't exist"
            test_results = []

        result_logs = []
        server_log = os.path.join(server_log_path, "clickhouse-server.log")
        stderr_log = os.path.join(server_log_path, "stderr.log")
        if os.path.exists(server_log):
            result_logs.append(server_log)
        if os.path.exists(stderr_log):
            result_logs.append(stderr_log)
        if os.path.exists(text_log_path):
            result_logs.append(text_log_path)
        if os.path.exists(query_log_path):
            result_logs.append(query_log_path)
        return state, description, test_results, result_logs


class UnitTestDeprecated:

    @staticmethod
    def get_test_name(line):
        elements = reversed(line.split(' '))
        for element in elements:
            if '(' not in element and ')' not in element:
                return element
        raise Exception("No test name in line '{}'".format(line))

    @staticmethod
    def process_result(task, result_folder, server_log_folder, perfraw_path, commit, repo, pull_request):
        OK_SIGN = 'OK ]'
        FAILED_SIGN = 'FAILED  ]'
        SEGFAULT = 'Segmentation fault'
        SIGNAL = 'received signal SIG'
        PASSED = 'PASSED'

        summary = []
        total_counter = 0
        failed_counter = 0
        result_log_path = '{}/test_result.txt'.format(result_folder)
        if not os.path.exists(result_log_path):
            logging.info("No output log on path %s", result_log_path)
            return "exception", "No output log", []

        status = "success"
        description = ""
        passed = False
        with open(result_log_path, 'r') as test_result:
            for line in test_result:
                if OK_SIGN in line:
                    logging.info("Found ok line: '%s'", line)
                    test_name = UnitTestDeprecated.get_test_name(line.strip())
                    logging.info("Test name: '%s'", test_name)
                    summary.append((test_name, "OK"))
                    total_counter += 1
                elif FAILED_SIGN in line and 'listed below' not in line and 'ms)' in line:
                    logging.info("Found fail line: '%s'", line)
                    test_name = UnitTestDeprecated.get_test_name(line.strip())
                    logging.info("Test name: '%s'", test_name)
                    summary.append((test_name, "FAIL"))
                    total_counter += 1
                    failed_counter += 1
                elif SEGFAULT in line:
                    logging.info("Found segfault line: '%s'", line)
                    status = "failure"
                    description += "Segmentation fault. "
                    break
                elif SIGNAL in line:
                    logging.info("Received signal line: '%s'", line)
                    status = "failure"
                    description += "Exit on signal. "
                    break
                elif PASSED in line:
                    logging.info("PASSED record found: '%s'", line)
                    passed = True

        if not passed:
            status = "failure"
            description += "PASSED record not found. "

        if failed_counter != 0:
            status = "failure"

        if not description:
            description += "fail: {}, passed: {}".format(failed_counter, total_counter - failed_counter)

        return status, description, summary, [result_log_path]


class IntergationTestDeprecated:
    MAX_RETRY = 2
    SLEEP_BETWEEN_RETRIES = 5
    CLICKHOUSE_BINARY_PATH = "/usr/bin/clickhouse"
    CLICKHOUSE_ODBC_BRIDGE_BINARY_PATH = "/usr/bin/clickhouse-odbc-bridge"
    CLICKHOUSE_LIBRARY_BRIDGE_BINARY_PATH = "/usr/bin/clickhouse-library-bridge"

    @staticmethod
    def _get_deselect_option(tests):
        return ' '.join(['--deselect {}'.format(t) for t in tests])

    @staticmethod
    def parse_test_results_output(fname):
        read = False
        description_output = []
        with open(fname, 'r') as out:
            for line in out:
                if read and line.strip() and not line.startswith('=='):
                    description_output.append(line.strip())
                if 'short test summary info' in line:
                    read = True
        return description_output

    @staticmethod
    def get_counters(output):
        counters = {
            "ERROR": set([]),
            "PASSED": set([]),
            "FAILED": set([]),
        }

        for line in output:
            if '.py' in line:
                line_arr = line.strip().split(' ')
                state = line_arr[0]
                test_name = ' '.join(line_arr[1:])
                if ' - ' in test_name:
                    test_name = test_name[:test_name.find(' - ')]
                if state in counters:
                    counters[state].add(test_name)
                else:
                    logging.info("Strange line %s", line)
            else:
                logging.info("Strange line %s")
        return {k: list(v) for k, v in counters.iteritems()}

    @staticmethod
    def parse_test_times(fname):
        read = False
        description_output = []
        with open(fname, 'r') as out:
            for line in out:
                if read and '==' in line:
                    break
                if read and line.strip():
                    description_output.append(line.strip())
                if 'slowest durations' in line:
                    read = True
        return description_output

    @staticmethod
    def get_test_times(output):
        result = defaultdict(float)
        for line in output:
            if '.py' in line:
                line_arr = line.strip().split(' ')
                test_time = line_arr[0]
                test_name = ' '.join([elem for elem in line_arr[2:] if elem])
                if test_name not in result:
                    result[test_name] = 0.0
                result[test_name] += float(test_time[:-1])
        return result

    @staticmethod
    def clear_ip_tables_and_restart_daemons():
        logging.info("Dump iptables after run %s", subprocess.check_output("iptables -L", shell=True))
        try:
            logging.info("Killing all alive docker containers")
            subprocess.check_output("docker kill $(docker ps -q)", shell=True)
        except subprocess.CalledProcessError as err:
            logging.info("docker kill excepted: " + str(err))

        try:
            logging.info("Removing all docker containers")
            subprocess.check_output("docker rm $(docker ps -a -q) --force", shell=True)
        except subprocess.CalledProcessError as err:
            logging.info("docker rm excepted: " + str(err))

        try:
            logging.info("Stopping docker daemon")
            subprocess.check_output("service docker stop", shell=True)
        except subprocess.CalledProcessError as err:
            logging.info("docker stop excepted: " + str(err))

        try:
            for i in range(200):
                try:
                    logging.info("Restarting docker %s", i)
                    subprocess.check_output("service docker start", shell=True)
                    subprocess.check_output("docker ps", shell=True)
                    break
                except subprocess.CalledProcessError as err:
                    time.sleep(0.5)
                    logging.info("Waiting docker to start, current %s", str(err))
            else:
                raise Exception("Docker daemon doesn't responding")
        except subprocess.CalledProcessError as err:
            logging.info("Can't reload docker: " + str(err))

        try:
            for i in range(1000):
                subprocess.check_call("iptables -D DOCKER-USER 1", shell=True)  # when rules will be empty, it will raise exception
        except:
            logging.info("All iptables rules cleared")

    @staticmethod
    def _can_run_with(task, path, opt):
        with open(path, 'r') as script:
            for line in script:
                if opt in line:
                    return True
        return False

    @staticmethod
    def _get_install_path(task):
        result_path_bin = os.path.join(str(task.path()), "clickhouse")
        result_path_odbc_bridge = os.path.join(str(task.path()), "clickhouse-odbc-bridge")
        result_path_library_bridge = os.path.join(str(task.path()), "clickhouse-library-bridge")
        return result_path_bin, result_path_odbc_bridge, result_path_library_bridge

    @staticmethod
    def _install_clickhouse(task, debs_path):
        for package in ('clickhouse-common-static_', 'clickhouse-server_', 'clickhouse-client', 'clickhouse-common-static-dbg_'):  # order matters
            logging.info("Installing package %s", package)
            for f in os.listdir(debs_path):
                if package in f:
                    full_path = os.path.join(debs_path, f)
                    logging.info("Package found in %s", full_path)
                    log_name = "install_" + f + ".log"
                    with sandbox.sdk2.helpers.ProcessLog(task, logger=log_name) as pl:
                        cmd = "dpkg -i {}".format(full_path)
                        logging.info("Executing installation cmd %s", cmd)
                        retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
                        if retcode == 0:
                            logging.info("Instsallation of %s successfull", full_path)
                        else:
                            raise Exception("Installation of %s failed", full_path)
                    break
            else:
                raise Exception("Package with {} not found".format(package))
        logging.info("Unstripping binary")
        # logging.info("Unstring %s", subprocess.check_output("eu-unstrip /usr/bin/clickhouse {}".format(CLICKHOUSE_BINARY_PATH), shell=True))
        logging.info("All packages installed")

        os.chmod(IntergationTestDeprecated.CLICKHOUSE_BINARY_PATH, 0o777)
        os.chmod(IntergationTestDeprecated.CLICKHOUSE_ODBC_BRIDGE_BINARY_PATH, 0o777)

        result_path_bin, result_path_odbc_bridge, result_path_library_bridge = IntergationTestDeprecated._get_install_path(task)
        shutil.copy(IntergationTestDeprecated.CLICKHOUSE_BINARY_PATH, result_path_bin)
        shutil.copy(IntergationTestDeprecated.CLICKHOUSE_ODBC_BRIDGE_BINARY_PATH, result_path_odbc_bridge)

        library_bridge_bin_path = IntergationTestDeprecated.CLICKHOUSE_LIBRARY_BRIDGE_BINARY_PATH
        if os.path.exists(library_bridge_bin_path):
            os.chmod(library_bridge_bin_path, 0o777)
            shutil.copy(IntergationTestDeprecated.CLICKHOUSE_LIBRARY_BRIDGE_BINARY_PATH, result_path_library_bridge)
        return result_path_bin, result_path_odbc_bridge, result_path_library_bridge

    @staticmethod
    def _compress_logs(task, path, result_path):
        subprocess.check_call("tar czf {} -C {} .".format(result_path, path), shell=True)

    @staticmethod
    def _get_all_tests(task, repo_path):
        image_cmd = IntergationTestDeprecated._get_runner_image_cmd(task, repo_path)
        cmd = "cd {}/tests/integration && ./runner {} ' --setup-plan' | grep '::' | sed 's/ (fixtures used:.*//g' | sed 's/^ *//g' > all_tests.txt".format(repo_path, image_cmd)
        logging.info("Getting all tests with cmd '%s'", cmd)
        subprocess.check_call(cmd, shell=True)

        all_tests_file_path = "{}/tests/integration/all_tests.txt".format(repo_path)
        if not os.path.isfile(all_tests_file_path) or os.path.getsize(all_tests_file_path) == 0:
            raise Exception("There is something wrong with getting all tests list: file '{}' is empty or does not exist.".format(all_tests_file_path))

        all_tests = []
        with open(all_tests_file_path, "r") as all_tests_file:
            for line in all_tests_file:
                all_tests.append(line.strip())
        return list(sorted(all_tests))

    @staticmethod
    def group_test_by_file(task, tests):
        result = {}
        for test in tests:
            test_file = test.split('::')[0]
            if test_file not in result:
                result[test_file] = []
            result[test_file].append(test)
        return result

    @staticmethod
    def _update_counters(task, main_counters, current_counters):
        for test in current_counters["PASSED"]:
            if test not in main_counters["PASSED"]:
                if test in main_counters["FAILED"]:
                    main_counters["FAILED"].remove(test)
                if test in main_counters["ERROR"]:
                    main_counters["ERROR"].remove(test)
                main_counters["PASSED"].append(test)

        for state in ("ERROR", "FAILED"):
            for test in current_counters[state]:
                if test in main_counters["PASSED"]:
                    continue
                if test not in main_counters[state]:
                    main_counters[state].append(test)

    @staticmethod
    def _get_runner_image_cmd(task, repo_path):
        image_cmd = ''
        if IntergationTestDeprecated._can_run_with(task, os.path.join(repo_path, "tests/integration", "runner"), '--docker-image-version'):
            for img in task.get_images_names():
                if img == "yandex/clickhouse-integration-tests-runner":
                    runner_version = task.get_single_image_version()
                    logging.info("Can run with custom docker image version %s", runner_version)
                    image_cmd += ' --docker-image-version={} '.format(runner_version)
                else:
                    if IntergationTestDeprecated._can_run_with(task, os.path.join(repo_path, "tests/integration", "runner"), '--docker-compose-images-tags'):
                        image_cmd += '--docker-compose-images-tags={} '.format(task.get_image_with_version(img))
        else:
            image_cmd = ''
            logging.info("Cannot run with custom docker image version :(")
        return image_cmd

    @staticmethod
    def run_test_group(task, repo_path, test_group, tests_in_group, my_env, num_tries):
        image_cmd = IntergationTestDeprecated._get_runner_image_cmd(task, repo_path)
        counters = {
            "ERROR": [],
            "PASSED": [],
            "FAILED": [],
        }
        tests_times = defaultdict(float)
        test_group_str = test_group.replace('/', '_').replace('.', '_')

        for i in range(num_tries):
            logging.info("Running test group %s for the %s retry", test_group, i)
            IntergationTestDeprecated.clear_ip_tables_and_restart_daemons()

            output_path = os.path.join(str(task.path()), "test_output_" + test_group_str + "_" + str(i) + ".log")
            log_name = "integration_run_" + test_group_str + "_" + str(i) + ".txt"
            logging.info("Will wait output inside %s", output_path)

            test_names = set([])
            for test_name in tests_in_group:
                if test_name not in counters["PASSED"]:
                    if '[' in test_name:
                        test_names.add(test_name[:test_name.find('[')])
                    else:
                        test_names.add(test_name)

            test_cmd = ' '.join([test for test in sorted(test_names)])
            cmd = "cd {}/tests/integration && ./runner {} '-ss {} -rfEp --color=no --durations=0 {}' | tee {}".format(
                repo_path, image_cmd, test_cmd, IntergationTestDeprecated._get_deselect_option(task.should_skip_tests()), output_path)

            with sandbox.sdk2.helpers.ProcessLog(task, logger=log_name) as pl:
                logging.info("Executing cmd: %s", cmd)
                retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout, env=my_env).wait()
                if retcode == 0:
                    logging.info("Run %s group successfully", test_group)
                else:
                    logging.info("Some tests failed")
                log_path = str(pl.stdout.path)

            if os.path.exists(output_path):
                lines = IntergationTestDeprecated.parse_test_results_output(output_path)
                new_counters = IntergationTestDeprecated.get_counters(lines)
                times_lines = IntergationTestDeprecated.parse_test_times(output_path)
                new_tests_times = IntergationTestDeprecated.get_test_times(times_lines)
                IntergationTestDeprecated._update_counters(task, counters, new_counters)
                for test_name, test_time in new_tests_times.items():
                    tests_times[test_name] = test_time
            if len(counters["PASSED"]) == len(tests_in_group):
                logging.info("All tests from group %s passed", test_group)
                break
            if len(counters["PASSED"]) >= 0 and len(counters["FAILED"]) == 0 and len(counters["ERROR"]) == 0:
                logging.info("Seems like all tests passed but some of them are skipped or deselected. Ignoring them and finishing group.")
                break
        else:
            for test in tests_in_group:
                if test not in counters["PASSED"] and test not in counters["ERROR"] and test not in counters["FAILED"]:
                    counters["ERROR"].append(test)

        return counters, tests_times, log_name, log_path

    @staticmethod
    def run_impl(task, commit, repo, pull_request, repo_path, build_path, my_env):
        IntergationTestDeprecated._install_clickhouse(task, build_path)
        logging.info("Dump iptables before run %s", subprocess.check_output("iptables -L", shell=True))
        all_tests = IntergationTestDeprecated._get_all_tests(task, repo_path)
        logging.info("Found %s tests first 3 %s", len(all_tests), ' '.join(all_tests[:3]))
        grouped_tests = IntergationTestDeprecated.group_test_by_file(task, all_tests)
        logging.info("Found %s tests groups", len(grouped_tests))

        counters = {
            "ERROR": [],
            "PASSED": [],
            "FAILED": [],
        }
        tests_times = defaultdict(float)

        logs = []
        items_to_run = list(grouped_tests.items())

        logging.info("Total test groups %s", len(items_to_run))
        if task.shuffle_test_groups():
            logging.info("Shuffling test groups")
            random.shuffle(items_to_run)

        for group, tests in items_to_run:
            logging.info("Running test group %s countaining %s tests", group, len(tests))
            group_counters, group_test_times, log_name, log_path = IntergationTestDeprecated.run_test_group(task, repo_path, group, tests, my_env, IntergationTestDeprecated.MAX_RETRY)
            task._update_desc(log_name, group)
            total_tests = 0
            for counter, value in group_counters.items():
                logging.info("Tests from group %s stats, %s count %s", group, counter, len(value))
                counters[counter] += value
                logging.info("Totally have %s with status %s", len(counters[counter]), counter)
                total_tests += len(counters[counter])
            logging.info("Totally finished tests %s/%s", total_tests, len(all_tests))

            for test_name, test_time in group_test_times.items():
                tests_times[test_name] = test_time
            logs.append(log_path)
            if len(counters["FAILED"]) + len(counters["ERROR"]) >= 20:
                logging.info("Collected more than 20 failed/error tests, stopping")
                break

        logging.info("Finally all tests done, going to compress test dir")
        test_logs = os.path.join(str(task.path()), "./test_dir.tar")
        IntergationTestDeprecated._compress_logs(task, "{}/tests/integration".format(repo_path), test_logs)
        logging.info("Compression finished")

        if counters["FAILED"] or counters["ERROR"]:
            logging.info("Overall status failure, because we have tests in FAILED or ERROR state")
            result_state = "failure"
        else:
            logging.info("Overall success!")
            result_state = "success"

        test_result = []
        for state in ("ERROR", "FAILED", "PASSED"):
            task.Parameters.description += '\n' + '\n'.join([state + " " + c for c in counters[state]])
            if state == "PASSED":
                text_state = "OK"
            elif state == "FAILED":
                text_state = "FAIL"
            else:
                text_state = state
            test_result += [(c, text_state, str(tests_times[c])) for c in counters[state]]

        status_text = ', '.join([str(n).lower().replace('failed', 'fail') + ': ' + str(len(c)) for n, c in counters.items()])

        if not counters or sum(len(counter) for counter in counters.values()) == 0:
            status_text = "No tests found for some reason! It's a bug"
            result_state = "failure"

        return result_state, status_text, test_result, log_path, [test_logs] + logs


class IntergationTestFlakyCheckDeprecated(IntergationTestDeprecated):
    TRIES_COUNT = 10
    MAX_TIME_SECONDS = 3600

    @staticmethod
    def filter_existing_tests(tests_to_run, repo_path):
        result = []
        for relative_test_path in tests_to_run:
            if os.path.exists(os.path.join(repo_path, 'tests/integration', relative_test_path)):
                result.append(relative_test_path)
            else:
                logging.info("Skipping test %s, seems like it was removed", relative_test_path)
        return result

    @staticmethod
    def run_impl(task, commit, repo, pull_request, repo_path, build_path, my_env):
        pr_info = PRInfo(pull_request, commit)

        # pytest swears, if we require to run some tests which was renamed or deleted
        tests_to_run = IntergationTestFlakyCheckDeprecated.filter_existing_tests(task.get_tests_to_run(pr_info), repo_path)
        if not tests_to_run:
            logging.info("No tests to run found")
            return 'success', 'Nothing to run', [('Nothing to run', 'OK')], ''

        IntergationTestFlakyCheckDeprecated._install_clickhouse(task, build_path)
        logging.info("Found '%s' tests to run", ' '.join(tests_to_run))
        result_state = "success"
        description_prefix = "No flaky tests: "
        start = time.time()
        logging.info("Starting check with retries")
        final_retry = 0
        log_paths = []
        for i in range(IntergationTestFlakyCheckDeprecated.TRIES_COUNT):
            final_retry += 1
            logging.info("Running tests for the %s time", i)
            counters, tests_times, log_name, log_path = IntergationTestFlakyCheckDeprecated.run_test_group(task, repo_path, "flaky", tests_to_run, my_env, 1)
            log_paths.append(log_path)
            task._update_desc(log_name, i)
            if counters["FAILED"]:
                logging.info("Found failed tests: %s", ' '.join(counters["FAILED"]))
                description_prefix = "Flaky tests found: "
                result_state = "failure"
                break
            if counters["ERROR"]:
                description_prefix = "Flaky tests found: "
                logging.info("Found error tests: %s", ' '.join(counters["ERROR"]))
                result_state = "error"
                break
            logging.info("Try is OK, all tests passed, going to clear env")
            IntergationTestFlakyCheckDeprecated.clear_ip_tables_and_restart_daemons()
            logging.info("And going to sleep for some time")
            if time.time() - start > IntergationTestFlakyCheckDeprecated.MAX_TIME_SECONDS:
                logging.info("Timeout reached, going to finish flaky check")
                break
            time.sleep(5)

        logging.info("Finally all tests done, going to compress test dir")
        test_logs = os.path.join(str(task.path()), "./test_dir.tar")
        IntergationTestFlakyCheckDeprecated._compress_logs(task, "{}/tests/integration".format(repo_path), test_logs)
        logging.info("Compression finished")

        test_result = []
        for state in ("ERROR", "FAILED", "PASSED"):
            task.Parameters.description += '\n' + '\n'.join([state + " " + c for c in counters[state]])
            if state == "PASSED":
                text_state = "OK"
            elif state == "FAILED":
                text_state = "FAIL"
            else:
                text_state = state
            test_result += [(c + ' (✕' + str(final_retry) + ')', text_state, str(tests_times[c])) for c in counters[state]]
        status_text = description_prefix + ', '.join([str(n).lower().replace('failed', 'fail') + ': ' + str(len(c)) for n, c in counters.items()])

        return result_state, status_text, test_result, log_path, [test_logs] + log_paths


class StyleCheckDeprecated:

    @staticmethod
    def process_result(task, result_folder, server_log_folder, perfraw_path, commit, repo, pull_request):
        status = "success"
        description = ""
        test_results = []

        style_log_path = '{}/style_output.txt'.format(result_folder)
        if not os.path.exists(style_log_path):
            logging.info("No style check log on path %s", style_log_path)
            return "exception", "No style check log", []
        elif os.stat(style_log_path).st_size != 0:
            description += "Style check failed. "
            test_results.append(("Style check", "FAIL"))
            status = "failure"  # Disabled for now
        else:
            test_results.append(("Style check", "OK"))

        typos_log_path = '{}/typos_output.txt'.format(result_folder)
        if not os.path.exists(style_log_path):
            logging.info("No typos check log on path %s", style_log_path)
            return "exception", "No typos check log", []
        elif os.stat(typos_log_path).st_size != 0:
            description += "Typos check failed. "
            test_results.append(("Typos check", "FAIL"))
            status = "failure"
        else:
            test_results.append(("Typos check", "OK"))

        whitespaces_log_path = '{}/whitespaces_output.txt'.format(result_folder)
        if not os.path.exists(style_log_path):
            logging.info("No whitespaces check log on path %s", style_log_path)
            return "exception", "No whitespaces check log", []
        elif os.stat(whitespaces_log_path).st_size != 0:
            description += "Whitespaces check failed. "
            test_results.append(("Whitespaces check", "FAIL"))
            status = "failure"
        else:
            test_results.append(("Whitespaces check", "OK"))

        duplicate_log_path = '{}/duplicate_output.txt'.format(result_folder)
        if not os.path.exists(duplicate_log_path):
            logging.info("No header duplicates check log on path %s", duplicate_log_path)
            return "exception", "No header duplicates check log", []
        elif os.stat(duplicate_log_path).st_size != 0:
            description += " Header duplicates check failed. "
            test_results.append(("Header duplicates check", "FAIL"))
            status = "failure"
        else:
            test_results.append(("Header duplicates check", "OK"))

        shellcheck_log_path = '{}/shellcheck_output.txt'.format(result_folder)
        if not os.path.exists(shellcheck_log_path):
            logging.info("No shellcheck  log on path %s", shellcheck_log_path)
            return "exception", "No shellcheck log", []
        elif os.stat(shellcheck_log_path).st_size != 0:
            description += " Shellcheck check failed. "
            test_results.append(("Shellcheck ", "FAIL"))
            status = "failure"
        else:
            test_results.append(("Shellcheck", "OK"))

        if not description:
            description += "Style check success"

        return status, description, test_results, [style_log_path, duplicate_log_path, whitespaces_log_path, typos_log_path, shellcheck_log_path]


class TestflowsDeprecated:

    @staticmethod
    def process_result(task, result_folder, server_log_path, perfraw_path, commit, repo, pull_request):
        json_path = os.path.join(result_folder, "results.json")
        if not os.path.exists(json_path):
            return "success", "No testflows in branch", None, []

        test_binary_log = os.path.join(result_folder, "test.log")
        with open(json_path) as source:
            results = json.loads(source.read())

        total_tests = 0
        total_ok = 0
        total_fail = 0
        total_other = 0
        test_results = []
        for test in results["tests"]:
            test_name = test['test']['test_name']
            test_result = test['result']['result_type'].upper()
            test_time = str(test['result']['message_rtime'])
            total_tests += 1
            if test_result == "OK":
                total_ok += 1
            elif test_result == "FAIL" or test_result == "ERROR":
                total_fail += 1
            else:
                total_other += 1

            test_results.append((test_name, test_result, test_time))
        if total_fail != 0:
            status = "failure"
        else:
            status = "success"

        description = "failed: {}, passed: {}, other: {}".format(total_fail, total_ok, total_other)
        return status, description, test_results, [json_path, test_binary_log]


class SqlancerDeprecated:

    @staticmethod
    def process_result(task, result_folder, server_log_folder, perfraw_path, commit, repo, pull_request):
        status = "success"
        summary = []
        paths = []
        tests = ["TLPWhere", "TLPGroupBy", "TLPHaving", "TLPWhereGroupBy", "TLPDistinct", "TLPAggregate"]

        for test in tests:
            err_path = '{}/{}.err'.format(result_folder, test)
            out_path = '{}/{}.out'.format(result_folder, test)
            if not os.path.exists(err_path):
                logging.info("No output err on path %s", err_path)
                summary.append((test, "SKIPPED"))
            elif not os.path.exists(out_path):
                logging.info("No output log on path %s", out_path)
            else:
                paths.append(err_path)
                paths.append(out_path)
                with open(err_path, 'r') as f:
                    if 'AssertionError' in f.read():
                        summary.append((test, "FAIL"))
                    else:
                        summary.append((test, "OK"))

        logs_path = '{}/logs.tar.gz'.format(result_folder)
        if not os.path.exists(logs_path):
            logging.info("No logs tar on path %s", logs_path)
        else:
            paths.append(logs_path)
        stdout_path = '{}/stdout.log'.format(result_folder)
        if not os.path.exists(stdout_path):
            logging.info("No stdout log on path %s", stdout_path)
        else:
            paths.append(stdout_path)
        stderr_path = '{}/stderr.log'.format(result_folder)
        if not os.path.exists(stderr_path):
            logging.info("No stderr log on path %s", stderr_path)
        else:
            paths.append(stderr_path)

        description = "SQLancer test run. See report"

        return status, description, summary, paths


class SplitBuildSmokeTestDeprecated:

    @staticmethod
    def process_result(task, result_folder, server_log_folder, perfraw_path,
                       commit, repo, pull_request):

        status = "success"
        description = 'Server started and responded'
        summary = [("Smoke test", "OK")]
        with open(os.path.join(result_folder, task.RESULT_LOG_NAME), 'r') as run_log:
            lines = run_log.read().split('\n')
            if not lines or lines[0].strip() != 'OK':
                status = "failure"
                logging.info("Lines is not ok: %s", str('\n'.join(lines)))
                summary = [("Smoke test", "FAIL")]
                description = 'Server failed to respond, see result in logs'

        result_logs = []
        server_log_path = os.path.join(server_log_folder, "clickhouse-server.log")
        stderr_log_path = os.path.join(server_log_folder, "stderr.log")
        client_stderr_log_path = os.path.join(server_log_folder, "clientstderr.log")

        if os.path.exists(server_log_path):
            result_logs.append(server_log_path)

        if os.path.exists(stderr_log_path):
            result_logs.append(stderr_log_path)

        if os.path.exists(client_stderr_log_path):
            result_logs.append(client_stderr_log_path)

        return status, description, summary, result_logs
