# -*- coding: utf-8 -*-
from sandbox.projects.clickhouse.util.resource_helper import ResourceHelper
from sandbox.projects.clickhouse.util.git_helper import GitHelper
import logging
import os
import requests
import json


def has_image_for_pr_commit(image_name, pull_request_number, commit_sha):
    tag = "{}-{}".format(pull_request_number, commit_sha)
    url = 'https://hub.docker.com/v2/repositories/{}/tags/{}'.format(image_name, tag)

    for _ in range(0, 5):
        response = requests.get(url)
        logging.info("Checked image availability on DockerHub: tag '{}', API URL '{}', status {}, text '{}'".format(
            tag, url, response.status_code, response.text))

        if response.status_code == 200:
            # A sanity check
            received_tag = response.json().get('name')
            if received_tag != tag:
                logging.info("Received wrong tag '{}' from DockerHub, expected '{}'".format(received_tag, tag))
                return None
            return tag
        elif response.status_code == 404:
            # Probably didn't publish yet, will wait
            return None
        else:
            # Network problem? Will retry.
            continue


class DockerImageHelper(object):

    def __init__(self, gh_helper):
        self.gh_helper = gh_helper
        self.changed_images_for_prs = {}

    def get_changed_files_from_repo(self, commit, pull_request, repo_path):
        return None

    def get_changed_docker_images(self, commit, pull_request, repo_path, image_file_path):
        images_dict = {}
        path_to_images_file = os.path.join(repo_path, image_file_path)
        if os.path.exists(path_to_images_file):
            with open(path_to_images_file, 'r') as dict_file:
                images_dict = json.load(dict_file)
        else:
            logging.info("Image file %s doesnt exists in repo %s", image_file_path, repo_path)

        dockerhub_repo_name = 'yandex'
        if not images_dict:
            return [], dockerhub_repo_name

        if pull_request.number != 0:
            githelper = GitHelper(work_dir=repo_path)
            files_changed = githelper.get_changed_files(images_dict.keys())
        else:
            files_changed = self.gh_helper.get_commit_changed_files(commit)

        logging.info("Changed files for PR %s @ %s: %s", pull_request.number, commit.sha, str(files_changed))

        changed_images = []

        for dockerfile_dir, image_description in images_dict.items():
            if image_description['name'].startswith('clickhouse/'):
                dockerhub_repo_name = 'clickhouse'

            for f in files_changed:
                if f.startswith(dockerfile_dir):
                    logging.info(
                        "Found changed file '%s' which affects docker image '%s' with path '%s'",
                        f, image_description['name'], dockerfile_dir)
                    changed_images.append(dockerfile_dir)
                    break

        # The order is important: dependents should go later than bases, so that
        # they are built with updated base versions.
        index = 0
        while index < len(changed_images):
            image = changed_images[index]
            for dependent in images_dict[image]['dependent']:
                logging.info(
                    "Marking docker image '%s' as changed because it depends on changed docker image '%s'",
                    dependent, image)
                changed_images.append(dependent)
            index += 1
            if index > 100:
                # Sanity check to prevent infinite loop.
                raise "Too many changed docker images, this is a bug." + str(changed_images)

        # If a dependent image was already in the list because its own files
        # changed, but then it was added as a dependent of a changed base, we
        # must remove the earlier entry so that it doesn't go earlier than its
        # base. This way, the dependent will be rebuilt later than the base, and
        # will correctly use the updated version of the base.
        seen = set()
        no_dups_reversed = []
        for x in reversed(changed_images):
            if x not in seen:
                seen.add(x)
                no_dups_reversed.append(x)

        result = [(x, images_dict[x]['name']) for x in reversed(no_dups_reversed)]
        logging.info("Changed docker images for PR %s @ %s: '%s'", pull_request.number, commit.sha, result)
        return result, dockerhub_repo_name

    def get_changed_docker_images_set(self, commit, pull_request, repo_path, image_file_path):
        changed_images, dockerhub_repo_name = self.get_changed_docker_images(commit, pull_request, repo_path, image_file_path)
        return set([image[1] for image in changed_images]), dockerhub_repo_name

    def get_changed_images_for_pr(self, repo, pr_info):
        combined_version = str(pr_info.number) + '-' + pr_info.sha
        if combined_version not in self.changed_images_for_prs:
            resource_helper = ResourceHelper(pr_info.pygithub_commit, repo, pr_info.pygithub_pr)
            repo_resource = resource_helper.get_repo_no_submodules_resource()
            if repo_resource:
                repo_path = resource_helper.save_any_repo_resource(repo_resource, "./" + combined_version)
                self.changed_images_for_prs[combined_version] = self.get_changed_docker_images_set(
                    pr_info.pygithub_commit, pr_info.pygithub_pr, repo_path, "docker/images.json")
                return self.changed_images_for_prs[combined_version]
            return None, None
        else:
            return self.changed_images_for_prs[combined_version]

    def get_changed_images_with_versions(self, required_images_names, changed_images, pull_request, commit, task_name):
        docker_images_with_versions = {}
        image_not_ready = False
        for image_name in required_images_names:
            if image_name in changed_images:
                image_version = has_image_for_pr_commit(image_name, pull_request.number, commit.sha)
                if image_version is None:
                    image_not_ready = True
                    logging.info("Docker image %s for task %s is not ready yet",  image_name, task_name)
                    break
                else:
                    docker_images_with_versions[image_name] = image_version
            else:
                logging.info("Docker image %s for task %s is not changed, will use latest version", image_name, task_name)
                docker_images_with_versions[image_name] = 'latest'

        if image_not_ready:
            return None
        return docker_images_with_versions
