# -* coding: utf-8 -*-

from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.ssh import Key
from sandbox import common
import sandbox
import subprocess
import datetime
import os
import logging
import time


class ClickHouseVersion(object):
    def __init__(self, major, minor, patch, tweak, revision):
        self.major = major
        self.minor = minor
        self.patch = patch
        self.tweak = tweak
        self.revision = revision

    def minor_update(self):
        return ClickHouseVersion(
            self.major,
            self.minor + 1,
            1,
            1,
            self.revision + 1)

    def patch_update(self):
        return ClickHouseVersion(
            self.major,
            self.minor,
            self.patch + 1,
            1,
            self.revision)

    def tweak_update(self):
        return ClickHouseVersion(
            self.major,
            self.minor,
            self.patch,
            self.tweak + 1,
            self.revision)

    def get_version_string(self):
        return '.'.join([
            str(self.major),
            str(self.minor),
            str(self.patch),
            str(self.tweak)
        ])

    def as_tuple(self):
        return (self.major, self.minor, self.patch, self.tweak)


class VersionType(object):
    STABLE = "stable"
    TESTING = "testing"


def build_version_description(version, version_type):
    return "v" + version.get_version_string() + "-" + version_type


def _get_version_from_line(line):
    _, ver_with_bracket = line.strip().split(' ')
    return ver_with_bracket[:-1]


class GitHelper(object):

    FILE_WITH_VERSION_PATH = "cmake/autogenerated_versions.txt"
    CHANGELOG_IN_PATH = "debian/changelog.in"
    CHANGELOG_PATH = "debian/changelog"
    CONTRIBUTORS_SCRIPT_DIR = "src/Storages/System/"

    def __init__(self, task=None, repo=None, work_dir=None, owner=None, email=None, ssh_key=None):
        self.repo = repo
        self.work_dir = work_dir
        self.task = task
        self.ssh_key = ssh_key
        self.owner = owner
        self.email = email

    def _get_read_url(self, pull_request):
        if pull_request.number == 0:
            return self.repo.clone_url
        else:
            return pull_request.raw_data['head']['repo']['clone_url']

    def _get_write_url(self, pull_request):
        if pull_request.number == 0:
            return self.repo.ssh_url
        else:
            return pull_request.raw_data['head']['repo']['ssh_url']

    def _get_ref(self, pull_request):
        if pull_request.number == 0:
            return "master"
        else:
            return pull_request.raw_data['head']['ref']

    def _checkout_to_pr_head(self, git, pull_request):
        try:
            git.execute("fetch", "origin", "+refs/pull/{}/merge".format(pull_request.number), cwd=self.work_dir)
            git.execute("checkout", "FETCH_HEAD", cwd=self.work_dir)
            logging.info("Mergecommit fetched")
            return True
        except common.errors.SubprocessError:
            return False

    def clone(self, git, url, branch="master", commit=None):
        # We shouldn't clone with --depth=1 here, because we'll need the branch
        # history to determine the list of files changed in a PR.
        # git in sandbox is very old and doesn't have `--single-branch`
        # or `--shallow-submodules` options for `clone`.
        git.execute("clone", url, self.work_dir)
        git.execute("checkout", branch, cwd=self.work_dir)
        if commit:
            git.execute("checkout", commit, cwd=self.work_dir)

    # return True, if cloned from mergecommit or master, otherwise return false
    def clone_repo(self, pull_request, commit, with_key=False):
        logging.info("Clone repo called. pull_request={}, commit={}, with_key={}".format(pull_request, commit, with_key))
        if not with_key:
            read_url = self._get_read_url(pull_request)
            git = Git(read_url)
            if pull_request.number == 0:
                self.clone(git, read_url, "master", commit.sha)
                return True

            # can have 3 different states (https://developer.github.com/v3/pulls/#response-1):
            # true -- merge-commit created
            # false -- cannot merge with master
            # null (None) -- merge commit preparing by github
            is_pr_mergeable = pull_request.raw_data['mergeable']
            if is_pr_mergeable is None:
                return None

            base_url = pull_request.raw_data['base']['repo']['clone_url']
            base_git = Git(base_url)
            self.clone(base_git, base_url, "master")
            if not pull_request.raw_data['mergeable'] or not self._checkout_to_pr_head(base_git, pull_request):
                logging.info("Can't fetch mergecommit")
                base_git.execute("fetch", "origin", "+refs/pull/{}/head".format(pull_request.number), cwd=self.work_dir)
                base_git.execute("checkout", "FETCH_HEAD", cwd=self.work_dir)
                logging.info("Checked out unmerged PR head")
                return False
            return True
        else:
            with Key(self.task, self.owner, self.ssh_key):
                write_url = self._get_write_url(pull_request)
                git = Git(write_url)
                if pull_request.number == 0:
                    self.clone(git, write_url, "master")
                    return True
                is_pr_mergeable = pull_request.raw_data['mergeable']
                if is_pr_mergeable is None:
                    return None

                base_url = pull_request.raw_data['base']['repo']['ssh_url']
                base_git = Git(base_url)
                self.clone(base_git, base_url, "master")
                if not pull_request.raw_data['mergeable'] or not self._checkout_to_pr_head(base_git, pull_request):
                    base_git.execute("fetch", "origin", "+refs/pull/{}/head".format(pull_request.number), cwd=self.work_dir)
                    base_git.execute("checkout", "FETCH_HEAD", cwd=self.work_dir)
                    return False
                return True

    def update_submodules(self, pull_request, logger_name_sync, logger_name_update):
        git = Git(self._get_read_url(pull_request))
        git.execute("submodule", "sync", cwd=self.work_dir, logger_name=logger_name_sync)
        last_ex = None
        for i in range(5):
            try:
                # We'd want to avoid cloning the submodules with full history,
                # but git in Sandbox is very old and doesnt' have the
                # `update --depth=1` command. `git submodule foreach` skips the
                # submodules for which you didn't to the `update`, so it's also
                # useless.
                git.execute("submodule", "update", "--init", "--recursive", cwd=self.work_dir, logger_name=logger_name_update)
                return
            except Exception as ex:
                last_ex = ex
                logging.info("Exception during submodules update %s", ex)
                time.sleep(3 * i)
        raise last_ex

    def get_version_from_repo(self):
        path_to_file = os.path.join(self.work_dir, self.FILE_WITH_VERSION_PATH)
        major = 0
        minor = 0
        patch = 0
        tweak = 0
        version_revision = 0
        with open(path_to_file, 'r') as ver_file:
            for line in ver_file:
                if "VERSION_MAJOR" in line and "math" not in line and "SET" in line:
                    major = _get_version_from_line(line)
                elif "VERSION_MINOR" in line and "math" not in line and "SET" in line:
                    minor = _get_version_from_line(line)
                elif "VERSION_PATCH" in line and "math" not in line and "SET" in line:
                    patch = _get_version_from_line(line)
                elif "VERSION_REVISION" in line and "math" not in line:
                    version_revision = _get_version_from_line(line)
        return ClickHouseVersion(major, minor, patch, tweak, version_revision)

    def _update_cmake_version(self, version, commit, version_type):
        cmd = """sed -i --follow-symlinks -e "s/SET(VERSION_REVISION [^) ]*/SET(VERSION_REVISION {revision}/g;" \
                -e "s/SET(VERSION_DESCRIBE [^) ]*/SET(VERSION_DESCRIBE {version_desc}/g;" \
                -e "s/SET(VERSION_GITHASH [^) ]*/SET(VERSION_GITHASH {sha}/g;" \
                -e "s/SET(VERSION_MAJOR [^) ]*/SET(VERSION_MAJOR {major}/g;" \
                -e "s/SET(VERSION_MINOR [^) ]*/SET(VERSION_MINOR {minor}/g;" \
                -e "s/SET(VERSION_PATCH [^) ]*/SET(VERSION_PATCH {patch}/g;" \
                -e "s/SET(VERSION_STRING [^) ]*/SET(VERSION_STRING {version_string}/g;" \
                {path}""".format(
            revision=version.revision,
            version_desc=build_version_description(version, version_type),
            sha=commit.sha,
            major=version.major,
            minor=version.minor,
            patch=version.patch,
            version_string=version.get_version_string(),
            path=os.path.join(self.work_dir, self.FILE_WITH_VERSION_PATH),
        )
        subprocess.check_call(cmd, shell=True)

    def _update_changelog(self, version):
        cmd = """sed \
            -e "s/[@]VERSION_STRING[@]/{version_str}/g" \
            -e "s/[@]DATE[@]/{date}/g" \
            -e "s/[@]AUTHOR[@]/clickhouse-release/g" \
            -e "s/[@]EMAIL[@]/clickhouse-release@yandex-team.ru/g" \
            < {in_path} > {changelog_path}
        """.format(
            version_str=version.get_version_string(),
            date=datetime.datetime.now().strftime("%a, %d %b %Y %H:%M:%S") + " +0300",
            in_path=os.path.join(self.work_dir, self.CHANGELOG_IN_PATH),
            changelog_path=os.path.join(self.work_dir, self.CHANGELOG_PATH)
        )
        subprocess.check_call(cmd, shell=True)

    def _update_contributors(self):
        with sandbox.sdk2.helpers.ProcessLog(None, logger="update_contributors") as pl:
            cmd = "cd {} && ./StorageSystemContributors.sh".format(os.path.join(self.work_dir, self.CONTRIBUTORS_SCRIPT_DIR))
            logging.info("Updating contributors")
            retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            if retcode != 0:
                raise Exception("Cannot update system.contributors")

            logging.info("Update finished")

    def update_versions_list_in_master(self, version, push):
        git = Git(self.repo.ssh_url)
        with Key(self.task, self.owner, self.ssh_key):
            git.execute("config", "user.name", "robot-clickhouse", cwd=self.work_dir)
            git.execute("config", "user.email", self.email, cwd=self.work_dir)
            git.execute("fetch", cwd=self.work_dir)
            git.execute("checkout", "master", cwd=self.work_dir)
            git.execute("pull", "origin", "master", cwd=self.work_dir)

            with sandbox.sdk2.helpers.ProcessLog(None, logger="update_versions") as pl:
                cmd = "cd {path} && utils/list-versions/list-versions.sh > utils/list-versions/version_date.tsv".format(path=self.work_dir)
                retcode = subprocess.Popen(cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
                if retcode != 0:
                    raise Exception("Cannot update utils/list-versions/version_date.tsv")
            git.execute("add", "utils/list-versions/version_date.tsv", cwd=self.work_dir)
            git.execute("commit", "-a", "-m", "Update version_date.tsv after release {}".format(version.get_version_string()), cwd=self.work_dir)

            if push:
                git.execute("push", "origin", "master", cwd=self.work_dir)

    def _update_dockerfile(self, version):
        version_str_for_docker = '.'.join([str(version.major), str(version.minor), str(version.patch), '*'])
        cmd = "ls -1 {path}/docker/*/Dockerfile | xargs sed -i -r -e 's/ARG version=.+$/ARG version='{ver}'/'".format(path=self.work_dir, ver=version_str_for_docker)
        subprocess.check_call(cmd, shell=True)

    def update_version_local(self, commit, version, version_type="testing"):
        self._update_contributors()
        self._update_cmake_version(version, commit, version_type)
        self._update_changelog(version)
        self._update_dockerfile(version)

    def checkout_branch(self, pull_request, branch=None):
        with Key(self.task, self.owner, self.ssh_key):
            git = Git(self._get_write_url(pull_request))
            if not branch:
                branch = self._get_ref(pull_request)
            git.execute("checkout", branch, cwd=self.work_dir)
            git.execute("pull", "origin", branch, cwd=self.work_dir)
            return branch

    def commit_all(self, version, pull_request, commit, push=False, to_branch=None):
        git = Git(self._get_write_url(pull_request))

        with Key(self.task, self.owner, self.ssh_key):
            git.execute("config", "user.name", "robot-clickhouse", cwd=self.work_dir)
            git.execute("config", "user.email", self.email, cwd=self.work_dir)

            branch = self.checkout_branch(pull_request, to_branch)
            git.execute("commit", "-a", "-m", "Auto version update to [{}] [{}]".format(version.get_version_string(), version.revision), cwd=self.work_dir)

            if push:
                git.execute("push", "origin", branch, cwd=self.work_dir)

    def tag_version_and_push(self, commit, version, version_type, pull_request, push=False, update_cmake=False):
        git = Git(self._get_write_url(pull_request))
        if update_cmake:
            self._update_cmake_version(version, commit, version_type)
            self.commit_all(version, pull_request, commit, push)

        tag_text = build_version_description(version, version_type)

        with Key(self.task, self.owner, self.ssh_key):
            git.execute("config", "user.name", "robot-clickhouse", cwd=self.work_dir)
            git.execute("config", "user.email", self.email, cwd=self.work_dir)

            git.execute("tag", "--force", "-a", tag_text, commit.sha, "-m", tag_text, cwd=self.work_dir)
            if push:
                git.execute("push", "origin", tag_text, cwd=self.work_dir)
        return tag_text

    def create_branch_and_push(self, branch_name, pull_request, commit, push=False):
        git = Git(self._get_write_url(pull_request))

        with Key(self.task, self.owner, self.ssh_key):
            git.execute("config", "user.name", "robot-clickhouse", cwd=self.work_dir)
            git.execute("config", "user.email", self.email, cwd=self.work_dir)
            git.execute("checkout", commit.sha, cwd=self.work_dir)

            try:
                git.execute("checkout", "-b", branch_name, cwd=self.work_dir)
            except Exception as ex:
                raise Exception("Seems like branch {} already exists, actually '{}'".format(branch_name, str(ex)))
            if push:
                git.execute("push", "origin", branch_name, cwd=self.work_dir)

    def create_version_patch_branch(self, version, pull_request, commit, push=False):
        if pull_request.number != 0:
            raise Exception("Trying to create fix branch from branch {}, possible bug".format(self._get_ref(pull_request)))

        branch_name = str(version.major) + "." + str(version.minor)
        self.create_branch_and_push(branch_name, pull_request, commit, push)
        return branch_name

    def get_changed_files(self, files_to_check):
        git = Git('')
        result = []
        for file_to_check in files_to_check:
            git_proc = git.raw_execute(['diff', '--name-only', '..master', file_to_check], cwd=self.work_dir)
            stdout, stderr = git_proc.communicate()
            if stdout:
                for line in stdout.split('\n'):
                    result.append(line.strip())
        return result
