import pytz
import logging
import re
import datetime
import json
from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common.nanny import nanny

ST_URL = 'https://st-api.yandex-team.ru/v2/issues/'


class CloseTicketsAfterRelease(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment("startrek_client"),
        )

    class Parameters(sdk2.Parameters):
        wait_time = sdk2.parameters.Integer("Hours to wait before closing released tickets", default=6, required=True)
        queue_name = sdk2.parameters.String("Name of queue in startreker", default="SEAREL", required=True)
        last_days = sdk2.parameters.Integer("How many last days to check", default=2, required=True)

    def on_execute(self):
        if self.Requirements.tasks_resource is None:
            logging.info('self.Requirements.tasks_resource is None')
        if not common.system.inside_the_binary():
            logging.info('Not inside the binary')
        from sandbox.projects.release_machine.helpers.startrek_helper import STHelper

        try:
            self.nanny_token = sdk2.Vault.data('bayan98', "OAUTH_TOKEN_NANNY")
            self.st_token = sdk2.Vault.data('REPORT_CORE', 'zomb-prj-10-st-oauth')
        except common.errors.VaultError:
            raise common.errors.TaskFailure("Cannot get tokens")
        self.st_helper = STHelper(self.st_token)
        self.nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=self.nanny_token)
        logging.info('nanny client %s', self.nanny_client)
        start_date = datetime.datetime.utcnow().replace(tzinfo=pytz.utc) - datetime.timedelta(days=self.Parameters.last_days)
        logging.info('check from %s date', start_date)

        req = {
            'limit': 1000,
            'query': {
                'sandboxTaskType': 'BUILD_REPORT_CORE',
                'sandboxReleaseType': ['stable'],
                'status': ['CLOSED'],
                'endTimeGte': start_date.isoformat()
                },
        }
        logging.info('dumps {}'.format(json.dumps(req)))
        resp = self.nanny_client.find_releases(req)
        logging.info('resp total %s', len(resp))
        self.queue_name_regexp = re.compile(r'{}-\w+'.format(self.Parameters.queue_name))
        self.word_regexp = re.compile(r'\w+')
        self.issue_name_regexp = re.compile(r'[A-Z]+-\d+')
        for release in resp:
            logging.info('checking %s', release['spec']['title'])
            logging.info('Nanny id %s', release['id'])
            try:
                logging.info('release resource type is %s', release['spec']['sandboxRelease']['resourceTypes'])
                if 'REPORT_CORE_PACKAGE' not in release['spec']['sandboxRelease']['resourceTypes']:
                    continue
            except:
                logging.info('can not check release resource type')
                continue
            if self.is_released(release):
                self.close_tickets(release['spec']['desc'], release['id'])

    def close_tickets(self, desc, release_id):
        st_id = re.search(self.queue_name_regexp, desc).group(0)
        logging.info('release ticket id %s', st_id)
        release_ticket = self.st_helper.get_ticket_by_key(st_id)
        logging.info('release_ticket %s', release_ticket)
        st_tickets = set()
        comment_to_add = {
            'text': 'Release was released\nNanny release: https://nanny.yandex-team.ru/ui/#/r/{}/\nRelease ticket https://st.yandex-team.ru/{}'.format(release_id, st_id),
        }

        if '|| **Revision, Review** | **Commit author** | **Summary** | **Tickets** ||' in release_ticket.description:
            for line in release_ticket.description.split('||\n')[1:-1]:
                for ticket in re.findall(self.issue_name_regexp, line.split('|')[-1]):
                    st_tickets.add(ticket)

        for comment in release_ticket.comments:
            if '|| **Tag** | **Revisions merged** | **Branch revision** | **Revision info** | **Action** ||' in comment['text']:
                for line in comment['text'].split('||\n')[1:-1]:
                    if re.findall(self.word_regexp, line)[-1] == 'merge':
                        for ticket in re.findall(self.issue_name_regexp, line):
                            logging.info('merge %s', ticket)
                            st_tickets.add(ticket)
                    elif re.findall(self.word_regexp, line)[-1] == 'rollback':
                        for ticket in re.findall(self.issue_name_regexp, line):
                            logging.info('rollback %s', ticket)
                            if ticket in st_tickets:
                                st_tickets.remove(ticket)
        logging.info('st_tickets %s', st_tickets)

        for st_ticket in st_tickets:
            try:
                ticket = self.st_helper.get_ticket_by_key(st_ticket)
            except:
                logging.info('can not get %s', st_ticket)
                continue
            logging.info('ticket %s status is %s', st_ticket, ticket.status.key)
            if ticket.status.key == 'resolved':
                logging.info('closing ticket')
                self.st_helper.close_issue(ticket, comment_to_add)

        logging.info('release ticket status %s', release_ticket.status.key)
        if release_ticket.status.key != 'closed':
            logging.info('closing release ticket')
            self.st_helper.close_issue(release_ticket, comment_to_add)

    def is_released(self, release):
        logging.info('check if release is released')
        time_to_close = datetime.datetime.strptime(release['status']['endTime'], '%Y-%m-%dT%H:%M:%SZ') + datetime.timedelta(hours=self.Parameters.wait_time)
        logging.info('%s is time to close release', time_to_close)
        if time_to_close > datetime.datetime.now():
            logging.info('Not time to close')
            return False
        req = {
            'limit': 1000,
            'query': {
                'releaseId': release['id'],
            }
        }

        resp = self.nanny_client.filter_tickets(req)
        logging.info('Number of tickets in nanny release %s', len(resp))

        for ticket in resp:
            try:
                ticket_title = ticket['spec']['title']
                ticket_status = ticket['status']['status']
            except:
                logging.info('can not read ticket info')
                continue
            logging.info('ticket title %s', ticket_title)
            logging.info('ticket status %s', ticket_status)
            if ticket_title.startswith(('production_report', 'prestable_report')) and 'video' not in ticket_title:
                logging.info('ticket should be checked')
                if 'DEPLOY_SUCCESS' not in str(ticket_status):
                    logging.info('ticket is not DEPLOY_SUCCESS')
                    return False
        logging.info('RELEASE is released')
        return True
