import logging
import re
import base64
import tempfile
import tarfile
import glob

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

CONTAINER_RESOURCE_ID = 773239891

class UploadImageToRegistry(sdk2.Task):
    """ uploading docker image from file to cloud registry """


    class Requirements(sdk2.Task.Requirements):
        container_resource = CONTAINER_RESOURCE_ID


    class Parameters(sdk2.task.Parameters):
        image_file = sdk2.parameters.Resource(
                'YA_PACKAGE output containing image archive', required=True)
        vault_json_key_secret = sdk2.parameters.YavSecret(
                "Yav Secret with json_key for Service account authorization", required=True)
        vault_json_key_secret_key = sdk2.parameters.String(
                "Key in Yav Secret", required=True)
        destination_repository = sdk2.parameters.String(
                "Docker Registry repository to push in", required=True)
        image_tag = sdk2.parameters.String(
                "Image tag", required=True)


    def _check_return_code(self, return_code, msg):
        if return_code != 0:
            raise RuntimeError("{}: {}".format(return_code, msg))


    def _check_outs(self, outs):
        if outs[1] != '':
            logging.info(outs[1])
        logging.info(outs[0])


    def _fully_qualified_tag(self):
        fully_qualified_tag = "{}:{}".format(
                self.Parameters.destination_repository,
                self.Parameters.image_tag
                )
        return fully_qualified_tag


    def _login_into_docker_registry(self):
        logging.info("Trying to login into Docker registry")

        p = sp.Popen(["docker", "login",
            "--username", "json_key",
            "--password-stdin",
            "cr.yandex"],
            stdin=sp.PIPE,
            stdout=sp.PIPE,
            stderr=sp.PIPE
            )

        res = p.communicate(base64.b64decode(self.Parameters.vault_json_key_secret.data()[
            self.Parameters.vault_json_key_secret_key]))

        self._check_outs(res)
        self._check_return_code(p.returncode, "Failed to login")


    def _load_image_from_file(self):
        logging.info("Trying to load Docker image from file")

        destination_directory = tempfile.mkdtemp()
        package_file = str(sdk2.ResourceData(self.Parameters.image_file).path)
        tarfile.open(package_file).extractall(destination_directory)

        images = glob.glob(destination_directory + "/*.img")

        if len(images) != 1:
            raise RuntimeError("Incorrect number of images: {}".format(len(images)))

        docker_image_file = images[0]

        p = sp.Popen(["docker", "load", "--input", docker_image_file],
            stdout=sp.PIPE,
            stderr=sp.PIPE
            )

        res = p.communicate()

        self._check_outs(res)
        self._check_return_code(p.returncode, "Failed to load image from file")

        regex = re.compile("Loaded image: ([^\s]*)")
        match = regex.search(res[0])

        if match is None:
            raise RuntimeError("Image was not loaded")

        return match.group(1)


    def _tag_image(self, image_name):
        p = sp.Popen(["docker", "tag", image_name,  self._fully_qualified_tag()],
            stdout=sp.PIPE,
            stderr=sp.PIPE
            )

        res = p.communicate()

        self._check_outs(res)
        self._check_return_code(p.returncode, "Failed to tag image")


    def _push_image_to_registry(self):
        p = sp.Popen(["docker", "push", self._fully_qualified_tag()],
            stdout=sp.PIPE,
            stderr=sp.PIPE
            )

        res = p.communicate()

        self._check_outs(res)
        self._check_return_code(p.returncode, "Failed to push image into registry")


    def _upload_image_into_registry(self):
        """
        # Reproducing command sequence
        docker image load --input image.tar.gz
        docker image tag ...
        docker push ...
        """
        self._login_into_docker_registry()
        image_name = self._load_image_from_file()
        self._tag_image(image_name)
        self._push_image_to_registry()


    def on_execute(self):
        self._upload_image_into_registry()

