from sandbox import sdk2
from sandbox.projects.common.juggler import jclient
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk import errors


class AnalyticsTask(sdk2.Task):
    INHERITED_REQS = ('ram', 'disk_space')

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8072
        disk_space = 5 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        juggler_host = sdk2.parameters.String(
            'Host Name in Juggler',
            default='cloud-analytics-scheduled-jobs',
            required=False
        )
        juggler_service_name = sdk2.parameters.String(
            'Service name in Juggler',
            default='',
            required=False
        )

    @property
    def sandbox_url(self):
        return self._get_sandbox_url(self)

    @staticmethod
    def _get_sandbox_url(task):
        return 'https://sandbox.yandex-team.ru/task/{}/view'.format(task.id)

    def on_success(self, prev_status):
        super(AnalyticsTask, self).on_success(prev_status)
        self.send_event_to_juggler('OK')

    def send_event_to_juggler(self, status, host=None, service=None, description=None):
        host = host or self.Parameters.juggler_host
        service = service or self.Parameters.juggler_service_name
        if not host or not service:
            return

        if status.upper() == 'OK':
            description = description or 'Scheduled Job OK. %s' % self.sandbox_url
        else:
            description = description or 'Scheduled Job Failed. %s' % self.sandbox_url

        jclient.send_events_to_juggler(
            host,
            service,
            status,
            description
        )

    def _wait_for_subtask(self, stage, subtask_class, **kwargs):
        with self.memoize_stage[stage]:
            task_class = sdk2.Task[subtask_class]
            kwargs.update({
                'description': 'Child of task {}'.format(self.id),
                'create_sub_task': False,
            })
            sub_task = task_class(
                self,
                **kwargs
            )
            for requirement in self.INHERITED_REQS:
                setattr(sub_task.Requirements, requirement, getattr(self.Requirements, requirement))
            sub_task = sub_task.save().enqueue()
            self.Context.last_sub_task_id = sub_task.id
            raise sdk2.WaitTask(
                sub_task,
                (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)
            )

        if self.Context.last_sub_task_id is ctm.NotExists:
            raise errors.SandboxTaskFailureError("last_sub_task_id doesn't exists in task context.")

        last_sub_task_id = int(self.Context.last_sub_task_id)
        last_sub_task = list(self.find(id=last_sub_task_id))[0]
        if last_sub_task.status not in ctt.Status.Group.SUCCEED:
            raise errors.SandboxTaskFailureError('Child task is failed.')
