from .session import SolomonAPISession, SolomonDataAPISession, SolomonDataAPIFuturesSession
from .exceptions import SolomonDataAPIParseError


class SolomonAPI(object):
    ENDPOINT = 'https://solomon.yandex-team.ru/api/v2'

    def __init__(self, token, endpoint=ENDPOINT):
        self.token = token
        self.endpoint = endpoint

        self.session = SolomonAPISession()
        self.session.init_oauth(token)

    def labels(self, project, params):
        return self.session.get(self.endpoint+'/projects/{project}/sensors/labels'.format(project=project), params=params).json()

    def sensors(self, project, params):
        return self.session.get(self.endpoint+'/projects/{project}/sensors'.format(project=project), params=params).json()


class SolomonDataAPI(object):
    ENDPOINT = 'http://solomon.yandex.net/data-api'
    TIME_FORMAT = '%Y-%m-%dT%H:%M:%SZ'

    def __init__(self, who, use_futures=False, endpoint=ENDPOINT):
        self.endpoint = endpoint
        self.use_futures = use_futures
        self.who = who

        if self.use_futures:
            self.session = SolomonDataAPIFuturesSession()
        else:
            self.session = SolomonDataAPISession()

    @staticmethod
    def _parse(resp, *args, **kwargs):
        resp.data = resp.json()
        if 'sensors' not in resp.data:
            raise SolomonDataAPIParseError('Cannot find sensors key in "{}" ({})'.format(resp, resp.text))

    def get(self, project, cluster, service, extra_params):
        return self._request('get', project, cluster, service, extra_params)

    def _request(self, action, project, cluster, service, extra_params):
        params = extra_params  # type: dict
        params['project'] = project
        params['cluster'] = cluster
        params['service'] = service
        params['who'] = self.who
        if self.use_futures:
            result = self.session.get(self.endpoint + '/{action}'.format(action=action), params=params, hooks={'response': self._parse})
        else:
            result = self._parse(self.session.get(self.endpoint + '/{action}'.format(action=action), params=params))

        return result
