from startrek_client import Startrek
import logging
import requests
from ruamel import yaml
from datetime import date, datetime
import pytz
from BusinessHours import BusinessHours
from dateutil.parser import parse

requests.packages.urllib3.disable_warnings(requests.packages.urllib3.exceptions.InsecureRequestWarning)

# FIXME: update QUERY
# FIXME: ask for ROBOT token in Vault

CONFIG = "config.yaml"

with open(CONFIG, 'r') as stream:
    try:
        config = yaml.load(stream, Loader=yaml.Loader)
    except yaml.YAMLError as exc:
        logging.critical(exc)
        exit(1)


class Approval:
    def __init__(self):
        self.planning = {"state": False, "person": None}
        self.legal = {"state": False, "person": None}
        self.taxes = {"state": False, "person": None}
        self.sales = {"state": False, "person": None}
        self.manager = {"state": False, "person": None}
        self.ceo = {"state": False, "person": None}


def getST(Startrek):
    client = Startrek(
        useragent="Python script for Cloud business",
        base_url=config["STURL"],
        token=config["TOKEN"]
    )
    return client


def getIssues(st):
    # issues = st.issues.find(config["QUERY"])
    issues = st.issues.find(filter={'queue': 'CLOUDDOCS', 'assignee': 'thatguy'})
    return [issue.key for issue in issues]


def getConractType(ticket, approval):
    counter = 0
    for component in ticket.components:
        if component.name == "tender":
            counter += 1
        if component.name == "внесение_правок":
            counter += 1
        if component.name == "без_правок":
            counter += 1
            approval.planning = {"state": True, "person": config["ROBOT"]}
            approval.legal = {"state": True, "person": config["ROBOT"]}
            approval.taxes = {"state": True, "person": config["ROBOT"]}
            approval.ceo = {"state": True, "person": config["CEO"]["staff"]}
    if counter >= 2:
        logging.critical(
            "{}: Mixed components, please check ticket components and leave any one of 'tender', 'внесение_правок', 'без_правок' or none of them.".format(
                ticket.key))
        ticket.comments.create(
            text="Mixed components, please check ticket components and leave any one of 'tender', 'внесение_правок', 'без_правок' or none of them.".format(
                ticket.key), summonees=config["ROBOT"])
        exit(1)
    if counter == 0:
        approval.planning = {"state": True, "person": config["ROBOT"]}
        approval.legal = {"state": True, "person": config["ROBOT"]}
        approval.taxes = {"state": True, "person": config["ROBOT"]}
        approval.sales = {"state": True, "person": config["ROBOT"]}
        approval.manager = {"state": True, "person": config["ROBOT"]}
        approval.ceo = {"state": True, "person": config["ROBOT"]}
    return approval


def checkApproves(ticket, approval):
    if len(ticket.votedBy) > 0:
        if approval.planning["state"] is False:
            for person in ticket.votedBy:
                if (config["PLANNING"]["staff"] and person.login in config["PLANNING"]["staff"]) or (
                        config["PLANNING"]["department"] and config["PLANNING"]["department"] in checkDepartment(
                    person.login)):
                    approval.planning["state"] = True
                    approval.planning["person"] = person.login
        if approval.legal["state"] is False:
            for person in ticket.votedBy:
                if (config["LEGAL"]["staff"] and person.login in config["LEGAL"]["staff"]) or (
                        config["LEGAL"]["department"] and config["LEGAL"]["department"] in checkDepartment(
                    person.login)):
                    approval.legal["state"] = True
                    approval.legal["person"] = person.login
        if approval.taxes["state"] is False:
            for person in ticket.votedBy:
                if (config["TAXES"]["staff"] and person.login in config["TAXES"]["staff"]) or (
                        config["TAXES"]["department"] and config["TAXES"]["department"] in checkDepartment(
                    person.login)):
                    approval.taxes["state"] = True
                    approval.taxes["person"] = person.login
        if approval.sales["state"] is False:
            for person in ticket.votedBy:
                if (config["SALES"]["staff"] and person.login in config["SALES"]["staff"]) or (
                        config["SALES"]["department"] and config["SALES"]["department"] in checkDepartment(
                    person.login)):
                    approval.sales["state"] = True
                    approval.sales["person"] = person.login
        if approval.manager["state"] is False:
            for person in ticket.votedBy:
                if config["MANAGER"]["staff"] and person.login in config["MANAGER"]["staff"]:
                    approval.manager["state"] = True
                    approval.manager["person"] = person.login
        if approval.ceo["state"] is False:
            for person in ticket.votedBy:
                if config["CEO"]["staff"] and person.login in config["CEO"]["staff"]:
                    approval.ceo["state"] = True
                    approval.ceo["person"] = person.login
    return approval


def checkDepartment(person):
    header = dict()
    header["Authorization"] = "OAuth " + config["TOKEN"]
    payload = {"person.login": person, "_limit": config["STAFFLIMIT"], "_fields": "joined_at,group",
               "group.type": "department"}
    answer = requests.get(config["STAFFURL"] + config["DEPAPIURL"], params=payload, headers=header)
    if answer.status_code == 200:
        answer = answer.json()
        departments = [department["department"]["url"] for department in answer["result"][0]["group"]["ancestors"]]
        departments.append(answer["result"][0]["group"]["department"]["url"])
        return departments
    else:
        logging.critical(answer.reason)
        return None


def isApproved(approval):
    if approval.planning["state"] is True and approval.legal["state"] is True and approval.taxes["state"] is True and \
            approval.sales["state"] is True and approval.manager["state"] is True and approval.ceo["state"] is True:
        return True
    else:
        return False


def isAbsent(person):
    # https://wiki.yandex-team.ru/staff/pool/gap2/api/#poiskotsutstvijjpofiltru
    header = dict()
    header["Authorization"] = "OAuth " + config["TOKEN"]
    payload = {"login_list": person}
    status = requests.get(config["GAPURL"] + config["GAPAPIURL"], params=payload, headers=header)
    if status.status_code == 200:
        return status.json()["staff"][person]["is_available"]
    else:
        return False


def notify(ticket, stage, commentId):
    if datetime.now().hour < 9 or datetime.now().hour > 20:
        logging.critical("It's not working hours (9-20), so notifications will not be sent.")
        return
    if commentId:
        comment = ticket.comments[commentId]
        if checkUpdateDate(comment):
            if stage["staff"]:
                counter = 0
                for staff in stage["staff"]:
                    if isAbsent(staff):
                        counter += 1
                    else:
                        ticket.comments[commentId].update(text="кто:{}, пожалуйста, посмотрите предложенный договор и вынесите свое решение.".format(staff), summonees=staff)
                        break
                if counter == len(stage["staff"]):
                    ticket.comments[commentId].update(text="В данный момент никто из {} недоступен. Требуется ручное вмешательство в процесс согласования.".format(stage["name"]), summonees=config["ADMINISTRATOR"])
                    ticket.update(tags="skip")
            elif stage["email"]:
                ticket.comments[commentId].update(
                    text="{}@, пожалуйста, посмотрите предложенный договор и вынесите свое решение.".format(
                        stage["email"]), summonees=stage["email"])
            elif stage["department"]:
                ticket.comments[commentId].update(
                    text="Please ask someone from {} department to add review for this contract.".format(
                        stage["department"]))
            else:
                logging.critical(
                    "{}: No contacts are available to notify for {} stage.".format(ticket.key, stage["name"]))
    else:
        if len(stage["staff"]) == 1:
            ticket.comments.create(
                text="кто:{}, пожалуйста, посмотрите предложенный договор и вынесите свое решение.".format(
                    stage["staff"]), summonees=stage["staff"])
        elif len(stage["staff"]) > 1:
            ticket.comments.create(
                text="кто:{}, пожалуйста, посмотрите предложенный договор и вынесите свое решение.".format(
                    stage["staff"][0]),
                summonees=stage["staff"][0])
        elif stage["email"]:
            ticket.comments.create(
                text="{}@, пожалуйста, посмотрите предложенный договор и вынесите свое решение.".format(
                    stage["email"]), summonees=stage["email"])
        elif stage["department"]:
            ticket.comments.create(
                text="Please ask someone from {} department to add review for this contract.".format(
                    stage["department"]))
        else:
            logging.critical(
                "{}: No contacts are available to notify for {} stage.".format(ticket.key, stage["name"]))
    if commentId and deadlineExeeded(ticket, commentId, stage):
        ticket.comments.create(text="Срок согласования {} стадии подошел к концу, обратите внимание.".format(stage),
                               summonees=config["ADMINISTRATOR"])


def createMessage(approval):
    if approval.planning["state"] and approval.legal["state"] and approval.taxes["state"] and approval.sales[
        "state"] and approval.manager["state"] and approval.ceo["state"]:
        status = "согласован"
    else:
        status = "!!не согласован!!"
    if approval.planning["person"]:
        approval.planning["person"] = "кто:" + approval.planning["person"]
    if approval.legal["person"]:
        approval.legal["person"] = "кто:" + approval.legal["person"]
    if approval.taxes["person"]:
        approval.taxes["person"] = "кто:" + approval.taxes["person"]
    if approval.sales["person"]:
        approval.sales["person"] = "кто:" + approval.sales["person"]
    if approval.manager["person"]:
        approval.manager["person"] = "кто:" + approval.manager["person"]
    if approval.ceo["person"]:
        approval.ceo["person"] = "кто:" + approval.ceo["person"]
    file = open(config["MESSAGE"], mode="r")
    message = file.read().format(
        approval.planning["state"], approval.planning["person"],
        approval.legal["state"], approval.legal["person"],
        approval.taxes["state"], approval.taxes["person"],
        approval.sales["state"], approval.sales["person"],
        approval.manager["state"], approval.manager["person"],
        approval.ceo["state"], approval.ceo["person"],
        status
    )
    file.close()
    return message


def findCommentId(ticket, text):
    comments = ticket.comments.get_all()
    for comment in comments:
        if comment.createdBy.login == config["ROBOT"] and comment.text.startswith(text):
            commentId = comment.id
            return commentId
    return None


def updateStatus(ticket, message):
    commentId = findCommentId(ticket, "Статус")
    if commentId:
        ticket.comments[commentId].update(text=message)
    else:
        ticket.comments.create(text=message)


def checkUpdateDate(comment):
    today = date.today().isoformat()
    updateDate = comment.updatedAt
    if today > updateDate:
        return True
    else:
        return False


def deadlineExeeded(ticket, commentId, stage):
    comment = ticket.comments[commentId]
    creationDate = comment["createdAt"]
    diff = BusinessHours(parse(creationDate), datetime.utcnow().replace(tzinfo=pytz.utc)).getdays()
    if diff >= stage["sla"]:
        return True
    else:
        return False


def main():
    st = getST(Startrek)
    issues = getIssues(st)
    logging.critical("Issues that will be passed through the script: {}".format(issues))

    for issue in issues:
        ticket = st.issues[issue]
        config["MANAGER"]["staff"] = [ticket.createdBy.login]
        approval = Approval()
        approval = getConractType(ticket, approval)
        approval = checkApproves(ticket, approval)
        if not isApproved(approval):
            commentId = findCommentId(ticket, "кто:")
            if approval.planning["state"] is not True:
                notify(ticket, config["PLANNING"], commentId)
            elif approval.legal["state"] is not True:
                notify(ticket, config["LEGAL"], commentId)
            elif approval.taxes["state"] is not True:
                notify(ticket, config["TAXES"], commentId)
            elif approval.sales["state"] is not True:
                notify(ticket, config["SALES"], commentId)
            elif approval.manager["state"] is not True:
                notify(ticket, config["MANAGER"], commentId)
            elif approval.ceo["state"] is not True:
                notify(ticket, config["CEO"], commentId)
        message = createMessage(approval)
        updateStatus(ticket, message)
        logging.critical(message)


if __name__ == "__main__":
    main()

