import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import errors, environments
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.projects.cloud.analytics.common.analytics_task \
    import AnalyticsTask


class CloudFormsToCRM(AnalyticsTask):
    """Task to export forms data for CRM export"""
    SCHEMA = [
        {"name": "created_time", "type": "uint64", "sort_order": "ascending"},
        {"name": "yandex_puid", "type": "string", "sort_order": "ascending"},
        {"name": "form_id", "type": "string", "required": True},
        {"name": "yandex_login", "type": "string"},
        {"name": "ba_id", "type": "string"},
        {"name": "timezone", "type": "string"},
    ]

    class Parameters(AnalyticsTask.Parameters):
        dst_path = sdk2.parameters.String(
            'Path for CRM resulting tables',
            required=True
        )
        form_created = sdk2.parameters.String(
            'Exported form creation date in "%Y-%m-%dT%H:%M:%S"',
            required=True
        )
        form_id = sdk2.parameters.Integer(
            'Exported form identifier',
            required=True
        )
        export_fields = sdk2.parameters.List(
            'Form fields for exporting in table',
            required=True
        )
        yql_file = sdk2.parameters.String(
            'Path to YQL file in arcadia',
            default='/arc/trunk/arcadia/cloud/analytics/yql/'
                    'export_forms_to_crm.yql',
            required=True
        )
        yt_cluster = sdk2.parameters.String(
            'Cluster with source/destination tables',
            default='hahn',
            required=True
        )
        yt_token_name = sdk2.parameters.String(
            'YT Token secret name',
            default='robot-clanalytics-yt-yt-token',
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            default='YQL_TOKEN',
            required=True
        )


    class Requirements(AnalyticsTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

   
    def on_execute(self):
        import yt.wrapper as yt

        hst_table = self.Parameters.dst_path + "_history"
        query = Arcadia.cat(
            ':'.join([Arcadia.ARCADIA_SCHEME, self.Parameters.yql_file])
        )

        answers = [
            "$str(answers['{0}']) AS `{0}`".format(answer)
                for answer in self.Parameters.export_fields if answer != 'email'
        ]

        if 'email' in self.Parameters.export_fields:
            answers.append("$email_to_canonical($str(answers['email'])) AS `email`")

        yt.config['token'] = sdk2.Vault.data(
            self.owner,
            self.Parameters.yt_token_name
        )
        yt.config['proxy']['url'] = self.Parameters.yt_cluster

        if not yt.exists(hst_table):
            schema = CloudFormsToCRM.SCHEMA
            schema.extend([ {'name': f, 'type': 'string'} for f in self.Parameters.export_fields])

            yt.create(
                'table',
                hst_table,
                attributes={
                    'schema': schema
                }
            )

        self._wait_for_subtask(
            "run_yql",
            "RUN_YQL_2",
            query=query,
            trace_query=True,
            owner=self.Parameters.owner,
            publish_query=True,
            use_v1_syntax=True,
            custom_placeholders={
                '%ANSWERS%': ', '.join(answers),
                '%DST_PATH%': self.Parameters.dst_path,
                '%FORM_ID%': self.Parameters.form_id,
                '%FORM_CREATED%': self.Parameters.form_created,
                '%YT_CLUSTER%': self.Parameters.yt_cluster
            }
        )
