from datetime import datetime, timedelta
import logging
import threading
import Queue

from sandbox.sandboxsdk import environments
from sandbox import sdk2

from sandbox.projects.cloud.analytics.common.utils import ISO_FORMAT
from sandbox.projects.cloud.analytics.common.analytics_task import AnalyticsTask


class ThreadErrors(Exception):

    def __init__(self, errors):
        self.errors = errors

    def __str__(self):
        return '\n'.join([str(e) for e in self.errors])


class CloudIamToYT(AnalyticsTask):
    """ Task to import data from IAM to YT for cloud analytics"""
    CLEANUP_INTERVAL_DAYS = 3

    class Requirements(AnalyticsTask.Requirements):
        # TODO(syndicut): Use prebuilt wheels here
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('requests'),
        )

    class Parameters(AnalyticsTask.Parameters):
        dst_cluster = sdk2.parameters.String(
            'Destination YT cluster',
            default='hahn',
            required=True
        )
        cloud_owners_dst_yt_prefix = sdk2.parameters.String(
            'Destination YT prefix for CloudOwners tables',
            default='//home/cloud_analytics/import/iam/cloud_owners/1h',
            required=True
        )
        cloud_folders_dst_yt_prefix = sdk2.parameters.String(
            'Destination YT prefix for CloudFolders tables',
            default='//home/cloud_analytics/import/iam/cloud_folders/1h',
            required=True
        )
        iam_api_endpoint = sdk2.parameters.String(
            'IAM api endpoint',
            default='https://identity.private-api.cloud.yandex.net:14336/v1',
            required=True
        )
        yt_token_name = sdk2.parameters.String(
            'YT Token secret name',
            default='robot-clanalytics-yt-yt-token',
            required=True
        )
        threads_count = sdk2.parameters.Integer(
            'Threads Count',
            default=10,
            required=True
        )
        cloud_creators_requests_delay = sdk2.parameters.Integer(
            'Delay in sec. between requests to cloud:creators',
            default=0,
            required=True
        )

    def on_execute(self):
        import yt.wrapper as yt
        from sandbox.projects.cloud.analytics.common.iam.client import IamAPI
        from sandbox.projects.cloud.analytics.common.iam_to_yt import \
            cloud_folders_schema, cloud_owners_schema

        yt.config['token'] = sdk2.Vault.data(
            self.owner,
            self.Parameters.yt_token_name
        )
        yt.config['proxy']['url'] = self.Parameters.dst_cluster

        iam_api = IamAPI(endpoint=self.Parameters.iam_api_endpoint)
        now = datetime.utcnow()
        cloud_owners_table = '/'.join([self.Parameters.cloud_owners_dst_yt_prefix, now.strftime(ISO_FORMAT)])
        cloud_folders_table = '/'.join([self.Parameters.cloud_folders_dst_yt_prefix, now.strftime(ISO_FORMAT)])
        clouds = Queue.Queue()
        folders = Queue.Queue()
        thread_errors = list()

        def folders_download_thread():
            while not clouds.empty():
                cloud = clouds.get()
                try:
                    raw_folders = iam_api.cloud_folders(cloud['id'])
                    for raw_folder in raw_folders:
                        folders.put(
                            {
                                'cloud_id': raw_folder['cloudId'],
                                'folder_id': raw_folder['id'],
                                'folder_created_at': raw_folder['createdAt'],
                                'folder_name': raw_folder['name'],
                                'cloud_status': cloud['status']
                            }
                        )
                except Exception as e:
                    thread_errors.append(e)
                    return

        def folders_queue_iterator():
            while (not clouds.empty() or not folders.empty()) \
                    and not thread_errors:
                try:
                    yield folders.get(timeout=10)
                except Queue.Empty:
                    pass

        def parse_owner(cloud_creators):
            for cloud_creator in cloud_creators:
                if 'passportUserAccount' not in cloud_creator or cloud_creator['passportUserAccount'] is None:
                    continue

                clouds.put(
                    {
                        'id': cloud_creator['cloud']['id'],
                        'status': cloud_creator['cloud']['status']
                    }
                )
                if cloud_creator['userSettings'] is None:
                    cloud_creator['userSettings'] = {}

                yield {
                    'cloud_id': cloud_creator['cloud']['id'],
                    'cloud_name': cloud_creator['cloud']['name'],
                    'cloud_status': cloud_creator['cloud']['status'],
                    'cloud_created_at': cloud_creator['cloud']['createdAt'],
                    'passport_uid': cloud_creator['passportUserAccount']['passportUid'],
                    'first_name': cloud_creator['passportUserAccount']['firstName'],
                    'last_name': cloud_creator['passportUserAccount']['lastName'],
                    'id': cloud_creator['passportUserAccount']['id'],
                    'phone': cloud_creator['passportUserAccount']['phone'],
                    'login': cloud_creator['passportUserAccount']['login'],
                    'email': cloud_creator['passportUserAccount']['email'],
                    'timezone': cloud_creator['passportUserAccount']['timezone'],
                    'user_settings_email': cloud_creator['userSettings'].get(
                        'email',
                        cloud_creator['passportUserAccount']['email']
                    ),
                    'user_settings_language': cloud_creator['userSettings'].get('language', 'ru'),
                    'mail_billing': cloud_creator['userSettings'].get('mail_billing', True),
                    'mail_event': cloud_creator['userSettings'].get('mail_event', True),
                    'mail_feature': cloud_creator['userSettings'].get('mail_feature', True),
                    'mail_info': cloud_creator['userSettings'].get('mail_info', True),
                    # for marketing we only use forms_mail_marketing
                    # 'mail_marketing': cloud_creator['userSettings'].get('mail_marketing', True),
                    'mail_promo': cloud_creator['userSettings'].get('mail_promo', True),
                    'mail_support': cloud_creator['userSettings'].get('mail_support', True),
                    'mail_tech': cloud_creator['userSettings'].get('mail_tech', True),
                    'mail_technical': cloud_creator['userSettings'].get('mail_technical', True),
                    'mail_testing': cloud_creator['userSettings'].get('mail_testing', True)
                }

        yt.create_table(
            cloud_owners_table,
            attributes={"schema": cloud_owners_schema}
        )
        yt.write_table(
            cloud_owners_table,
            parse_owner(iam_api.clouds_creators(
                delay=self.Parameters.cloud_creators_requests_delay))
        )

        threads = dict()
        for i in range(self.Parameters.threads_count):
            thread_name = 'get_folders_thread_%s' % i
            threads[thread_name] = threading.Thread(
                target=folders_download_thread,
                args=(),
                name=thread_name
            )
            threads[thread_name].start()

        yt.create_table(
            cloud_folders_table,
            attributes={"schema": cloud_folders_schema}
        )
        yt.write_table(cloud_folders_table, folders_queue_iterator())

        if thread_errors:
            yt.remove(cloud_folders_table)
            raise ThreadErrors(errors=thread_errors)

        logging.info('Cleanup tables older than {} days'.format(self.CLEANUP_INTERVAL_DAYS))
        dst_prefixes_to_clean = [
            self.Parameters.cloud_owners_dst_yt_prefix,
            self.Parameters.cloud_folders_dst_yt_prefix
        ]
        for dst_prefix in dst_prefixes_to_clean:
            for table in yt.list(dst_prefix):
                table_ts = datetime.strptime(table, ISO_FORMAT)
                if now - table_ts > timedelta(days=self.CLEANUP_INTERVAL_DAYS):
                    logging.info('Remove {}'.format(table))
                    yt.remove('/'.join([dst_prefix, table]))
