import logging
import time

import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.projects.cloud.analytics.common.analytics_task import AnalyticsTask


class CloudMarketoSandboxData(AnalyticsTask):
    """Task to prepare and export data for Marketo Sandbox"""

    class Requirements(AnalyticsTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yandex-yt-transfer-manager-client')
        )

    class Parameters(AnalyticsTask.Parameters):
        src_cluster = sdk2.parameters.String(
            'Source Cluster',
            default='hahn',
            required=True
        )
        dst_cluster = sdk2.parameters.String(
            'Destination Cluster',
            default='hume',
            required=False,
        )
        src_prefix = sdk2.parameters.String(
            'Tables prefix on source cluster',
            default='//home/cloud_analytics/export/marketo',
            required=True
        )
        dst_prefix = sdk2.parameters.String(
            'Tables prefix on destination cluster',
            default='//projects/marketo-testing/export',
            required=False
        )
        yql_file = sdk2.parameters.String(
            'Path to YQL file in arcadia',
            default='/arc/trunk/arcadia/cloud/analytics/yql/testing/marketo/marketo_sandbox_data.yql',
            required=True
        )
        yt_tables = sdk2.parameters.List(
            'YT tables to copy',
            default=[
                'lead',
                'account',
                'opportunity',
                'opportunity_role',
                'ya_attend_event',
                'ya_click_email',
                'ya_consumption_daily',
                'ya_fill_out_form',
                'ya_open_email',
                'ya_send_email',
                'ya_visit_webpage'
            ],
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            default='YT_TOKEN',
            required=True
        )

    def on_execute(self):
        logging.basicConfig(level=logging.INFO)
        log = logging.getLogger('main')
        log.info('Executing YQL script')
        with self.memoize_stage.create_children:
            query = Arcadia.cat(':'.join([
                Arcadia.ARCADIA_SCHEME,
                self.Parameters.yql_file
            ]))
            sub_task = RunYQL2(
                self,
                description='Child of task {}'.format(self.id),
                create_sub_task=False,
                query=query,
                trace_query=True,
                owner=self.Parameters.owner,
                publish_query=True,
                yql_token_vault_name=self.Parameters.yql_token_name,
                use_v1_syntax=True,
                custom_placeholders={
                    '%MARKETO_EXPORT_TABLES%': self.Parameters.src_prefix,
                    '%SRC_YT_CLUSTER%': self.Parameters.src_cluster
                }
            ).enqueue()
            raise sdk2.WaitTask(
                sub_task,
                (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)
            )
        logging.info('Checking child tasks status')
        child_tasks = self.find()
        for task in child_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                raise errors.SandboxTaskFailureError('Child task failed.')

        import yt.wrapper as yt
        import yt.transfer_manager.client as tm

        yt.config['token'] = sdk2.Vault.data(
            self.owner,
            self.Parameters.yql_token_name
        )
        yt.config.set_proxy(self.Parameters.dst_cluster)

        tm_tasks = []
        for yt_table in self.Parameters.yt_tables:
            src_table_path = '%s/sandbox/%s' % (self.Parameters.src_prefix, yt_table)
            dst_table_path = '%s/%s' % (self.Parameters.dst_prefix, yt_table)
            if yt.exists(dst_table_path):
                yt.remove(dst_table_path)
            tm_tasks.append(tm.add_task(
                self.Parameters.src_cluster,
                src_table_path,
                self.Parameters.dst_cluster,
                dst_table_path
            ))
            logging.info('Added transfer_manager '
                         'task to copy table: {}'.format(yt_table))

        while len(tm_tasks) > 0:
            time.sleep(5)
            for tm_task in tm_tasks:
                tm_task_info = tm.get_task_info(tm_task)
                if tm_task_info['state'] in ('pending', 'running'):
                    continue
                else:
                    logging.info('Task execution completed with '
                                 'the following state: {}'.format(str(tm_task_info)))
                    if tm_task_info['state'] == 'completed':
                        tm_tasks.remove(tm_task)
                    else:
                        raise Exception(
                            'Transfer manager task failed with '
                            'the following state: {}'.format(tm_task_info['state'])
                        )
