import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import errors, environments
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.projects.cloud.analytics.common.analytics_task \
    import AnalyticsTask


class CloudMqlToCRM2(AnalyticsTask):
    """Task to create MQL tables for CRM export"""

    class Parameters(AnalyticsTask.Parameters):
        mkto_campaigns_to_ignore = sdk2.parameters.List(
            'The list of campaign names in Marketo which needs to be ignored',
            default=[
                'OP-Data Management-2019-Q2-Consulting '
                'Requests.Q2-2019-Consulting-Requests'
            ],
            required=True
        )
        yql_file = sdk2.parameters.String(
            'Path to YQL file in arcadia',
            default='/arc/trunk/arcadia/cloud/analytics/yql/'
                    'export_mql_to_crm.yql',
            required=True
        )
        yt_cluster = sdk2.parameters.String(
            'Cluster with source/destination tables',
            default='hahn',
            required=True
        )
        src_tables_prefix = sdk2.parameters.String(
            'MQL source tables path',
            default='//home/cloud_analytics/import/marketo',
            required=True
        )
        dst_tables_prefix = sdk2.parameters.String(
            'Tables prefix for resulting MQL tables, for CRM',
            default='//home/cloud_analytics/export/crm',
            required=False
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            default='YQL_TOKEN',
            required=True
        )
        yt_token_name = sdk2.parameters.String(
            'YT Token secret name',
            default='robot-clanalytics-yt-yt-token',
            required=True
        )
        input_row_limit = sdk2.parameters.Integer(
            'Select Input Row Limit',
            default=2000000,
            required=True
        )
        batch_limit = sdk2.parameters.Integer(
            'Select Batch Limit',
            default=100000,
            required=True
        )

    class Requirements(AnalyticsTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    def on_execute(self):
        import yt.wrapper as yt

        change_data_value = \
            self.Parameters.src_tables_prefix + '/change_data_value'
        mql_history_table = \
            self.Parameters.dst_tables_prefix + '/mql_history'
        mql_delta_table = \
            self.Parameters.dst_tables_prefix + '/mql_delta'

        yt.config['token'] = sdk2.Vault.data(
            self.owner,
            self.Parameters.yt_token_name
        )
        yt.config['proxy']['url'] = self.Parameters.yt_cluster

        last_id = 1
        # get the last marketo_id from export history
        if yt.exists(mql_history_table):
            last_record_index = int(yt.row_count(mql_history_table)) - 1
            last_record = list(
                yt.read_table(
                    yt.TablePath(
                        mql_history_table,
                        exact_index=last_record_index
                    )
                )
            )
            last_id = last_record[0]['marketo_id']

        if yt.exists(mql_delta_table):
            yt.remove(mql_delta_table)
        yt.create(
            'table',
            mql_delta_table,
            attributes={
                'schema': yt.get_attribute(change_data_value, 'schema')}
        )
        yt.write_table(
            mql_delta_table,
            yt.select_rows(
                "* FROM [%s] WHERE marketo_id > %s "
                "AND new_value='MQL' "
                "AND NOT campaign_name IN ('%s') "
                "ORDER BY marketo_id LIMIT %s"
                % (
                    change_data_value,
                    last_id,
                    "','".join(self.Parameters.mkto_campaigns_to_ignore),
                    self.Parameters.batch_limit
                ),
                input_row_limit=self.Parameters.input_row_limit,
            )
        )

        query = Arcadia.cat(
            ':'.join([Arcadia.ARCADIA_SCHEME, self.Parameters.yql_file])
        )

        with self.memoize_stage.create_children:
            sub_task = RunYQL2(
                self,
                description='Child of task {}'.format(self.id),
                create_sub_task=False,
                query=query,
                trace_query=True,
                owner=self.Parameters.owner,
                publish_query=True,
                use_v1_syntax=True,
                custom_placeholders={
                    '%MQL_SRC_TABLES_PREFIX%':
                        self.Parameters.src_tables_prefix,
                    '%MQL_DST_TABLES_PREFIX%':
                        self.Parameters.dst_tables_prefix,
                    '%YT_CLUSTER%': self.Parameters.yt_cluster
                }
            ).enqueue()
            raise sdk2.WaitTask(
                sub_task,
                (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)
            )

        child_tasks = self.find()
        for task in child_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                raise errors.SandboxTaskFailureError('Child task is failed.')
