import logging
import os

import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.projects.cloud.analytics.common.analytics_task import AnalyticsTask


class CloudSalesToCRM(AnalyticsTask):
    """Task to move data from sales table to temporary table for CRM export"""

    class Requirements(AnalyticsTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yandex-yt-transfer-manager-client')
        )

    class Parameters(AnalyticsTask.Parameters):
        yql_file = sdk2.parameters.String(
            'Path to yql file for 5min diff tables',
            default='/arc/trunk/arcadia/cloud/analytics/yql/sales_tables_for_crm.yql',
            required=True
        )
        src_cluster = sdk2.parameters.String(
            'Cluster with source table',
            default='vanga',
            required=True
        )
        src_log_table = sdk2.parameters.String(
            'Source table with sales requests',
            default='//home/cloud-www/sales-testing',
            required=True
        )
        src_prefix = sdk2.parameters.String(
            'Diff tables prefix on source cluster',
            default='//home/cloud_analytics/export/sales-testing/5min',
            required=True
        )
        cleanup_interval = sdk2.parameters.String(
            'Cleanup interval for diff tables (in days)',
            default='2',
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            default='robot-clcrm-yt-prod',
            required=True
        )
        copy_tables = sdk2.parameters.Bool(
            'Copy diff tables to another cluster'
        )
        dst_cluster = sdk2.parameters.String(
            'Destination Cluster for diff tables',
            default='hume',
            required=False,
        )
        dst_prefix = sdk2.parameters.String(
            'Diff tables prefix on destination cluster',
            default='//projects/cloudcrm-testing/export/sales/5min',
            required=False
        )

    def on_execute(self):
        logging.basicConfig(level=logging.INFO)
        log = logging.getLogger('main')
        log.info('Executing YQL script')
        with self.memoize_stage.create_children:
            query = Arcadia.cat(':'.join([
                Arcadia.ARCADIA_SCHEME,
                self.Parameters.yql_file
            ]))
            sub_task = RunYQL2(
                self,
                description='Child of task {}'.format(self.id),
                create_sub_task=False,
                use_v1_syntax=True,
                query=query,
                trace_query=True,
                owner=self.Parameters.owner,
                publish_query=True,
                yql_token_vault_name=self.Parameters.yql_token_name,
                custom_placeholders={
                    '%SALES_LOG_TABLE%': self.Parameters.src_log_table,
                    '%SALES_EXPORT_PATH%': self.Parameters.src_prefix,
                    '%CLEANUP_INTERVAL%': self.Parameters.cleanup_interval
                }
            ).enqueue()
            raise sdk2.WaitTask(
                sub_task,
                (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)
            )
        logging.info('Checking child tasks status')
        child_tasks = self.find()
        for task in child_tasks:
            if task.status not in ctt.Status.Group.SUCCEED:
                raise errors.SandboxTaskFailureError('Child task failed.')

        import yt.wrapper as yt
        import yt.transfer_manager.client as tm

        yt.config['token'] = sdk2.Vault.data(
            self.owner,
            self.Parameters.yql_token_name
        )
        src_yt_client = yt.YtClient(
            proxy=self.Parameters.src_cluster,
            config=yt.config.config
        )
        dst_yt_client = yt.YtClient(
            proxy=self.Parameters.dst_cluster,
            config=yt.config.config
        )

        if self.Parameters.copy_tables and \
                self.Parameters.dst_cluster and self.Parameters.dst_prefix:
            log.info('Starting sync between yt clusters through transfer manager')
            src_tables = set(src_yt_client.list(self.Parameters.src_prefix))
            dst_tables = set(dst_yt_client.list(self.Parameters.dst_prefix))
            tables_to_copy = src_tables - dst_tables
            tables_to_remove = dst_tables - src_tables
            log.info('Tables to copy: {}'.format(tables_to_copy))
            for table in tables_to_copy:
                task = tm.add_task(
                    self.Parameters.src_cluster,
                    os.path.join(self.Parameters.src_prefix, table),
                    self.Parameters.dst_cluster,
                    os.path.join(self.Parameters.dst_prefix, table)
                )
                log.info('Added transfer_manager task: {}'.format(
                    str(tm.get_task_info(task))))
            log.info('Tables to remove from destination: {}'.format(
                tables_to_remove))
            for table in tables_to_remove:
                log.info('Remove {}'.format(table))
                dst_yt_client.remove(os.path.join(
                    self.Parameters.dst_prefix, table))
