import logging
import time
from datetime import datetime
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk import environments

from sandbox import sdk2

TABLE = "home/metering/abc_tree/last"


class AutoBinaryTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)


class ActualizeInternalBindings(AutoBinaryTask):
    """ Task to actualize abc->BillingAccount bindings"""

    class Parameters(AutoBinaryTask.Parameters):
        yt_cluster = sdk2.parameters.String(
            'YT cluster',
            default='hahn',
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            required=True
        )
        env = sdk2.parameters.StrictString(
            'Environment (prestable/prod)',
            regexp="^prestable|prod$",
            required=True
        )
        jwt_private_key_name = sdk2.parameters.String(
            "JWT private key secret name",
            required=True
        )

    __token_service_endpoint_config_dict = {
        'url': 'ts.prestable.cloud-internal.yandex.net:443',
        'tls': {'enabled': True, 'root_certs_file': '/etc/ssl/certs/ca-certificates.crt'}
    }

    class Requirements(AutoBinaryTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yql'),
        )

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        import yt.wrapper as ytw
        from cloud.billing.utils.scripts.get_actual_abc import get_actual_abc
        logging.basicConfig(level=logging.INFO)
        self._log = logging.getLogger('main')
        yql_token = sdk2.Vault.data(self.owner, self.Parameters.yql_token_name)
        yt_proxy = self.Parameters.yt_cluster
        yql_client = YqlClient(db=yt_proxy, token=yql_token)
        ytw.config['token'] = yql_token
        ytw.config['proxy']['url'] = yt_proxy

        result, _ = get_actual_abc(yql_client=yql_client,
                                   path=TABLE,
                                   )
        self.__iam_token = self.__auth()

        for service in result:
            if service["state"] == "deleted":
                self._bind_to_wastebucket(service["id"])
                self._log.info("=== CALLED WASTE")
        self._log.info("===Done!")

    def _bind_to_wastebucket(self, abc_id):
        self._log.info("Binding to wastebucket %s", abc_id)
        from cloud.billing.utils.clients.billing.billing_client import BillingClient

        client = BillingClient(
            "https://billing-internal-prestable.private-api.cloud.yandex.net:16465/billing/v1/private",
            self.__iam_token)
        cloud_id = client.resolve_abc_to_cloud(abc_id=abc_id)
        if cloud_id:
            operation = client.bind_to_billing_account(cloud_id=cloud_id,
                                                       billing_account_id="yc.billing.wastebasket")
            error = operation.get("error")
            if error:
                raise RuntimeError(error)

    def __prepare_jwt(self):
        import jwt
        now = int(time.time())
        self._log.info("NOW: %s", now)

        payload = {
            # todo replace with params
            "iss": "sandbox-service-account",
            "aud": "https://iam.prestable.cloud-internal.yandex.net/iam/v1/tokens",
            "iat": now,
            "exp": now + 3600,
        }

        headers = {
            "typ": "JWT",
            "alg": "PS256",
            "kid": "sandbox-service-account-key",
        }

        return jwt.encode(payload, sdk2.Vault.data(self.owner, self.Parameters.jwt_private_key_name), algorithm="PS256",
                          headers=headers)

    def __auth(self):

        from cloud.billing.utils.clients.grpc.base import GrpcEndpointConfig
        from cloud.billing.utils.clients.iam.yc_token_client import get_token_client
        endpoint_config = GrpcEndpointConfig(self.__token_service_endpoint_config_dict)
        client = get_token_client(endpoint_config)
        return client.create_token(jwt=self.__prepare_jwt())
